import { Head, Link, router, usePage } from '@inertiajs/react';
import AppLayout from '@/layouts/app-layout';
import { BreadcrumbItem } from '@/types';
import { Button } from '@/components/ui/button';
import { Eye, Pencil, Trash } from 'lucide-react';
import { useState } from 'react';
import { Card, CardHeader, CardTitle, CardContent } from "@/components/ui/card";
import ReactPaginate from 'react-paginate';
import Swal from 'sweetalert2';
import moment from 'moment';
import { PagePropsWithMedicationRequestsAll } from '@/types';
import { Modal, Form, Input, message, Select, Row, Col } from 'antd';
import { useDarkMode } from '@/hooks/useDarkMode';
import { Label } from "@/components/ui/label";
import axios from 'axios';
import { Badge } from "@/components/ui/badge"
import { API } from '@/config';

const breadcrumbs: BreadcrumbItem[] = [{ title: 'Medication Requests', href: '/medication-requests' }];

export default function MedicationRequestsAll() {
    const { requests, clinics, filters, batches } = usePage<PagePropsWithMedicationRequestsAll>().props;
    const [searchQuery, setSearchQuery] = useState(filters.search || '');
    const [isModalOpen, setIsModalOpen] = useState(false);
    const [editRequestId, setEditRequestId] = useState<number | null>(null);
    const [isTransferModalOpen, setIsTransferModalOpen] = useState(false);
    const [selectedRequest, setSelectedRequest] = useState<any>(null);
    const [filteredBatches, setFilteredBatches] = useState<any[]>([]);
    const [selectedBatch, setSelectedBatch] = useState<any>(null);
    const [form] = Form.useForm();

    const handleSearch = () => {
        router.get(route('medication-requests.indexAll'), { search: searchQuery }, { preserveState: true });
    };

    const handlePageChange = (selected: { selected: number }) => {
        router.get(route('medication-requests.indexAll'), { page: selected.selected + 1, search: searchQuery }, { preserveState: true });
    };

    const handleDelete = (id: number) => {
        Swal.fire({
            title: 'Are you sure?',
            text: 'This medication request will be permanently deleted.',
            icon: 'warning',
            showCancelButton: true,
            confirmButtonColor: '#e3342f',
            cancelButtonColor: '#6c757d',
            confirmButtonText: 'Yes, delete it!',
            width: '350px',
        }).then((result) => {
            if (result.isConfirmed) {
                router.delete(route('medication-requests.destroy', id), {
                    onSuccess: () => {
                        Swal.fire({
                            title: 'Deleted!',
                            text: 'The medication request has been deleted.',
                            icon: 'success',
                            timer: 1500,
                            showConfirmButton: false,
                        });
                    },
                    onError: () => {
                        Swal.fire({
                            title: 'Error!',
                            text: 'Something went wrong.',
                            icon: 'error',
                        });
                    },
                });
            }
        });
    };

    const handleEdit = (id: number) => {
        setEditRequestId(id);
        setIsModalOpen(true);
        const request = requests.data.find((req) => req.id === id);
        if (request) {
            form.setFieldsValue({
                quantity: request.quantity,
                status: request.status,
            });
        }
    };

    const handleModalCancel = () => {
        setIsModalOpen(false);
        form.resetFields();
        setEditRequestId(null);
    };

    const handleModalOk = () => {
        form.validateFields().then((values) => {
            if (editRequestId) {
                const loading = Swal.fire({
                    title: 'Updating...',
                    allowOutsideClick: false,
                    allowEscapeKey: false,
                    didOpen: () => {
                        Swal.showLoading();
                    },
                });

                router.put(route('medication-requests.update', editRequestId), values, {
                    onSuccess: () => {
                        Swal.close();
                        message.success('Medication request updated successfully.');
                        setIsModalOpen(false);
                        form.resetFields();
                        setEditRequestId(null);
                    },
                    onError: (errors) => {
                        Swal.close();
                        message.error('Something went wrong.');
                        console.error(errors);
                    },
                });
            }
        });
    };

    const handleTransfer = (request: any) => {
        setSelectedRequest(request);
        setIsTransferModalOpen(true);
        const filteredBatches = batches.filter((batch: any) => batch.medication_id === request.medication.id);
        setFilteredBatches(filteredBatches);
        setSelectedBatch(null);
        form.resetFields();

        // Set default clinic and transfer_date
        if (request.clinic?.id) {
            form.setFieldsValue({
                clinic_id: request.clinic.id,
                transfer_date: new Date().toISOString().split("T")[0],
            });
        } else {
            form.setFieldsValue({
                transfer_date: new Date().toISOString().split("T")[0],
            });
        }
    };

    const handleTransferModalCancel = () => {
        setIsTransferModalOpen(false);
        setSelectedRequest(null);
        setSelectedBatch(null);
        setFilteredBatches([]);
        form.resetFields();
    };

    const handleTransferModalOk = () => {
        form.validateFields()
            .then((values) => {
                if (!selectedBatch) {
                    message.error('Please select a batch before transferring.');
                    return;
                }

                if (!selectedRequest || !selectedRequest.id) {
                    message.error('No medication request selected.');
                    return;
                }

                // Show loading Swal
                Swal.fire({
                    title: 'Transferring Stock...',
                    allowOutsideClick: false,
                    allowEscapeKey: false,
                    didOpen: () => {
                        Swal.showLoading();
                    },
                });

                const { clinic_id, quantity, transfer_date } = values;

                const data = {
                    clinic_id,
                    quantity,
                    transfer_date,
                    request_id: selectedRequest.id,
                };

                axios.post(
                    `${API}/central-stores/1/batches/${selectedBatch.id}/transfer`,
                    data
                )
                    .then((response) => {
                        Swal.close();

                        if (response.status >= 200 && response.status < 300) {
                            Swal.fire({
                                icon: 'success',
                                title: 'Stock Transferred!',
                                text: response.data?.message || 'Stock transferred successfully.',
                                confirmButtonText: 'OK'
                            }).then(() => {
                                setIsTransferModalOpen(false);
                                setSelectedRequest(null);
                                setSelectedBatch(null);
                                setFilteredBatches([]);
                                form.resetFields();

                                window.location.reload();
                                router.get(route('medication-requests.indexAll'));
                            });
                        }
                    })
                    .catch((error) => {
                        Swal.close();

                        if (error.response?.status === 422) {
                            const errors = error.response.data?.errors || {};
                            Object.values(errors).forEach((err) => {
                                message.error(String(err));
                            });
                        } else {
                            Swal.fire({
                                icon: 'error',
                                title: 'Transfer Failed',
                                text: error.response?.data?.message || 'Failed to transfer stock. Please try again.',
                                confirmButtonText: 'OK'
                            });
                        }

                        console.error('Transfer Error:', error);
                    });
            })
            .catch(() => {
                message.error('Please correct the errors in the form before submitting.');
            });
    };



    const handleBatchChange = (value: any) => {
        const selected = filteredBatches.find((batch: any) => batch.id === value);
        setSelectedBatch(selected);
        form.setFieldsValue({
            batch_number: selected?.batch_number,
            current_stock: selected?.quantity,
            expiry_date: selected?.expiry_date,
        });
    };

    const { isDarkMode } = useDarkMode();

    return (
        <AppLayout breadcrumbs={breadcrumbs}>
            <Head title="Medication Requests" />
            <div className="bg-background text-foreground mx-8 my-6 flex flex-1 flex-col gap-4 rounded-xl p-6 shadow-sm">
                <div className="mb-4 flex flex-wrap items-center justify-between gap-4">
                    <h1 className="text-2xl font-bold">Medication Requests for All Clinics</h1>
                </div>

                <div className="mb-4 flex gap-2">
                    <div className="relative w-72">
                        <input
                            type="text"
                            value={searchQuery}
                            onChange={(e) => setSearchQuery(e.target.value)}
                            onKeyDown={(e) => e.key === 'Enter' && handleSearch()}
                            placeholder="Search medication requests..."
                            className="border-border bg-background placeholder-muted-foreground w-full rounded-lg border py-2 pr-3 pl-10 text-sm focus:ring-2 focus:ring-blue-600 focus:outline-none"
                        />
                    </div>
                    <button onClick={handleSearch} className="rounded-lg bg-blue-600 px-4 py-2 text-sm font-medium text-white hover:bg-blue-700">
                        Search
                    </button>
                </div>

                <div className="overflow-x-auto rounded-xl border">
                    <table className="min-w-full divide-y overflow-hidden rounded-xl bg-white text-sm dark:bg-neutral-900">
                        <thead className="bg-muted">
                            <tr>
                                {['Clinic', 'Medication', 'Quantity', 'Transferred Quantity', 'Request Date', 'Status', 'Actions'].map((header) => (
                                    <th
                                        key={header}
                                        className="text-muted-foreground px-4 py-3 text-left text-xs font-semibold tracking-wider uppercase"
                                    >
                                        {header}
                                    </th>
                                ))}
                            </tr>
                        </thead>
                        <tbody className="divide-y">
                            {requests.data.map((request) => (
                                <tr key={request.id} className="hover:bg-muted/50 transition">
                                    <td className="px-4 py-3">{request?.clinic?.name}</td>
                                    <td className="px-4 py-3">{request.medication.name}</td>
                                    <td className="px-4 py-3">{request.quantity}</td>
                                    <td className="px-4 py-3">
                                        {request.status === 'approved' && (
                                            <Badge className="bg-green-500 text-white hover:bg-green-600">
                                                {request.transferred_quantity}
                                            </Badge>
                                        )}
                                    </td>
                                    <td className="px-4 py-3">
                                        {moment(request.request_date).format('MMM D, YYYY')}
                                    </td>
                                    <td className="px-4 py-3">
                                        <span
                                            className={`inline-block rounded-full px-3 py-1 text-sm font-semibold ${request.status === 'approved'
                                                ? 'bg-green-100 text-green-800'
                                                : request.status === 'pending'
                                                    ? 'bg-yellow-100 text-yellow-800'
                                                    : request.status === 'rejected'
                                                        ? 'bg-red-100 text-red-800'
                                                        : 'bg-gray-100 text-gray-800'
                                                }`}
                                            style={{ cursor: 'pointer', textTransform: "capitalize" }}
                                        >
                                            {request.status}
                                        </span>
                                    </td>
                                    <td className="flex items-center gap-2 px-4 py-3">
                                        <button
                                            onClick={() => handleEdit(request.id)}
                                            className="inline-flex items-center justify-center rounded-full bg-yellow-500 p-2 text-white hover:bg-yellow-600"
                                            title="Edit"
                                            style={{ cursor: 'pointer' }}
                                        >
                                            <Pencil className="h-4 w-4" />
                                        </button>
                                        <button
                                            onClick={() => handleDelete(request.id)}
                                            className="inline-flex items-center justify-center rounded-full bg-red-600 p-2 text-white hover:bg-red-700"
                                            title="Delete"
                                            style={{ cursor: 'pointer' }}
                                        >
                                            <Trash className="h-4 w-4" />
                                        </button>
                                        <button
                                            onClick={() => handleTransfer(request)}
                                            disabled={request.status === 'approved'}
                                            className={`inline-flex items-center justify-center border border-gray-300 bg-transparent p-2 text-gray-700 hover:bg-blue-500 ${request.status === 'approved' ? 'opacity-50 cursor-not-allowed' : ''
                                                }`}
                                            title="Transfer Medication"
                                            style={{
                                                cursor: request.status === 'approved' ? 'not-allowed' : 'pointer',
                                                borderRadius: '10px',
                                                color: '#fff',
                                            }}
                                        >
                                            <span className="text-sm font-medium">Transfer</span>
                                        </button>
                                    </td>
                                </tr>
                            ))}
                        </tbody>
                    </table>
                </div>

                <div className="mt-6 flex justify-center">
                    <ReactPaginate
                        pageCount={requests.last_page}
                        forcePage={requests.current_page - 1}
                        onPageChange={handlePageChange}
                        marginPagesDisplayed={1}
                        pageRangeDisplayed={3}
                        previousLabel="← Prev"
                        nextLabel="Next →"
                        breakLabel="..."
                        containerClassName="flex items-center gap-2 text-sm"
                        pageClassName="px-3 py-1 border border-border rounded hover:bg-muted"
                        activeClassName="bg-blue-600 text-white"
                        previousClassName="px-3 py-1 border border-border rounded hover:bg-muted"
                        nextClassName="px-3 py-1 border border-border rounded hover:bg-muted"
                        breakClassName="px-2"
                    />
                </div>
            </div>

            <Modal
                title="Edit Medication Request"
                open={isModalOpen}
                onCancel={handleModalCancel}
                footer={[
                    <Button key="back" onClick={handleModalCancel}>
                        Cancel
                    </Button>,
                    <Button key="submit" variant={"secondary"} onClick={handleModalOk}>
                        Update
                    </Button>,
                ]}
                className={isDarkMode ? 'dark-modal' : ''}
            >
                <Form form={form} layout="vertical">
                    <Form.Item label="Quantity" name="quantity" rules={[{ required: true, message: 'Please enter the quantity!' }]}>
                        <Input type="number" min={1} />
                    </Form.Item>

                    <Form.Item label="Status" name="status" rules={[{ required: true, message: 'Please enter the status!' }]}>
                        <Input />
                    </Form.Item>
                </Form>
            </Modal>

            <Modal
                title="Transfer Medication Stock"
                open={isTransferModalOpen}
                onCancel={handleTransferModalCancel}
                width={800}
                footer={[
                    <Button key="back" onClick={handleTransferModalCancel} style={{ cursor: 'pointer' }}>
                        Cancel
                    </Button>,
                    <Button key="submit" variant={"secondary"} onClick={handleTransferModalOk} style={{ cursor: 'pointer' }}>
                        Transfer
                    </Button>,
                ]}
                className={isDarkMode ? 'dark-modal' : ''}
            >
                <Form form={form} layout="vertical">
                    <Row gutter={16}>
                        <Col span={12}>
                            <Form.Item
                                label="Medication Batch"
                                name="medication_batch_id"
                                rules={[{ required: true, message: "Select Medication Batch" }]}
                            >
                                <Select
                                    options={filteredBatches.map((batch: any) => ({
                                        value: batch.id,
                                        label: batch.batch_number,
                                    }))}
                                    placeholder="Select a Batch..."
                                    onChange={handleBatchChange}
                                />
                            </Form.Item>
                        </Col>

                        {selectedBatch && (
                            <Col span={24}>
                                <Card className="bg-muted border shadow-sm mt-2">
                                    <CardHeader>
                                        <CardTitle className="text-base">Batch Information</CardTitle>
                                    </CardHeader>
                                    <CardContent className="grid grid-cols-2 gap-4">
                                        <div>
                                            <Label>Batch Number</Label>
                                            <p className="text-sm text-muted-foreground">
                                                {selectedBatch.batch_number}
                                            </p>
                                        </div>
                                        <div>
                                            <Label>Current Stock</Label>
                                            <p className="text-sm text-muted-foreground">
                                                {selectedBatch.quantity}
                                            </p>
                                        </div>
                                        <div>
                                            <Label>Expiry Date</Label>
                                            <p className="text-sm text-muted-foreground">
                                                {selectedBatch.expiry_date}
                                            </p>
                                        </div>
                                    </CardContent>
                                </Card>
                            </Col>
                        )}

                        <Col span={12} style={{ marginTop: "10px" }}>
                            <Form.Item
                                label="Clinic"
                                name="clinic_id"
                                rules={[{ required: true, message: "Please select a clinic!" }]}
                            >
                                <Select
                                    options={Array.isArray(clinics)
                                        ? clinics.map((clinic: any) => ({
                                            value: clinic.id,
                                            label: clinic.name,
                                        }))
                                        : []}
                                    placeholder="Select a clinic..."
                                />
                            </Form.Item>
                        </Col>

                        <Col span={12} style={{ marginTop: "10px" }}>
                            <Form.Item
                                label="Quantity to Transfer"
                                name="quantity"
                                rules={[
                                    { required: true, message: "Please enter the quantity!" },
                                    {
                                        validator: (_, value) => {
                                            if (!value) {
                                                return Promise.resolve(); // handled by required rule
                                            }
                                            if (selectedBatch && value > selectedBatch.quantity) {
                                                return Promise.reject(new Error(`Quantity cannot exceed available stock (${selectedBatch.quantity})`));
                                            }
                                            if (value < 1) {
                                                return Promise.reject(new Error('Quantity must be at least 1'));
                                            }
                                            return Promise.resolve();
                                        }
                                    }
                                ]}
                            >
                                <Input
                                    type="number"
                                    min={1}
                                    max={selectedBatch?.quantity}
                                    required
                                />
                            </Form.Item>
                        </Col>

                        <Col span={12}>
                            <Form.Item
                                label="Transfer Date"
                                name="transfer_date"
                                rules={[{ required: true, message: "Please enter the transfer date!" }]}
                                initialValue={new Date().toISOString().split("T")[0]}
                            >
                                <Input
                                    type="date"
                                    defaultValue={new Date().toISOString().split("T")[0]}
                                    required
                                />
                            </Form.Item>
                        </Col>
                    </Row>
                </Form>
            </Modal>
        </AppLayout>
    );
}
