import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import AppLayout from '@/layouts/app-layout';
import { BreadcrumbItem } from '@/types';
import { Head, useForm, usePage } from '@inertiajs/react';
import { useEffect, useState } from 'react';
import Select from 'react-select';
import moment from 'moment';
import Swal from 'sweetalert2';

type OptionType = {
    value: number;
    label: string;
};

export default function RequestMedication() {
    const { clinic, medications, requests } = usePage().props as any;

    const { data, setData, post, processing, errors, reset } = useForm({
        medication_id: '',
        quantity: '',
    });

    const [isDarkMode, setIsDarkMode] = useState(false);

    useEffect(() => {
        const checkDarkMode = () => {
            setIsDarkMode(document.documentElement.classList.contains('dark'));
        };
        checkDarkMode();
        const observer = new MutationObserver(checkDarkMode);
        observer.observe(document.documentElement, {
            attributes: true,
            attributeFilter: ['class'],
        });
        return () => observer.disconnect();
    }, []);

    const medicationOptions: OptionType[] = medications.map((m: any) => ({
        value: m.id,
        label: `${m.name} (${m.dosage || ''} ${m.unit || ''}) ${m.form}`,

    }));

    const handleSubmit = (e: React.FormEvent) => {
        e.preventDefault();

        post(route('clinics.medication-requests.store', clinic.id), {
            onSuccess: () => {
                Swal.fire({
                    icon: 'success',
                    title: 'Success!',
                    text: 'Medication request submitted successfully.',
                    confirmButtonColor: '#3085d6',
                });

                reset(); // ✅ Clear the form
            },
            onError: () => {
                Swal.fire({
                    icon: 'error',
                    title: 'Oops!',
                    text: 'There was an error submitting the request.',
                });
            }
        });
    };

    const selectStyles = {
        control: (base: any) => ({
            ...base,
            backgroundColor: isDarkMode ? '#000' : '#fff',
            borderColor: isDarkMode ? '#374151' : '#d1d5db',
            color: isDarkMode ? '#fff' : '#000',
        }),
        menu: (base: any) => ({ ...base, backgroundColor: isDarkMode ? '#000' : '#fff' }),
        option: (base: any, { isFocused }: any) => ({
            ...base,
            backgroundColor: isFocused ? (isDarkMode ? '#1e40af' : '#3b82f6') : 'transparent',
            color: isFocused ? '#fff' : isDarkMode ? '#fff' : '#000',
        }),
        singleValue: (base: any) => ({ ...base, color: isDarkMode ? '#fff' : '#000' }),
    };

    const breadcrumbs: BreadcrumbItem[] = [
        { title: 'Clinics', href: route('clinics.index') },
        { title: 'Request Medication', href: '#' },
    ];

    return (
        <AppLayout breadcrumbs={breadcrumbs}>
            <Head title="Request Medication" />
            <div className="grid grid-cols-12 gap-8 px-4 py-6 sm:px-6">
                {/* LEFT SIDE */}
                <div className="col-span-12 lg:col-span-8 space-y-8">
                    {/* FORM */}
                    <form
                        onSubmit={handleSubmit}
                        className="bg-card rounded-lg border p-6 shadow-sm dark:bg-neutral-900"
                    >
                        <h2 className="mb-4 text-xl font-semibold">Request Medication</h2>
                        <div className="grid grid-cols-1 gap-4 sm:grid-cols-2">
                            <div className="space-y-2">
                                <Label htmlFor="medication_id">Medication</Label>
                                <Select
                                    id="medication_id"
                                    options={medicationOptions}
                                    value={medicationOptions.find(
                                        (opt) => opt.value === Number(data.medication_id)
                                    )}
                                    onChange={(option) =>
                                        setData('medication_id', option?.value.toString() ?? '')
                                    }
                                    placeholder="Select medication..."
                                    styles={selectStyles}
                                />
                                {errors.medication_id && (
                                    <p className="text-sm text-red-500">{errors.medication_id}</p>
                                )}
                            </div>

                            <div className="space-y-2">
                                <Label htmlFor="quantity">Quantity</Label>
                                <Input
                                    id="quantity"
                                    type="number"
                                    min="1"
                                    value={data.quantity}
                                    onChange={(e) => setData('quantity', e.target.value)}
                                />
                                {errors.quantity && (
                                    <p className="text-sm text-red-500">{errors.quantity}</p>
                                )}
                            </div>
                        </div>

                        <div className="mt-6 flex justify-end gap-4">
                            <Button type="button" variant="outline" onClick={() => window.history.back()}>
                                Cancel
                            </Button>
                            <Button type="submit" disabled={processing}>
                                {processing ? 'Submitting...' : 'Request Medication'}
                            </Button>
                        </div>
                    </form>

                    {/* PREVIOUS REQUESTS TABLE */}
                    <div className="bg-card rounded-lg border shadow-sm dark:bg-neutral-900">
                        <h2 className="px-6 py-4 text-lg font-semibold border-b dark:border-neutral-700">
                            Previous Requests
                        </h2>
                        {requests.length === 0 ? (
                            <p className="px-6 py-4 text-sm text-muted-foreground">No requests yet.</p>
                        ) : (
                            <div className="overflow-x-auto">
                                <table className="min-w-full text-sm">
                                    <thead className="border-b dark:border-neutral-700">
                                        <tr>
                                            <th className="px-6 py-3 text-left font-medium text-muted-foreground">
                                                Medication
                                            </th>
                                            <th className="px-6 py-3 text-left font-medium text-muted-foreground">
                                                Quantity
                                            </th>
                                            <th className="px-6 py-3 text-left font-medium text-muted-foreground">
                                                Request Date
                                            </th>
                                            <th className="px-6 py-3 text-left font-medium text-muted-foreground">
                                                Status
                                            </th>
                                        </tr>
                                    </thead>
                                    <tbody className="divide-y dark:divide-neutral-700">
                                        {requests.map((req: any) => (
                                            <tr key={req.id}>
                                                <td className="px-6 py-3">{req.medication.name}</td>
                                                <td className="px-6 py-3">{req.quantity}</td>
                                                <td className="px-6 py-3">
                                                    {moment(req.request_date).format('dddd D MMMM YYYY')}
                                                </td>

                                                <td className="px-6 py-3">
                                                    <span
                                                        className={`inline-flex rounded-full px-2 py-0.5 text-xs font-medium
                                                    ${req.status === 'pending'
                                                                ? 'bg-yellow-100 text-yellow-800'
                                                                : ''
                                                            }
                                                    ${req.status === 'approved'
                                                                ? 'bg-green-100 text-green-800'
                                                                : ''
                                                            }
                                                    ${req.status === 'rejected'
                                                                ? 'bg-red-100 text-red-800'
                                                                : ''
                                                            }
                                                    `}
                                                    >
                                                        {req.status}
                                                    </span>
                                                </td>
                                            </tr>
                                        ))}
                                    </tbody>
                                </table>
                            </div>
                        )}
                    </div>
                </div>

                {/* RIGHT SIDE — Clinic Info */}
                <div className="col-span-12 lg:col-span-4 space-y-4">
                    <div className="bg-card rounded-lg border p-6 shadow-sm dark:bg-neutral-900">
                        <h2 className="mb-2 text-lg font-semibold">Clinic Info</h2>
                        <p className="text-muted-foreground text-sm">Name: {clinic.name}</p>
                        <p className="text-muted-foreground text-sm">Location: {clinic?.address},{" "}{clinic?.city},{" "}{clinic.country}</p>
                    </div>
                </div>
            </div>
        </AppLayout>
    );
}
