import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { API } from '@/config';
import AppLayout from '@/layouts/app-layout';
import { Clinic, ClinicMedicationStock } from '@/types/medication';
import { PageProps } from '@inertiajs/core';
import { Head, router, usePage } from '@inertiajs/react';
import { useState } from 'react';

interface Props extends PageProps {
    clinic: Clinic;
    stock: ClinicMedicationStock;
}

export default function MedicationStockEdit() {
    const { clinic, stock } = usePage<Props>().props;
    const [formData, setFormData] = useState({
        quantity: stock.quantity,
        expiry_date: stock.expiry_date || '',
        batch_number: stock.batch_number || '',
    });

    const handleSubmit = (e: React.FormEvent) => {
        e.preventDefault();
        router.put(`${API}/clinics/${clinic.id}/medication-stocks/${stock.id}`, formData);
    };

    const breadcrumbs = [
        { title: 'Clinics', href: `${API}/clinics` },
        { title: clinic.name, href: `${API}/clinics/${clinic.id}` },
        { title: 'Medication Stocks', href: `${API}/clinics/${clinic.id}/medication-stocks` },
        { title: 'Edit Stock', href: `${API}/clinics/${clinic.id}/medication-stocks/${stock.id}/edit` },
    ];

    return (
        <AppLayout breadcrumbs={breadcrumbs}>
            <Head title={`Edit ${stock.medication.name} Stock`} />
            <div className="w-full px-4 py-6 sm:px-6">
                <div className="mb-6 flex items-center justify-between">
                    <h1 className="text-2xl font-bold">Edit {stock.medication.name} Stock</h1>
                </div>

                {/* Rest of your edit form remains the same */}
                {/* ... */}
                <form onSubmit={handleSubmit} className="space-y-8">
                    <div className="bg-card rounded-lg border p-6 shadow-sm">
                        <h2 className="mb-4 text-xl font-semibold">Stock Information</h2>
                        <div className="grid grid-cols-1 gap-6 md:grid-cols-3">
                            <div className="space-y-2">
                                <Label htmlFor="medication">Medication</Label>
                                <Input id="medication" value={`${stock.medication.name} (${stock.medication.dosage || 'No dosage'})`} disabled />
                            </div>

                            <div className="space-y-2">
                                <Label htmlFor="quantity">Quantity*</Label>
                                <Input
                                    type="number"
                                    id="quantity"
                                    min="0"
                                    value={formData.quantity}
                                    onChange={(e) => setFormData({ ...formData, quantity: parseInt(e.target.value) })}
                                    required
                                />
                            </div>

                            <div className="space-y-2">
                                <Label htmlFor="expiry_date">Expiry Date</Label>
                                <Input
                                    type="date"
                                    id="expiry_date"
                                    value={formData.expiry_date}
                                    onChange={(e) => setFormData({ ...formData, expiry_date: e.target.value })}
                                />
                            </div>

                            <div className="space-y-2">
                                <Label htmlFor="batch_number">Batch Number</Label>
                                <Input
                                    type="text"
                                    id="batch_number"
                                    value={formData.batch_number}
                                    onChange={(e) => setFormData({ ...formData, batch_number: e.target.value })}
                                />
                            </div>
                        </div>
                    </div>

                    <div className="flex justify-end gap-4">
                        <Button type="button" variant="outline" onClick={() => window.history.back()}>
                            Cancel
                        </Button>
                        <Button type="submit">Update Stock</Button>
                    </div>
                </form>
            </div>
        </AppLayout>
    );
}
