import { API } from '@/config';
import AppLayout from '@/layouts/app-layout';
import { Clinic } from '@/types/medication';
import { type SharedData } from '@/types';
import { PageProps } from '@inertiajs/core';
import { Head, router, usePage } from '@inertiajs/react';
import axios from 'axios';
import { Pencil, Search as SearchIcon, Trash, Download } from 'lucide-react';
import { useEffect, useMemo, useState } from 'react';
import ReactPaginate from 'react-paginate';
import Swal from 'sweetalert2';
import * as XLSX from 'xlsx';
import moment from 'moment';
import { useDebouncedCallback } from 'use-debounce';

// shadcn/ui
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogFooter,
} from '@/components/ui/dialog';
import { Checkbox } from '@/components/ui/checkbox';
import { Textarea } from '@/components/ui/textarea';
import { Badge } from '@/components/ui/badge';

export interface StockItem {
  id: number;
  quantity: number;
  medication_batch: {
    batch_number: string;
    expiry_date: string | null;
    medication: {
      name: string;
      dosage?: string;
      form?: string;
    };
  };
}

export interface PaginatedStocks {
  data: StockItem[];
  current_page: number;
  last_page: number;
}

type UserRole =
  | 'superadmin'
  | 'receptionist'
  | 'nurse'
  | 'doctor'
  | 'lab_technician'
  | 'radiologist'
  | 'pharmacist'
  | 'nurse_aid'
  | 'admin'
  | 'specialist'
  | 'casualty'
  | 'user'
  | 'patient';

interface Props extends PageProps {
  clinic: Clinic;
  stocks: PaginatedStocks;
  users: { id: number; name: string }[];
  filters: {
    search?: string;
  };
}

export default function MedicationStockIndex() {
  const { clinic, stocks, users, filters } = usePage<Props>().props;

  const page = usePage<SharedData>();
  const userRole = page.props.auth.user?.role as UserRole | undefined;
  const canAct = ['superadmin', 'admin', 'pharmacist'].includes(userRole ?? '');

  const [searchQuery, setSearchQuery] = useState<string>(filters.search || '');
  const [pendingCount, setPendingCount] = useState(0);

  // Edit modal state (shadcn)
  const [isEditOpen, setIsEditOpen] = useState(false);
  const [editQuantity, setEditQuantity] = useState<number | ''>('');
  const [editMeta, setEditMeta] = useState<{ name: string; batch: string; expiry: string } | null>(null);

  // Transfer modal state
  const [isTransferOpen, setIsTransferOpen] = useState(false);
  const [transferQuantity, setTransferQuantity] = useState<number | ''>('');
  const [transferNurseId, setTransferNurseId] = useState<number | ''>('');

  // Return modal state
  const [isReturnOpen, setIsReturnOpen] = useState(false);
  const [returnQuantity, setReturnQuantity] = useState<number | ''>('');
  const [returnReason, setReturnReason] = useState('');
  const [returnDeleteIfZero, setReturnDeleteIfZero] = useState(false);

  const [selectedStock, setSelectedStock] = useState<StockItem | null>(null);

  // keep local input in sync with server filter when navigating/paginating
  useEffect(() => {
    setSearchQuery(filters.search || '');
  }, [filters.search]);

  useEffect(() => {
    const fetchPendingCount = async () => {
      try {
        const response = await axios.get(`${API}/clinics/${clinic.id}/pending-stock-transfers/count`);
        setPendingCount(response.data.pending_count);
      } catch (error) {
        console.error('Error fetching pending stock transfers count:', error);
      }
    };
    fetchPendingCount();
  }, [clinic.id]);

  // Debounced query when typing
  const debouncedSearch = useDebouncedCallback((val: string) => {
    router.get(
      route('clinics.medication-stocks.index', { clinic: clinic.id }),
      { search: val, page: 1 },
      { preserveState: true, replace: true, preserveScroll: true }
    );
  }, 500);

  const handleSearchClick = () => {
    router.get(
      route('clinics.medication-stocks.index', { clinic: clinic.id }),
      { search: searchQuery, page: 1 },
      { preserveState: true, replace: true, preserveScroll: true }
    );
  };

  const handleSearchInput = (e: React.ChangeEvent<HTMLInputElement>) => {
    const val = e.target.value;
    setSearchQuery(val);
    debouncedSearch(val);
  };

  const handlePageChange = (selected: { selected: number }) => {
    router.get(
      route('clinics.medication-stocks.index', { clinic: clinic.id }),
      { page: selected.selected + 1, search: searchQuery },
      { preserveState: true, replace: true, preserveScroll: true }
    );
  };

  const handleDelete = (id: number) => {
    if (!canAct) return;
    Swal.fire({
      title: 'Are you sure?',
      text: 'This batch will be permanently deleted.',
      icon: 'warning',
      showCancelButton: true,
      confirmButtonColor: '#e3342f',
      cancelButtonColor: '#6c757d',
      confirmButtonText: 'Yes, delete it!',
      width: '350px',
    }).then((result) => {
      if (result.isConfirmed) {
        router.delete(route('received-batches.destroy', id), {
          onSuccess: () => {
            Swal.fire({
              title: 'Deleted!',
              text: 'The batch has been deleted.',
              icon: 'success',
              timer: 1500,
              showConfirmButton: false,
            });
          },
          onError: () => {
            Swal.fire({
              title: 'Error!',
              text: 'Something went wrong.',
              icon: 'error',
            });
          },
        });
      }
    });
  };

  const formatExpiry = (d?: string | null) => (d ? moment(d).format('YYYY-MM-DD') : '-');

  const exportToExcel = () => {
    const data: Record<string, string | number>[] = stocks.data.map((stock) => ({
      'Medication Name': stock.medication_batch?.medication?.name || '-',
      Dosage: stock.medication_batch?.medication?.dosage || '-',
      Form: stock.medication_batch?.medication?.form || '-',
      Quantity: stock.quantity ?? '-',
      'Batch Number': stock.medication_batch?.batch_number || '-',
      'Expiry Date': formatExpiry(stock.medication_batch?.expiry_date),
    }));

    const worksheet = XLSX.utils.json_to_sheet(data);
    const keys = data.length > 0 ? Object.keys(data[0]) : [];
    worksheet['!cols'] = keys.map((key) => {
      const maxLength = Math.max(
        key.length,
        ...data.map((row) => (row[key as keyof typeof row] ? row[key as keyof typeof row].toString().length : 0))
      );
      return { wch: maxLength + 2 };
    });

    const workbook = XLSX.utils.book_new();
    XLSX.utils.book_append_sheet(workbook, worksheet, 'Medication Stocks');
    XLSX.writeFile(workbook, 'Medication_Stocks.xlsx', { compression: true });
  };

  // Server-generated XLSX with SweetAlert during download
  const downloadServerExcel = async () => {
    try {
      Swal.fire({
        title: 'Preparing download…',
        text: 'Generating clinic stock report',
        allowOutsideClick: false,
        didOpen: () => Swal.showLoading(),
        width: '420px',
      });

      const params = new URLSearchParams();
      if (searchQuery) params.append('search', searchQuery);
      // If you add UI for these, append them here:
      // params.append('expiry_from', 'YYYY-MM-DD');
      // params.append('expiry_to', 'YYYY-MM-DD');
      // params.append('include_zero', '1');

      const url =
        route('clinic-reports.stock-take', { clinic: clinic.id }) +
        (params.toString() ? `?${params.toString()}` : '');

      const res = await axios.get(url, { responseType: 'blob' });

      let filename = 'Clinic_Stock_Take.xlsx';
      const dispo = res.headers['content-disposition'];
      if (dispo) {
        const match = /filename[^;=\n]*=((['"]).*?\2|[^;\n]*)/.exec(dispo);
        if (match?.[1]) filename = decodeURIComponent(match[1].replace(/['"]/g, ''));
      }

      const blob = new Blob([res.data], {
        type: 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
      });
      const blobUrl = window.URL.createObjectURL(blob);
      const a = document.createElement('a');
      a.href = blobUrl;
      a.download = filename;
      document.body.appendChild(a);
      a.click();
      a.remove();
      window.URL.revokeObjectURL(blobUrl);

      Swal.close();
      Swal.fire({ icon: 'success', title: 'Download started', timer: 1200, showConfirmButton: false, width: '300px' });
    } catch (err: any) {
      Swal.close();
      const message = err?.response?.data?.message || err?.message || 'Failed to download the report.';
      Swal.fire({ icon: 'error', title: 'Error', text: message });
    }
  };

  // ----- shadcn: Edit Modal handlers -----
  const openEditModal = (stock: StockItem) => {
    if (!canAct) return;
    setSelectedStock(stock);
    setEditQuantity(stock.quantity ?? 0);
    setEditMeta({
      name: stock.medication_batch?.medication?.name || '',
      batch: stock.medication_batch?.batch_number || '',
      expiry: formatExpiry(stock.medication_batch?.expiry_date),
    });
    setIsEditOpen(true);
  };
  const closeEditModal = () => {
    setIsEditOpen(false);
    setSelectedStock(null);
    setEditQuantity('');
    setEditMeta(null);
  };
  const handleEditSubmit = async () => {
    if (!selectedStock || editQuantity === '' || editQuantity < 0) {
      return;
    }
    try {
      Swal.fire({ title: 'Updating stock...', allowOutsideClick: false, didOpen: () => Swal.showLoading() });
      await axios.put(`${API}/clinic-medication-stocks/${selectedStock.id}/update-quantity`, {
        quantity: Number(editQuantity),
      });
      Swal.close();
      Swal.fire({ icon: 'success', title: 'Updated!', text: 'Medication stock updated successfully.', timer: 1500, showConfirmButton: false });
      closeEditModal();
      router.reload();
    } catch (error) {
      Swal.close();
      Swal.fire({ icon: 'error', title: 'Error', text: 'Failed to update medication stock.' });
    }
  };

  // ----- shadcn: Transfer Modal handlers -----
  const openTransferModal = (stock: StockItem) => {
    if (!canAct) return;
    setSelectedStock(stock);
    setTransferQuantity('');
    setTransferNurseId('');
    setIsTransferOpen(true);
  };
  const closeTransferModal = () => {
    setIsTransferOpen(false);
    setSelectedStock(null);
    setTransferQuantity('');
    setTransferNurseId('');
  };
  const handleTransferSubmit = async () => {
    if (!selectedStock || transferQuantity === '' || Number(transferQuantity) < 1 || transferNurseId === '') {
      Swal.fire({ icon: 'warning', title: 'Incomplete', text: 'Please fill all required fields.' });
      return;
    }
    const qty = Number(transferQuantity);
    const max = selectedStock.quantity ?? 0;
    if (qty > max) {
      Swal.fire({ icon: 'warning', title: 'Too many units', text: `Only ${max} available to transfer.` });
      return;
    }
    try {
      Swal.fire({ title: 'Transferring stock...', allowOutsideClick: false, didOpen: () => Swal.showLoading() });
      await axios.post(`${API}/clinic-medication-stocks/${selectedStock.id}/transfer-to-nurse`, {
        quantity: qty,
        nurse_id: Number(transferNurseId),
      });
      Swal.close();
      Swal.fire({ icon: 'success', title: 'Transferred!', text: 'Medication stock transferred successfully.', timer: 1500, showConfirmButton: false });
      closeTransferModal();
      router.reload();
    } catch (error) {
      Swal.close();
      Swal.fire({ icon: 'error', title: 'Error', text: 'Failed to transfer medication stock.' });
    }
  };

  // ----- shadcn: Return Modal handlers -----
  const openReturnModal = (stock: StockItem) => {
    if (!canAct) return;
    setSelectedStock(stock);
    setReturnQuantity(1);
    setReturnReason('');
    setReturnDeleteIfZero(false);
    setIsReturnOpen(true);
  };
  const closeReturnModal = () => {
    setIsReturnOpen(false);
    setSelectedStock(null);
    setReturnQuantity('');
    setReturnReason('');
    setReturnDeleteIfZero(false);
  };
  const handleReturnSubmit = async () => {
    if (!selectedStock || returnQuantity === '' || Number(returnQuantity) < 1) {
      Swal.fire({ icon: 'warning', title: 'Invalid quantity', text: 'Please enter at least 1.' });
      return;
    }
    const qty = Number(returnQuantity);
    const max = selectedStock.quantity ?? 0;
    if (qty > max) {
      Swal.fire({ icon: 'warning', title: 'Too many units', text: `Only ${max} available to return.` });
      return;
    }
    try {
      Swal.fire({ title: 'Returning stock...', allowOutsideClick: false, didOpen: () => Swal.showLoading() });
      await axios.post(`${API}/central-stores/clinic-stocks/${selectedStock.id}/return`, {
        quantity: qty,
        delete_if_zero: !!returnDeleteIfZero,
        reason: returnReason || undefined,
      });
      Swal.close();
      Swal.fire({ icon: 'success', title: 'Returned!', text: 'Stock returned to central store.', timer: 1500, showConfirmButton: false });
      closeReturnModal();
      router.reload();
    } catch (error: any) {
      Swal.close();
      const message = error?.response?.data?.message || error?.message || 'Failed to return stock.';
      Swal.fire({ icon: 'error', title: 'Error', text: message });
    }
  };

  const breadcrumbs = useMemo(
    () => [
      { title: 'Clinics', href: route('clinics.index') },
      { title: clinic.name, href: route('clinics.show', { clinic: clinic.id }) },
      { title: 'Medication Stocks', href: route('clinics.medication-stocks.index', { clinic: clinic.id }) },
    ],
    [clinic]
  );

  return (
    <AppLayout breadcrumbs={breadcrumbs}>
      <Head title={`${clinic.name} - Medication Stocks`} />

      <div className="bg-background text-foreground mx-8 my-6 flex flex-1 flex-col gap-4 rounded-xl p-6 shadow-sm">
        {/* Header */}
        <div className="mb-4 flex flex-wrap items-center justify-between gap-4">
          <h1 className="text-2xl font-bold">{clinic.name} - Medication Stocks</h1>
          <div className="flex gap-4">
            {/* Download server-generated report (with SweetAlert) */}
            <Button onClick={downloadServerExcel} className="inline-flex items-center gap-2" style={{
              cursor: 'pointer'
            }}>
              <Download className="h-4 w-4" />
              Download Report
            </Button>



            <Button
              onClick={() => router.get(route('clinics.pending-stock-transfers', { clinic: clinic.id }))}
              className="relative"
            >
              Pending Stock Transfers
              {pendingCount > 0 && (
                <Badge variant="destructive" className="ml-2">
                  {pendingCount}
                </Badge>
              )}
            </Button>

            <Button
              onClick={() => router.get(route('clinics.request-medication.create', { clinic: clinic.id }))}
              variant="outline"
            >
              Request Medication
            </Button>
          </div>
        </div>

        {/* Search */}
        <div className="mb-4 flex gap-2">
          <div className="relative w-72">
            <SearchIcon className="text-muted-foreground absolute left-3 top-2.5 h-5 w-5" />
            <Input
              type="text"
              value={searchQuery}
              onChange={handleSearchInput}
              onKeyDown={(e) => e.key === 'Enter' && handleSearchClick()}
              placeholder="Search by medication name..."
              aria-label="Search by medication name"
              className="pl-10"
            />
          </div>
          <Button onClick={handleSearchClick}>Search</Button>
        </div>

        {/* Stock List */}
        <div className="overflow-x-auto rounded-xl border">
          <table className="min-w-full divide-y overflow-hidden rounded-xl bg-white text-sm dark:bg-neutral-900">
            <thead className="bg-muted">
              <tr>
                {['Medication', 'Dosage', 'Form', 'Quantity', 'Expiry', 'Batch']
                  .concat(canAct ? ['Actions'] : [])
                  .map((header) => (
                    <th
                      key={header}
                      className="text-muted-foreground px-4 py-3 text-left text-xs font-semibold uppercase tracking-wider"
                    >
                      {header}
                    </th>
                  ))}
              </tr>
            </thead>

            <tbody className="divide-y">
              {stocks.data.length === 0 && (
                <tr>
                  <td className="px-4 py-6 text-center text-sm text-gray-500" colSpan={canAct ? 7 : 6}>
                    No medication stocks found{searchQuery ? ` for “${searchQuery}”` : ''}.
                  </td>
                </tr>
              )}

              {stocks.data.map((stock) => {
                const medication = stock.medication_batch?.medication;

                return (
                  <tr key={stock.id} className="transition hover:bg-muted/50">
                    <td className="px-4 py-3 font-medium">{medication?.name || '-'}</td>
                    <td className="px-4 py-3">{medication?.dosage || '-'}</td>
                    <td className="px-4 py-3">{medication?.form || '-'}</td>
                    <td className="px-4 py-3">{stock.quantity}</td>
                    <td className="px-4 py-3">{formatExpiry(stock.medication_batch?.expiry_date)}</td>
                    <td className="px-4 py-3">{stock.medication_batch?.batch_number || '-'}</td>

                    {canAct && (
                      <td className="px-4 py-3">
                        <div className="flex flex-wrap items-center gap-2">
                          <Button
                            onClick={() => openEditModal(stock)}
                            className="inline-flex items-center justify-center rounded-full"
                            size="icon"
                            variant="secondary"
                            title="Edit"
                            style={{
                              cursor: 'pointer'
                            }}
                          >
                            <Pencil className="h-4 w-4" />
                          </Button>

                          <Button
                            onClick={() => handleDelete(stock.id)}
                            className="inline-flex items-center justify-center rounded-full"
                            size="icon"
                            variant="destructive"
                            title="Delete"
                            style={{
                              cursor: 'pointer'
                            }}
                          >
                            <Trash className="h-4 w-4" />
                          </Button>

                          <Button
                            onClick={() => openTransferModal(stock)}
                            variant="outline"
                            title="Transfer Medication"
                            style={{
                              cursor: 'pointer'
                            }}
                          >
                            Transfer To Cabinet
                          </Button>

                          <Button
                            onClick={() => openReturnModal(stock)}
                            variant="outline"
                            title="Return to Central Store"
                            style={{
                              cursor: 'pointer'
                            }}
                          >
                            Return to Central
                          </Button>
                        </div>
                      </td>
                    )}
                  </tr>
                );
              })}
            </tbody>
          </table>
        </div>

        {/* Pagination */}
        <div className="mt-6 flex justify-center">
          <ReactPaginate
            pageCount={stocks.last_page}
            forcePage={Math.max(0, (stocks.current_page || 1) - 1)}
            onPageChange={handlePageChange}
            marginPagesDisplayed={1}
            pageRangeDisplayed={3}
            previousLabel="← Prev"
            nextLabel="Next →"
            breakLabel="..."
            containerClassName="flex items-center gap-2 text-sm"
            pageClassName="px-3 py-1 border border-border rounded hover:bg-muted cursor-pointer"
            activeClassName="bg-blue-600 text-white"
            previousClassName="px-3 py-1 border border-border rounded hover:bg-muted cursor-pointer"
            nextClassName="px-3 py-1 border border-border rounded hover:bg-muted cursor-pointer"
            breakClassName="px-2"
          />
        </div>
      </div>

      {/* EDIT STOCK (Dialog) */}
      <Dialog open={isEditOpen} onOpenChange={(o) => (!o ? closeEditModal() : setIsEditOpen(o))}>
        <DialogContent className="sm:max-w-md">
          <DialogHeader>
            <DialogTitle>Edit Medication Stock</DialogTitle>
          </DialogHeader>

          <div className="grid gap-4">
            <div>
              <Label>Medication Name</Label>
              <Input value={editMeta?.name ?? ''} disabled />
            </div>
            <div>
              <Label>Quantity</Label>
              <Input
                type="number"
                min={0}
                value={editQuantity}
                onChange={(e) => setEditQuantity(e.target.value === '' ? '' : Number(e.target.value))}
              />
            </div>
            <div>
              <Label>Batch Number</Label>
              <Input value={editMeta?.batch ?? ''} disabled />
            </div>
            <div>
              <Label>Expiry Date</Label>
              <Input value={editMeta?.expiry ?? ''} disabled />
            </div>
          </div>

          <DialogFooter className="gap-2">
            <Button onClick={handleEditSubmit}>Save Changes</Button>
            <Button variant="outline" onClick={closeEditModal}>
              Cancel
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      {/* TRANSFER (Dialog) */}
      <Dialog open={isTransferOpen} onOpenChange={(o) => (!o ? closeTransferModal() : setIsTransferOpen(o))}>
        <DialogContent className="sm:max-w-md">
          <DialogHeader>
            <DialogTitle>Transfer Medication</DialogTitle>
          </DialogHeader>

          <div className="grid gap-4">
            <div>
              <Label>Transfer Quantity</Label>
              <Input
                type="number"
                min={1}
                max={selectedStock?.quantity ?? undefined}
                value={transferQuantity}
                onChange={(e) => setTransferQuantity(e.target.value === '' ? '' : Number(e.target.value))}
                onKeyDown={(e) => {
                  if (!/[0-9]|Backspace|Tab|ArrowLeft|ArrowRight|Delete/.test(e.key)) e.preventDefault();
                }}
              />
              {selectedStock && (
                <p className="mt-1 text-xs text-muted-foreground">
                  Available: {selectedStock.quantity}
                </p>
              )}
            </div>

            <div>
              <Label>Destination User</Label>
              <select
                className="w-full rounded-md border border-gray-300 bg-white px-3 py-2 text-sm dark:border-gray-700 dark:bg-gray-800 dark:text-white"
                value={transferNurseId === '' ? '' : Number(transferNurseId)}
                onChange={(e) => setTransferNurseId(e.target.value === '' ? '' : Number(e.target.value))}
              >
                <option value="">Select a user</option>
                {users.map((u) => (
                  <option key={u.id} value={u.id}>
                    {u.name}
                  </option>
                ))}
              </select>
            </div>
          </div>

          <DialogFooter className="gap-2">
            <Button onClick={handleTransferSubmit}>Transfer</Button>
            <Button variant="outline" onClick={closeTransferModal}>
              Cancel
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      {/* RETURN TO CENTRAL (Dialog) */}
      <Dialog open={isReturnOpen} onOpenChange={(o) => (!o ? closeReturnModal() : setIsReturnOpen(o))}>
        <DialogContent className="sm:max-w-md">
          <DialogHeader>
            <DialogTitle>Return to Central Store</DialogTitle>
          </DialogHeader>

          <div className="grid gap-4">
            <div>
              <Label>Quantity to return</Label>
              <Input
                type="number"
                min={1}
                max={selectedStock?.quantity ?? undefined}
                value={returnQuantity}
                onChange={(e) => setReturnQuantity(e.target.value === '' ? '' : Number(e.target.value))}
                onKeyDown={(e) => {
                  if (!/[0-9]|Backspace|Tab|ArrowLeft|ArrowRight|Delete/.test(e.key)) e.preventDefault();
                }}
              />
              {selectedStock && (
                <p className="mt-1 text-xs text-muted-foreground">Available: {selectedStock.quantity}</p>
              )}
            </div>

            <div>
              <Label>Reason (optional)</Label>
              <Textarea
                rows={3}
                placeholder="Add a note for the audit log (optional)"
                value={returnReason}
                onChange={(e) => setReturnReason(e.target.value)}
              />
            </div>

            <div className="flex items-center gap-2">
              <Checkbox
                id="deleteIfZero"
                checked={returnDeleteIfZero}
                onCheckedChange={(v) => setReturnDeleteIfZero(Boolean(v))}
              />
              <Label htmlFor="deleteIfZero">Delete clinic stock row if quantity becomes zero</Label>
            </div>
          </div>

          <DialogFooter className="gap-2">
            <Button onClick={handleReturnSubmit}>Return</Button>
            <Button variant="outline" onClick={closeReturnModal}>
              Cancel
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </AppLayout>
  );
}
