import { API } from '@/config';
import AppLayout from '@/layouts/app-layout';
import { type NavItem, type SharedData } from '@/types';
import { PageProps } from '@inertiajs/core';
import { Head, router, usePage } from '@inertiajs/react';
import axios from 'axios';
import { Pencil, Search, Trash, Download, RotateCcw } from 'lucide-react';
import { useState } from 'react';
import ReactPaginate from 'react-paginate';
import Swal from 'sweetalert2';
import * as XLSX from 'xlsx';
import { Modal, Form, Input, Select, Button, DatePicker, InputNumber, Checkbox } from 'antd';
import moment from 'moment';
import { Button as ShButton } from '@/components/ui/button';

export interface PaginatedStocks {
  data: {
    id: number;
    quantity: number;
    medication_batch: {
      batch_number: string;
      expiry_date: string;
      medication: {
        name: string;
        dosage?: string;
        form?: string;
      };
    };
    nurse: {
      id: number;
      name: string;
    };
  }[];
  current_page: number;
  last_page: number;
}

interface Props extends PageProps {
  user: {
    id: number;
    name: string;
    clinic_id: number;
  };
  stocks: PaginatedStocks;
  filters: {
    search?: string;
    search_user?: string;
  };
}



type AuthUser = {
  id: number;
  name: string;
  role?: string;
  roles?: string[];
  can?: Record<string, boolean>;
};

const { TextArea } = Input;

export default function NurseMedicationStock() {
  const { user, stocks, filters } = usePage<Props>().props;
  const [searchQuery, setSearchQuery] = useState(filters.search || '');
  const [searchUser, setSearchUser] = useState(filters.search_user || '');
  const page = usePage<SharedData>();


  const { auth } = usePage<{ auth: { user: AuthUser } }>().props;

  // visible if role is exactly "superadmin" (with a couple of safe fallbacks)
  const canModify =
    auth?.user?.role === 'superadmin' ||
    auth?.user?.roles?.includes?.('superadmin') ||
    auth?.user?.can?.delete_consultations === true;

  const [isEditModalOpen, setIsEditModalOpen] = useState(false);
  const [editFormData, setEditFormData] = useState({
    name: '',
    quantity: 0,
    batchNumber: '',
    expiryDate: null as null | moment.Moment,
  });

  const [selectedStock, setSelectedStock] = useState<null | typeof stocks.data[0]>(null);

  // Return-to-clinic modal state + form
  const [isReturnModalOpen, setIsReturnModalOpen] = useState(false);
  const [returnForm] = Form.useForm<{
    quantity: number;
    reason?: string;
    deleteIfZero?: boolean;
  }>();

  const handleSearch = () => {
    router.get(
      route('nurse-medication-stocks.index', { user: user?.id }),
      { search: searchQuery, search_user: searchUser },
      { preserveState: true, replace: true }
    );
  };

  const handlePageChange = (selected: { selected: number }) => {
    router.get(
      route('nurse-medication-stocks.index', { user: user?.id }),
      { page: selected.selected + 1, search: searchQuery, search_user: searchUser },
      { preserveState: true, replace: true }
    );
  };

  const handleDelete = (id: number) => {
    Swal.fire({
      title: 'Are you sure?',
      text: 'This batch will be permanently deleted.',
      icon: 'warning',
      showCancelButton: true,
      confirmButtonColor: '#e3342f',
      cancelButtonColor: '#6c757d',
      confirmButtonText: 'Yes, delete it!',
      width: '350px',
    }).then((result) => {
      if (result.isConfirmed) {
        router.delete(route('received-batches.destroy', id), {
          onSuccess: () => {
            Swal.fire({
              title: 'Deleted!',
              text: 'The batch has been deleted.',
              icon: 'success',
              timer: 1500,
              showConfirmButton: false,
            });
          },
          onError: () => {
            Swal.fire({
              title: 'Error!',
              text: 'Something went wrong.',
              icon: 'error',
            });
          },
        });
      }
    });
  };

  // Local client-side export (kept)
  const exportToExcel = () => {
    if (!stocks || !stocks.data) {
      Swal.fire({
        title: 'Error!',
        text: 'No data available to export.',
        icon: 'error',
      });
      return;
    }

    const data: Record<string, string | number>[] = stocks.data.map((stock) => ({
      'Medication Name': stock.medication_batch?.medication?.name || '-',
      Dosage: stock.medication_batch?.medication?.dosage || '-',
      Form: stock.medication_batch?.medication?.form || '-',
      Quantity: stock.quantity || '-',
      'Batch Number': stock.medication_batch?.batch_number || '-',
      'Expiry Date': stock.medication_batch?.expiry_date || '-',
      Nurse: stock.nurse?.name || '-',
    }));

    const worksheet = XLSX.utils.json_to_sheet(data);

    const keys = data.length > 0 ? Object.keys(data[0]) : [];
    worksheet['!cols'] = keys.map((key) => {
      const maxLength = Math.max(
        key.length,
        ...data.map((row) => (row[key as keyof typeof row] ? row[key as keyof typeof row].toString().length : 0))
      );
      return { wch: maxLength + 2 };
    });

    const workbook = XLSX.utils.book_new();
    XLSX.utils.book_append_sheet(workbook, worksheet, 'Nurse Medication Stocks');

    XLSX.writeFile(workbook, 'Nurse_Medication_Stocks.xlsx', { compression: true });
  };

  // Server-generated XLSX with SweetAlert during download
  const downloadServerExcel = async () => {
    try {
      Swal.fire({
        title: 'Preparing download…',
        text: 'Generating your stock-take report',
        allowOutsideClick: false,
        didOpen: () => Swal.showLoading(),
        width: '520px',
      });

      const params = new URLSearchParams();
      if (searchQuery) params.append('search', searchQuery);
      // Add more params if you add UI for them:
      // if (includeZero) params.append('include_zero', '1');
      // if (expiryFrom) params.append('expiry_from', 'YYYY-MM-DD');
      // if (expiryTo) params.append('expiry_to', 'YYYY-MM-DD');

      const url = route('nurse-reports.stock-take') + (params.toString() ? `?${params.toString()}` : '');

      const res = await axios.get(url, {
        responseType: 'blob',
      });

      let filename = 'Nurse_Stock_Take.xlsx';
      const dispo = res.headers['content-disposition'];
      if (dispo) {
        const match = /filename[^;=\n]*=((['"]).*?\2|[^;\n]*)/.exec(dispo);
        if (match?.[1]) {
          filename = decodeURIComponent(match[1].replace(/['"]/g, ''));
        }
      }

      const blob = new Blob([res.data], {
        type: 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
      });
      const blobUrl = window.URL.createObjectURL(blob);

      const a = document.createElement('a');
      a.href = blobUrl;
      a.download = filename;
      document.body.appendChild(a);
      a.click();
      a.remove();
      window.URL.revokeObjectURL(blobUrl);

      Swal.close();
      Swal.fire({
        icon: 'success',
        title: 'Download started',
        timer: 1200,
        showConfirmButton: false,
        width: '600px',
      });
    } catch (err: any) {
      Swal.close();
      const message =
        err?.response?.data?.message ||
        err?.message ||
        'Failed to download the stock-take report.';
      Swal.fire({
        icon: 'error',
        title: 'Error',
        text: message,
      });
    }
  };

  const openEditModal = (stock: typeof stocks.data[0]) => {
    setSelectedStock(stock);
    setEditFormData({
      name: stock.medication_batch?.medication?.name || '',
      quantity: stock.quantity || 0,
      batchNumber: stock.medication_batch?.batch_number || '',
      expiryDate: stock.medication_batch?.expiry_date ? moment(stock.medication_batch.expiry_date) : null,
    });
    setIsEditModalOpen(true);
  };

  const closeEditModal = () => {
    setIsEditModalOpen(false);
    setSelectedStock(null);
  };

  const handleEditSubmit = async (values: any) => {
    if (!selectedStock) return;

    try {
      Swal.fire({
        title: 'Updating stock...',
        allowOutsideClick: false,
        didOpen: () => Swal.showLoading(),
      });

      const payload = {
        quantity: values.quantity,
      };

      await axios.put(`${API}/nurse-medication-stocks/${selectedStock.id}/update-quantity`, payload);

      Swal.close();

      Swal.fire({
        icon: 'success',
        title: 'Updated!',
        text: 'Medication stock updated successfully.',
        timer: 1500,
        showConfirmButton: false,
      });

      closeEditModal();

      router.reload();
    } catch (error) {
      Swal.close();

      Swal.fire({
        icon: 'error',
        title: 'Error',
        text: 'Failed to update medication stock.',
      });
    }
  };

  // Return-to-clinic handlers
  const openReturnModal = (stock: typeof stocks.data[0]) => {
    setSelectedStock(stock);
    returnForm.setFieldsValue({
      quantity: 1,
      reason: '',
      deleteIfZero: false,
    });
    setIsReturnModalOpen(true);
  };

  const closeReturnModal = () => {
    setIsReturnModalOpen(false);
    setSelectedStock(null);
    returnForm.resetFields();
  };

  const handleReturnSubmit = async (values: { quantity: number; reason?: string; deleteIfZero?: boolean }) => {
    if (!selectedStock) return;

    const max = selectedStock.quantity ?? 0;
    if (!values.quantity || values.quantity < 1) {
      Swal.fire({ icon: 'warning', title: 'Invalid quantity', text: 'Please enter a quantity of at least 1.' });
      return;
    }
    if (values.quantity > max) {
      Swal.fire({
        icon: 'warning',
        title: 'Too many units',
        width: '400px',
        text: `You only have ${max} unit(s) available to return.`,
      });
      return;
    }

    try {
      Swal.fire({
        title: 'Processing return...',
        allowOutsideClick: false,
        didOpen: () => Swal.showLoading(),
      });

      await axios.post(`${API}/nurse-medication-stocks/${selectedStock.id}/return`, {
        quantity: values.quantity,
        delete_if_zero: !!values.deleteIfZero,
        reason: values.reason || undefined,
      });

      Swal.close();

      Swal.fire({
        icon: 'success',
        title: 'Returned!',
        text: 'Medication returned to clinic batch successfully.',
        timer: 1500,
        showConfirmButton: false,
      });

      closeReturnModal();

      router.reload();
    } catch (error: any) {
      Swal.close();
      const message =
        error?.response?.data?.message ||
        error?.message ||
        'Failed to return medication. Please try again.';
      Swal.fire({ icon: 'error', title: 'Error', text: message });
    }
  };

  if (!user || !stocks || !stocks.data) {
    return <div>Loading...</div>;
  }

  return (
    <>
      <AppLayout breadcrumbs={[{ title: 'Users', href: `${API}/users` }]}>
        <Head title={`${user?.name} - Medication Stocks`} />

        <div className="bg-background text-foreground mx-8 my-6 flex flex-1 flex-col gap-4 rounded-xl p-6 shadow-sm">
          <div className="mb-4 flex flex-wrap items-center justify-between gap-4">
            <h1 className="text-2xl font-bold">{user?.name} - Medication Stocks</h1>
            <div className="flex gap-2 sm:gap-4">
              {/* Server generated report with Swal */}
              <ShButton
                onClick={downloadServerExcel}
                type="button"
                variant="outline"
                style={{
                  cursor: 'pointer'
                }}
              >
                <Download className="h-4 w-4" />
                Download Report
              </ShButton>

              {/* Optional: keep local export */}

            </div>

          </div>

          <div className="mb-4 flex gap-2">
            <div className="relative w-72">
              <Search className="text-muted-foreground absolute top-2.5 left-3 h-5 w-5" />
              <input
                type="text"
                value={searchQuery}
                onChange={(e) => setSearchQuery(e.target.value)}
                onKeyDown={(e) => e.key === 'Enter' && handleSearch()}
                placeholder="Search medication stocks..."
                className="border-border bg-background placeholder-muted-foreground w-full rounded-lg border py-2 pr-3 pl-10 text-sm focus:ring-2 focus:ring-blue-600 focus:outline-none"
              />
            </div>
            <div className="relative w-72">
              <Search className="text-muted-foreground absolute top-2.5 left-3 h-5 w-5" />
              <input
                type="text"
                value={searchUser}
                onChange={(e) => setSearchUser(e.target.value)}
                onKeyDown={(e) => e.key === 'Enter' && handleSearch()}
                placeholder="Search by nurse..."
                className="border-border bg-background placeholder-muted-foreground w-full rounded-lg border py-2 pr-3 pl-10 text-sm focus:ring-2 focus:ring-blue-600 focus:outline-none"
              />
            </div>
            <button
              onClick={handleSearch}
              className="rounded-lg bg-blue-600 px-4 py-2 text-sm font-medium text-white hover:bg-blue-700"
              type="button"
            >
              Search
            </button>
            <button
              onClick={() => {
                setSearchQuery('');
                setSearchUser('');
                router.get(
                  route('nurse-medication-stocks.index', { user: user?.id }),
                  {},
                  { preserveState: false, replace: true }
                );
              }}
              className="rounded-lg bg-gray-500 px-4 py-2 text-sm font-medium text-white hover:bg-gray-600"
              type="button"
            >
              Reset
            </button>
          </div>

          <div className="overflow-x-auto rounded-xl border">
            <table className="min-w-full divide-y overflow-hidden rounded-xl bg-white text-sm dark:bg-neutral-900">
              <thead className="bg-muted">
                <tr>
                  {['Medication', 'Dosage', 'Form', 'Quantity', 'Expiry', 'Batch', 'Nurse', 'Actions'].map((header) => (
                    <th
                      key={header}
                      className="text-muted-foreground px-4 py-3 text-left text-xs font-semibold tracking-wider uppercase"
                    >
                      {header}
                    </th>
                  ))}
                </tr>
              </thead>
              <tbody className="divide-y">
                {stocks.data.length > 0 ? (
                  stocks.data.map((stock) => {
                    const medication = stock.medication_batch?.medication;

                    return (
                      <tr key={stock.id} className="hover:bg-muted/50 transition">
                        <td className="px-4 py-3 font-medium">{medication?.name || '-'}</td>
                        <td className="px-4 py-3">{medication?.dosage || '-'}</td>
                        <td className="px-4 py-3">{medication?.form || '-'}</td>
                        <td className="px-4 py-3">{stock.quantity}</td>
                        <td className="px-4 py-3">{stock.medication_batch?.expiry_date || '-'}</td>
                        <td className="px-4 py-3">{stock.medication_batch?.batch_number || '-'}</td>
                        <td className="px-4 py-3">{stock.nurse?.name || '-'}</td>
                        <td className="flex items-center gap-2 px-4 py-3">

                          {
                            canModify && (
                              <>

                                <button
                                  onClick={() => openEditModal(stock)}
                                  className="inline-flex items-center justify-center rounded-full bg-yellow-500 p-2 text-white hover:bg-yellow-600"
                                  title="Edit"
                                  type="button"
                                  style={{ cursor: 'pointer' }}
                                >
                                  <Pencil className="h-4 w-4" />
                                </button>

                                <button
                                  onClick={() => openReturnModal(stock)}
                                  className="inline-flex items-center justify-center rounded-full bg-emerald-600 p-2 text-white hover:bg-emerald-700"
                                  title="Return to Clinic"
                                  type="button"
                                  style={{ cursor: 'pointer' }}
                                >
                                  <RotateCcw className="h-4 w-4" />
                                </button>

                                <button
                                  onClick={() => handleDelete(stock.id)}
                                  className="inline-flex items-center justify-center rounded-full bg-red-600 p-2 text-white hover:bg-red-700"
                                  title="Delete"
                                  type="button"
                                  style={{ cursor: 'pointer' }}
                                >
                                  <Trash className="h-4 w-4" />
                                </button>

                              </>


                            )
                          }

                          {/* Return-to-clinic button */}

                        </td>
                      </tr>
                    );
                  })
                ) : (
                  <tr>
                    <td colSpan={8} className="px-4 py-6 text-center text-muted-foreground">
                      No medication stock found.
                    </td>
                  </tr>
                )}
              </tbody>
            </table>
          </div>

          <div className="mt-6 flex justify-center">
            <ReactPaginate
              pageCount={stocks.last_page}
              forcePage={stocks.current_page - 1}
              onPageChange={handlePageChange}
              marginPagesDisplayed={1}
              pageRangeDisplayed={3}
              previousLabel="← Prev"
              nextLabel="Next →"
              breakLabel="..."
              containerClassName="flex items-center gap-2 text-sm"
              pageClassName="px-3 py-1 border border-border rounded hover:bg-muted cursor-pointer"
              activeClassName="bg-blue-600 text-white"
              previousClassName="px-3 py-1 border border-border rounded hover:bg-muted cursor-pointer"
              nextClassName="px-3 py-1 border border-border rounded hover:bg-muted cursor-pointer"
              breakClassName="px-2"
            />
          </div>
        </div>

        {/* Edit Modal */}
        <Modal
          title="Edit Medication Stock"
          open={isEditModalOpen}
          onCancel={closeEditModal}
          footer={null}
          centered
          bodyStyle={{ maxHeight: '70vh', overflowY: 'auto' }}
        >
          <Form layout="vertical" initialValues={editFormData} onFinish={handleEditSubmit} key={selectedStock?.id}>
            <Form.Item
              label="Medication Name"
              name="name"
              rules={[{ required: true, message: 'Please enter medication name' }]}
            >
              <Input disabled />
            </Form.Item>
            <Form.Item label="Quantity" name="quantity" rules={[{ required: true, message: 'Please enter quantity' }]}>
              <InputNumber min={0} style={{ width: '100%' }} />
            </Form.Item>
            <Form.Item
              label="Batch Number"
              name="batchNumber"
              rules={[{ required: true, message: 'Please enter batch number' }]}
            >
              <Input disabled />
            </Form.Item>
            <Form.Item
              label="Expiry Date"
              name="expiryDate"
              rules={[{ required: true, message: 'Please enter expiry date' }]}
            >
              <DatePicker style={{ width: '100%' }} disabled />
            </Form.Item>
            <Form.Item className="flex justify-end gap-2">
              <Button
                type="primary"
                htmlType="submit"
                style={{
                  cursor: 'pointer',
                  margin: '2px',
                }}
              >
                Save Changes
              </Button>
              <Button
                onClick={closeEditModal}
                style={{
                  cursor: 'pointer',
                  margin: '2px',
                }}
              >
                Cancel
              </Button>
            </Form.Item>
          </Form>
        </Modal>

        {/* Return-to-Clinic Modal */}
        <Modal
          title={selectedStock ? `Return to Clinic — ${selectedStock.medication_batch?.medication?.name ?? ''}` : 'Return to Clinic'}
          open={isReturnModalOpen}
          onCancel={closeReturnModal}
          footer={null}
          centered
          destroyOnClose
        >
          <Form
            form={returnForm}
            layout="vertical"
            onFinish={handleReturnSubmit}
            key={selectedStock?.id ? `return-${selectedStock.id}` : 'return'}
            initialValues={{ quantity: 1, reason: '', deleteIfZero: false }}
          >
            <Form.Item label="Quantity to return" name="quantity" rules={[{ required: true, message: 'Enter a quantity' }]}>
              <InputNumber
                min={1}
                max={selectedStock?.quantity ?? 1}
                style={{ width: '100%' }}
                placeholder="e.g. 5"
              />
            </Form.Item>

            <Form.Item label="Reason (optional)" name="reason">
              <TextArea rows={3} placeholder="Add a note for the log (optional)" />
            </Form.Item>

            <Form.Item name="deleteIfZero" valuePropName="checked">
              <Checkbox>Delete nurse stock row if quantity becomes zero</Checkbox>
            </Form.Item>

            <div className="mt-4 flex justify-end gap-2">
              <Button onClick={closeReturnModal}>Cancel</Button>
              <Button type="primary" htmlType="submit">
                Return
              </Button>
            </div>
          </Form>
        </Modal>
      </AppLayout>
    </>
  );
}
