import { API } from '@/config';
import AppLayout from '@/layouts/app-layout';
import { Clinic } from '@/types/medication';
import { PageProps } from '@inertiajs/core';
import { Head, router, usePage } from '@inertiajs/react';
import { Pencil, Search, Check, ArrowLeft } from 'lucide-react'; // Added ArrowLeft icon
import { useState, useEffect, useCallback } from 'react';
import ReactPaginate from 'react-paginate';
import { Modal, Input } from 'antd';
import moment from 'moment';
import { Button } from '@/components/ui/button';
import { Link } from '@inertiajs/react'; // Added Link component

export interface PaginatedStocks {
    data: {
        id: number;
        quantity: number;
        transfer_date: string;
        created_at: string;
        medication_batch: {
            batch_number: string;
            expiry_date: string;
            medication: {
                name: string;
                dosage?: string;
                form?: string;
            };
        };
    }[];
    current_page: number;
    last_page: number;
}

interface Props extends PageProps {
    clinic: Clinic;
    stocks: PaginatedStocks;
    filters: {
        search?: string;
    };
}

export default function PendingStockTransfers() {
    const { clinic, stocks, filters } = usePage<Props>().props;

    const [searchQuery, setSearchQuery] = useState(filters.search || '');
    const [isModalVisible, setIsModalVisible] = useState(false);
    const [selectedStockId, setSelectedStockId] = useState<number | null>(null);
    const [notes, setNotes] = useState('');
    const [quantity, setQuantity] = useState<number | null>(null);
    const [oldQuantity, setOldQuantity] = useState<number | null>(null);

    // Debounce function
    const debounce = (func: Function, delay: number) => {
        let timeoutId: NodeJS.Timeout;
        return (...args: any[]) => {
            clearTimeout(timeoutId);
            timeoutId = setTimeout(() => func.apply(null, args), delay);
        };
    };

    // Debounced search function
    const debouncedSearch = useCallback(
        debounce((query: string) => {
            router.get(
                `${API}/clinics/${clinic.id}/pending-stock-transfers`,
                { search: query },
                { preserveState: true, replace: true }
            );
        }, 500),
        [clinic.id]
    );

    // Handle search input change with debounce
    const handleSearchChange = (e: React.ChangeEvent<HTMLInputElement>) => {
        const query = e.target.value;
        setSearchQuery(query);
        debouncedSearch(query);
    };

    // Clear timeout on component unmount
    useEffect(() => {
        return () => {
            // Cleanup any pending debounced calls
            clearTimeout(debouncedSearch as any);
        };
    }, []);

    const handlePageChange = (selected: { selected: number }) => {
        router.get(
            `${API}/clinics/${clinic.id}/pending-stock-transfers`,
            { page: selected.selected + 1, search: searchQuery },
            { preserveState: true, replace: true }
        );
    };

    const showApproveModal = (id: number, currentQuantity: number) => {
        setSelectedStockId(id);
        setNotes('');
        setOldQuantity(currentQuantity);
        setQuantity(currentQuantity);
        setIsModalVisible(true);
    };

    const handleModalOk = () => {
        if (!notes.trim() || !selectedStockId || quantity === null || oldQuantity === null) return;

        router.post(
            `${API}/clinics/${clinic.id}/stock-transfers/${selectedStockId}/approve`,
            { 
                notes, 
                old_quantity: oldQuantity, 
                quantity 
            },
            {
                onSuccess: () => {
                    setIsModalVisible(false);
                    setSelectedStockId(null);
                    setNotes('');
                    setOldQuantity(null);
                    setQuantity(null);
                },
                onError: () => {
                    setIsModalVisible(false);
                },
            }
        );
    };

    const handleModalCancel = () => {
        setIsModalVisible(false);
        setNotes('');
        setSelectedStockId(null);
        setOldQuantity(null);
        setQuantity(null);
    };

    const breadcrumbs = [
        { title: 'Clinics', href: `${API}/clinics` },
        { title: clinic.name, href: `${API}/clinics/${clinic.id}` },
        { title: 'Medication Stocks', href: `${API}/clinics/${clinic.id}/medication-stocks` },
        { title: 'Pending Stock Transfers', href: `${API}/clinics/${clinic.id}/pending-stock-transfers` },
    ];

    return (
        <AppLayout breadcrumbs={breadcrumbs}>
            <Head title={`${clinic.name} - Pending Stock Transfers`} />
            <div className="bg-background text-foreground mx-8 my-6 flex flex-1 flex-col gap-4 rounded-xl p-6 shadow-sm">
                {/* Header with Back Button */}
                <div className="mb-4 flex flex-wrap items-center justify-between gap-4">
                    <div className="flex items-center gap-4">
                        <Link
                            href={`${API}/clinics/${clinic.id}/medication-stocks`}
                            className="inline-flex items-center gap-2 rounded-md bg-gray-200 px-3 py-2 text-sm font-medium text-gray-700 transition duration-150 hover:bg-gray-300 dark:bg-gray-700 dark:text-gray-200 dark:hover:bg-gray-600"
                        >
                            <ArrowLeft className="h-4 w-4" />
                            Back to Medication Stocks
                        </Link>
                        <h1 className="text-2xl font-bold">{clinic.name} - Pending Stock Transfers</h1>
                    </div>
                </div>

                {/* Search */}
                <div className="mb-4 flex gap-2">
                    <div className="relative w-72">
                        <Search className="text-muted-foreground absolute top-2.5 left-3 h-5 w-5" />
                        <input
                            type="text"
                            value={searchQuery}
                            onChange={handleSearchChange}
                            placeholder="Search by medication name..."
                            className="border-border bg-background placeholder-muted-foreground w-full rounded-lg border py-2 pr-3 pl-10 text-sm focus:ring-2 focus:ring-blue-600 focus:outline-none"
                        />
                    </div>
                </div>

                {/* Table */}
                <div className="overflow-x-auto rounded-xl border">
                    <table className="min-w-full divide-y overflow-hidden rounded-xl bg-white text-sm dark:bg-neutral-900">
                        <thead className="bg-muted">
                            <tr>
                                {['Date Of Transfer', 'Medication', 'Dosage', 'Form', 'Quantity', 'Expiry', 'Batch', 'Actions'].map((header) => (
                                    <th
                                        key={header}
                                        className="text-muted-foreground px-4 py-3 text-left text-xs font-semibold tracking-wider uppercase"
                                    >
                                        {header}
                                    </th>
                                ))}
                            </tr>
                        </thead>
                        <tbody className="divide-y">
                            {stocks.data.length > 0 ? (
                                stocks.data.map((stock) => {
                                    const medication = stock.medication_batch?.medication;

                                    return (
                                        <tr key={stock.id} className="hover:bg-muted/50 transition">
                                            <td className="px-4 py-3 font-medium">
                                                {stock.transfer_date 
                                                    ? moment(stock.transfer_date).format('MMMM Do YYYY')
                                                    : stock.created_at
                                                    ? moment(stock.created_at).format('MMMM Do YYYY')
                                                    : '-'}
                                            </td>
                                            <td className="px-4 py-3 font-medium">{medication?.name || '-'}</td>
                                            <td className="px-4 py-3">{medication?.dosage || '-'}</td>
                                            <td className="px-4 py-3">{medication?.form || '-'}</td>
                                            <td className="px-4 py-3">
                                                <span className="inline-block rounded-full bg-blue-100 px-3 py-1 text-xs font-semibold text-blue-800 dark:bg-blue-900 dark:text-blue-100">
                                                    {stock.quantity}
                                                </span>
                                            </td>
                                            <td className="px-4 py-3">{stock.medication_batch?.expiry_date || '-'}</td>
                                            <td className="px-4 py-3">{stock.medication_batch?.batch_number || '-'}</td>
                                            <td className="flex items-center gap-2 px-4 py-3">
                                                <button
                                                    onClick={() => showApproveModal(stock.id, stock.quantity)}
                                                    className="inline-flex items-center gap-2 rounded-md bg-green-600 px-4 py-2 text-sm font-medium text-white transition duration-150 hover:bg-green-700"
                                                    style={{ cursor: 'pointer' }}
                                                    title="Approve"
                                                >
                                                    <Check className="h-4 w-4" />
                                                    Approve
                                                </button>
                                            </td>
                                        </tr>
                                    );
                                })
                            ) : (
                                <tr>
                                    <td colSpan={8} className="px-4 py-6 text-center text-sm text-muted-foreground">
                                        No pending stock transfers found.
                                    </td>
                                </tr>
                            )}
                        </tbody>
                    </table>
                </div>

                {/* Pagination */}
                <div className="mt-6 flex justify-center">
                    <ReactPaginate
                        pageCount={stocks.last_page}
                        forcePage={stocks.current_page - 1}
                        onPageChange={handlePageChange}
                        marginPagesDisplayed={1}
                        pageRangeDisplayed={3}
                        previousLabel="← Prev"
                        nextLabel="Next →"
                        breakLabel="..."
                        containerClassName="flex items-center gap-2 text-sm"
                        pageClassName="px-3 py-1 border border-border rounded hover:bg-muted"
                        activeClassName="bg-blue-600 text-white"
                        previousClassName="px-3 py-1 border border-border rounded hover:bg-muted"
                        nextClassName="px-3 py-1 border border-border rounded hover:bg-muted"
                        breakClassName="px-2"
                    />
                </div>
            </div>

            {/* Ant Design Modal */}
            <Modal
                title="Approve Stock Transfer"
                visible={isModalVisible}
                onOk={handleModalOk}
                onCancel={handleModalCancel}
                okText="Approve"
                cancelText="Cancel"
            >
                <div className="flex flex-col gap-3">
                    {oldQuantity !== null && (
                        <p className="text-sm text-muted-foreground">
                            Original requested quantity: <strong>{oldQuantity}</strong> units.
                            If the quantity you received is different, you can update it below and add any notes.
                        </p>
                    )}

                    <Input
                        type="number"
                        placeholder="Quantity"
                        value={quantity ?? ''}
                        onChange={(e) => setQuantity(Number(e.target.value))}
                        min={1}
                    />

                    <Input.TextArea
                        rows={4}
                        placeholder="Enter notes"
                        value={notes}
                        onChange={(e) => setNotes(e.target.value)}
                    />
                </div>
            </Modal>
        </AppLayout>
    );
}