import * as React from 'react';
import { Head, Link, usePage } from '@inertiajs/react';
import AppLayout from '@/layouts/app-layout';
import { BreadcrumbItem } from '@/types';
import moment from 'moment';

// shadcn/ui
import { Card, CardHeader, CardTitle, CardContent, CardDescription } from '@/components/ui/card';
import { Tabs, TabsList, TabsTrigger, TabsContent } from '@/components/ui/tabs';
import { Table, TableHeader, TableRow, TableHead, TableBody, TableCell } from '@/components/ui/table';
import { Badge } from '@/components/ui/badge';
import { Separator } from '@/components/ui/separator';
import { Avatar, AvatarFallback } from '@/components/ui/avatar';
import { Button } from '@/components/ui/button';

// icons
import { Building2, Users, Stethoscope, Activity, ClipboardList, Pill, Boxes, Phone, Mail, MapPin, User2 } from 'lucide-react';

type Clinic = {
  id: number;
  name: string;
  address?: string | null;
  city?: string | null;
  country?: string | null;
  afhoz_number?: string | null;
  contact_person?: string | null;
  manager?: string | null;
  email_address?: string | null;

  // counts injected by loadCount()
  users_count?: number;
  triages_count?: number;
  consultations_count?: number;
  dispensations_count?: number;
  medication_requests_count?: number;
  medication_stocks_count?: number;

  // custom attribute we attached in controller
  unique_patients_count?: number;
};

type RecentUser = { id: number; name: string; email?: string | null; role?: string | null; created_at?: string };
type RecentPatient = { id: number; full_name: string; seen_at?: string };
type RecentTriage = {
  id: number;
  triage_level?: string | null;
  is_active: boolean;
  created_at?: string;
  patient?: { id: number; full_name: string } | null;
  consultor?: { id: number; name: string } | null;
};
type RecentConsultation = {
  id: number;
  consultation_date?: string | null;
  diagnosis?: string | null;
  patient?: { id: number; full_name: string } | null;
  doctor?: { id: number; name: string } | null;
};
type RecentMedicationRequest = {
  id: number;
  status?: string | null;
  created_at?: string;
  requested_by?: { id: number; name: string } | null;
};
type RecentStock = {
  id: number;
  medication?: { id: number; name: string } | null;
  quantity?: number | null;
  updated_at?: string;
};

type PageProps = {
  clinic: Clinic;
  recent_users: RecentUser[];
  recent_patients: RecentPatient[];
  recent_triages: RecentTriage[];
  recent_consultations: RecentConsultation[];
  recent_medication_requests: RecentMedicationRequest[];
  recent_stocks: RecentStock[];
};

const StatCard: React.FC<{
  icon: React.ReactNode;
  label: string;
  value: number | undefined;
  hint?: string;
}> = ({ icon, label, value, hint }) => (
  <Card>
    <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
      <CardTitle className="text-sm font-medium">{label}</CardTitle>
      <div className="text-muted-foreground">{icon}</div>
    </CardHeader>
    <CardContent>
      <div className="text-3xl font-bold">{value ?? 0}</div>
      {hint ? <p className="text-xs text-muted-foreground mt-1">{hint}</p> : null}
    </CardContent>
  </Card>
);

export default function ClinicShow() {
  const { clinic, recent_users, recent_patients, recent_triages, recent_consultations, recent_medication_requests, recent_stocks } =
    usePage<PageProps>().props;

  const breadcrumbs: BreadcrumbItem[] = [
    { title: 'Clinics', href: route('clinics.index') },
    { title: clinic.name ?? 'Clinic', href: route('clinics.show', clinic.id) },
  ];

  const AddressLine = () => (
    <div className="flex items-center gap-2 text-sm text-muted-foreground">
      <MapPin className="h-4 w-4" />
      <span>
        {[clinic.address, clinic.city, clinic.country].filter(Boolean).join(', ') || '—'}
      </span>
    </div>
  );

  const ContactLine = () => (
    <div className="flex flex-wrap items-center gap-x-6 gap-y-2 text-sm text-muted-foreground">
      <span className="inline-flex items-center gap-2">
        <Phone className="h-4 w-4" />
        {clinic.contact_person ? (
          <span>{clinic.contact_person}</span>
        ) : (
          <span className="text-muted-foreground/70">No contact person</span>
        )}
      </span>
      <span className="inline-flex items-center gap-2">
        <Mail className="h-4 w-4" />
        {clinic.email_address || <span className="text-muted-foreground/70">No email</span>}
      </span>
      <span className="inline-flex items-center gap-2">
        <User2 className="h-4 w-4" />
        {clinic.manager || <span className="text-muted-foreground/70">No manager</span>}
      </span>
    </div>
  );

  return (
    <AppLayout breadcrumbs={breadcrumbs}>
      <Head title={`${clinic.name} • Clinic`} />

      <div className="w-full px-4 py-6 sm:px-6 space-y-6">
        {/* Header */}
        <div className="flex flex-col gap-2">
          <div className="flex items-center gap-3">
            <Building2 className="h-6 w-6 text-primary" />
            <h1 className="text-2xl font-bold">{clinic.name}</h1>
          </div>
          <AddressLine />
          <ContactLine />
          {clinic.afhoz_number ? (
            <div className="text-xs text-muted-foreground">
              AFHOZ #: <span className="font-medium text-foreground">{clinic.afhoz_number}</span>
            </div>
          ) : null}
        </div>

        {/* Stats */}
        <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-4">
          <StatCard icon={<Users className="h-4 w-4" />} label="Staff" value={clinic.users_count} hint="Total users in this clinic" />
          <StatCard icon={<ClipboardList className="h-4 w-4" />} label="Patients" value={clinic.unique_patients_count} hint="Unique patients seen here" />
          <StatCard icon={<Activity className="h-4 w-4" />} label="Triages" value={clinic.triages_count} />
          <StatCard icon={<Stethoscope className="h-4 w-4" />} label="Consultations" value={clinic.consultations_count} />
          <StatCard icon={<Pill className="h-4 w-4" />} label="Dispensations" value={clinic.dispensations_count} />
          <StatCard icon={<Boxes className="h-4 w-4" />} label="Med Requests" value={clinic.medication_requests_count} />
          <StatCard icon={<Boxes className="h-4 w-4" />} label="Stock Items" value={clinic.medication_stocks_count} />
        </div>

        <Separator />

        {/* Tabs */}
        <Tabs defaultValue="overview" className="w-full">
          <TabsList className="flex flex-wrap">
            <TabsTrigger value="overview">Overview</TabsTrigger>
            <TabsTrigger value="triages">Recent Triages</TabsTrigger>
            <TabsTrigger value="consultations">Recent Consultations</TabsTrigger>
            <TabsTrigger value="patients">Recent Patients</TabsTrigger>
            <TabsTrigger value="staff">Recent Staff</TabsTrigger>
            <TabsTrigger value="medreqs">Med Requests</TabsTrigger>
            <TabsTrigger value="stock">Stock Peek</TabsTrigger>
          </TabsList>

          {/* Overview */}
          <TabsContent value="overview" className="mt-4">
            <Card>
              <CardHeader>
                <CardTitle>Clinic Overview</CardTitle>
                <CardDescription>Key details about this clinic.</CardDescription>
              </CardHeader>
              <CardContent className="grid gap-6 md:grid-cols-2">
                <div>
                  <div className="text-sm text-muted-foreground">Address</div>
                  <div className="mt-1">{[clinic.address, clinic.city, clinic.country].filter(Boolean).join(', ') || '—'}</div>
                </div>
                <div>
                  <div className="text-sm text-muted-foreground">Manager</div>
                  <div className="mt-1">{clinic.manager || '—'}</div>
                </div>
                <div>
                  <div className="text-sm text-muted-foreground">Contact Person</div>
                  <div className="mt-1">{clinic.contact_person || '—'}</div>
                </div>
                <div>
                  <div className="text-sm text-muted-foreground">Email</div>
                  <div className="mt-1">{clinic.email_address || '—'}</div>
                </div>
                <div>
                  <div className="text-sm text-muted-foreground">AFHOZ #</div>
                  <div className="mt-1">{clinic.afhoz_number || '—'}</div>
                </div>
              </CardContent>
            </Card>
          </TabsContent>

          {/* Triages */}
          <TabsContent value="triages" className="mt-4">
            <Card>
              <CardHeader>
                <CardTitle>Recent Triages</CardTitle>
                <CardDescription>Latest triage activity for this clinic.</CardDescription>
              </CardHeader>
              <CardContent>
                <Table>
                  <TableHeader>
                    <TableRow>
                      <TableHead>Patient</TableHead>
                      <TableHead>Consultor</TableHead>
                      <TableHead>Level</TableHead>
                      <TableHead>Status</TableHead>
                      <TableHead>Date</TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {recent_triages.length ? recent_triages.map((t) => (
                      <TableRow key={t.id}>
                        <TableCell>{t.patient?.full_name ?? '—'}</TableCell>
                        <TableCell>{t.consultor?.name ?? '—'}</TableCell>
                        <TableCell>
                          {t.triage_level ? <Badge variant="secondary">{t.triage_level}</Badge> : '—'}
                        </TableCell>
                        <TableCell>
                          <Badge className={t.is_active ? 'bg-emerald-600' : 'bg-gray-500'}>
                            {t.is_active ? 'Active' : 'Closed'}
                          </Badge>
                        </TableCell>
                        <TableCell>{t.created_at ? moment(t.created_at).format('D MMM YYYY, HH:mm') : '—'}</TableCell>
                      </TableRow>
                    )) : (
                      <TableRow><TableCell colSpan={5} className="text-center text-muted-foreground">No recent triages.</TableCell></TableRow>
                    )}
                  </TableBody>
                </Table>
              </CardContent>
            </Card>
          </TabsContent>

          {/* Consultations */}
          <TabsContent value="consultations" className="mt-4">
            <Card>
              <CardHeader>
                <CardTitle>Recent Consultations</CardTitle>
                <CardDescription>Latest consultations across this clinic.</CardDescription>
              </CardHeader>
              <CardContent>
                <Table>
                  <TableHeader>
                    <TableRow>
                      <TableHead>Patient</TableHead>
                      <TableHead>Doctor</TableHead>
                      <TableHead>Diagnosis</TableHead>
                      <TableHead>Date</TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {recent_consultations.length ? recent_consultations.map((c) => (
                      <TableRow key={c.id}>
                        <TableCell>{c.patient?.full_name ?? '—'}</TableCell>
                        <TableCell>{c.doctor?.name ?? '—'}</TableCell>
                        <TableCell className="max-w-[360px] truncate" title={c.diagnosis ?? ''}>
                          {c.diagnosis ?? '—'}
                        </TableCell>
                        <TableCell>{c.consultation_date ? moment(c.consultation_date).format('D MMM YYYY') : '—'}</TableCell>
                      </TableRow>
                    )) : (
                      <TableRow><TableCell colSpan={4} className="text-center text-muted-foreground">No recent consultations.</TableCell></TableRow>
                    )}
                  </TableBody>
                </Table>
              </CardContent>
            </Card>
          </TabsContent>

          {/* Patients */}
          <TabsContent value="patients" className="mt-4">
            <Card>
              <CardHeader>
                <CardTitle>Recent Patients</CardTitle>
                <CardDescription>Unique patients recently triaged at this clinic.</CardDescription>
              </CardHeader>
              <CardContent>
                <Table>
                  <TableHeader>
                    <TableRow>
                      <TableHead>Name</TableHead>
                      <TableHead>Last Seen</TableHead>
                      <TableHead>Actions</TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {recent_patients.length ? recent_patients.map((p) => (
                      <TableRow key={p.id}>
                        <TableCell>{p.full_name}</TableCell>
                        <TableCell>{p.seen_at ? moment(p.seen_at).format('D MMM YYYY, HH:mm') : '—'}</TableCell>
                        <TableCell>
                          <Button asChild size="sm" variant="secondary">
                            <Link href={route('patients.show', p.id)}>View</Link>
                          </Button>
                        </TableCell>
                      </TableRow>
                    )) : (
                      <TableRow><TableCell colSpan={3} className="text-center text-muted-foreground">No recent patients.</TableCell></TableRow>
                    )}
                  </TableBody>
                </Table>
              </CardContent>
            </Card>
          </TabsContent>

          {/* Staff */}
          <TabsContent value="staff" className="mt-4">
            <Card>
              <CardHeader>
                <CardTitle>Recent Staff Members</CardTitle>
                <CardDescription>Latest users added to this clinic.</CardDescription>
              </CardHeader>
              <CardContent>
                <Table>
                  <TableHeader>
                    <TableRow>
                      <TableHead>User</TableHead>
                      <TableHead>Email</TableHead>
                      <TableHead>Role</TableHead>
                      <TableHead>Joined</TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {recent_users.length ? recent_users.map((u) => (
                      <TableRow key={u.id}>
                        <TableCell>
                          <div className="flex items-center gap-2">
                            <Avatar className="h-8 w-8">
                              <AvatarFallback>{(u.name ?? 'U').slice(0,2).toUpperCase()}</AvatarFallback>
                            </Avatar>
                            <span className="font-medium">{u.name}</span>
                          </div>
                        </TableCell>
                        <TableCell className="text-muted-foreground">{u.email ?? '—'}</TableCell>
                        <TableCell>
                          {u.role ? <Badge variant="outline" className="capitalize">{u.role.replace(/_/g, ' ')}</Badge> : '—'}
                        </TableCell>
                        <TableCell>{u.created_at ? moment(u.created_at).format('D MMM YYYY') : '—'}</TableCell>
                      </TableRow>
                    )) : (
                      <TableRow><TableCell colSpan={4} className="text-center text-muted-foreground">No recent users.</TableCell></TableRow>
                    )}
                  </TableBody>
                </Table>
              </CardContent>
            </Card>
          </TabsContent>

          {/* Medication Requests */}
          <TabsContent value="medreqs" className="mt-4">
            <Card>
              <CardHeader>
                <CardTitle>Recent Medication Requests</CardTitle>
                <CardDescription>Last 5 requests raised by clinic staff.</CardDescription>
              </CardHeader>
              <CardContent>
                <Table>
                  <TableHeader>
                    <TableRow>
                      <TableHead>Requested By</TableHead>
                      <TableHead>Status</TableHead>
                      <TableHead>Date</TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {recent_medication_requests.length ? recent_medication_requests.map((r) => (
                      <TableRow key={r.id}>
                        <TableCell>{r.requested_by?.name ?? '—'}</TableCell>
                        <TableCell>
                          {r.status ? <Badge className="capitalize">{r.status.replace(/_/g, ' ')}</Badge> : '—'}
                        </TableCell>
                        <TableCell>{r.created_at ? moment(r.created_at).format('D MMM YYYY, HH:mm') : '—'}</TableCell>
                      </TableRow>
                    )) : (
                      <TableRow><TableCell colSpan={3} className="text-center text-muted-foreground">No recent medication requests.</TableCell></TableRow>
                    )}
                  </TableBody>
                </Table>
              </CardContent>
            </Card>
          </TabsContent>

          {/* Stock Peek */}
          <TabsContent value="stock" className="mt-4">
            <Card>
              <CardHeader>
                <CardTitle>Recent Stock Updates</CardTitle>
                <CardDescription>Quick view of recently updated medication stocks.</CardDescription>
              </CardHeader>
              <CardContent>
                <Table>
                  <TableHeader>
                    <TableRow>
                      <TableHead>Medication</TableHead>
                      <TableHead>Quantity</TableHead>
                      <TableHead>Updated</TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {recent_stocks.length ? recent_stocks.map((s) => (
                      <TableRow key={s.id}>
                        <TableCell>{s.medication?.name ?? '—'}</TableCell>
                        <TableCell>{typeof s.quantity === 'number' ? s.quantity : '—'}</TableCell>
                        <TableCell>{s.updated_at ? moment(s.updated_at).fromNow() : '—'}</TableCell>
                      </TableRow>
                    )) : (
                      <TableRow><TableCell colSpan={3} className="text-center text-muted-foreground">No recent stock updates.</TableCell></TableRow>
                    )}
                  </TableBody>
                </Table>
              </CardContent>
            </Card>
          </TabsContent>
        </Tabs>
      </div>
    </AppLayout>
  );
}
