import InputError from '@/components/input-error';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Checkbox } from '@/components/ui/checkbox';
import { Card, CardHeader, CardTitle, CardContent } from '@/components/ui/card';
import AppLayout from '@/layouts/app-layout';
import { BreadcrumbItem } from '@/types';
import { Head, useForm, router } from '@inertiajs/react';
import { FormEvent, useState } from 'react';
import * as Yup from 'yup';

interface CompanyForm {
  name: string;
  address: string;
  site_telephone: string;
  company_email: string;
  contact_person: string;
  province: string;
  designation: string;
  contact_number: string;
  is_private: boolean;   // ← NEW
  next?: string | null;
  [key: string]: any;
}

export default function CreateCompany() {
  const { data, setData, post, processing, errors, reset, transform } = useForm<CompanyForm>({
    name: '',
    address: '',
    site_telephone: '',
    company_email: '',
    contact_person: '',
    province: '',
    designation: '',
    contact_number: '',
    is_private: false, // ← NEW (default false)
    next: null,
  });

  // Default-checked: go to Add Patient after saving
  const [goToAddPatient, setGoToAddPatient] = useState<boolean>(true);

  const schema = Yup.object().shape({
    name: Yup.string().required('Company name is required'),
    company_email: Yup.string().email('Invalid email address').nullable(),
    is_private: Yup.boolean().nullable(), // ← NEW (validate as boolean)
  });

  // Fill everything EXCEPT name and company_email
  const fillGeneric = () => {
    const defaults: Partial<CompanyForm> = {
      address: '123 Sample Street, Harare',
      site_telephone: '0242 000 000',
      contact_person: 'HR Department',
      province: 'Harare',
      designation: 'Primary Contact',
      contact_number: '+263 77 000 0000',
    };
    (Object.keys(defaults) as (keyof CompanyForm)[]).forEach((k) => {
      setData(k, defaults[k] as string);
    });
  };

  // Make admin@{typedname}.co.zw from Company Name
  const useSystemEmail = () => {
    const base = (data.name || '').trim().toLowerCase();
    if (!base) return;
    const cleaned = base
      .normalize('NFKD')
      .replace(/[\u0300-\u036f]/g, '')
      .replace(/[^a-z0-9]+/g, '');
    if (!cleaned) return;
    setData('company_email', `admin@${cleaned}.co.zw`);
  };

  const handleSubmit = async (e: FormEvent) => {
    e.preventDefault();
    try {
      await schema.validate(data, { abortEarly: false });

      // Include "next" in the payload (server may optionally use it)
      transform((payload) => ({
        ...payload,
        next: goToAddPatient ? 'patients.create' : null,
        is_private: Boolean(payload.is_private), // ← ensure boolean
      }));

      post(route('companies.store'), {
        preserveScroll: true,
        onSuccess: () => {
          const target = goToAddPatient ? route('patients.create') : route('companies.index');
          reset();
          router.get(target, {}, { replace: true, preserveScroll: true });
        },
      });
    } catch {
      // Client-side validation errors are optional; server errors populate <InputError />
    }
  };

  const breadcrumbs: BreadcrumbItem[] = [
    { title: 'Companies', href: route('companies.index') },
    { title: 'Create Company', href: route('companies.create') },
  ];

  return (
    <AppLayout breadcrumbs={breadcrumbs}>
      <Head title="Create Company" />

      <div className="w-full space-y-8 px-4 py-10 sm:px-6">
        <div className="flex flex-col gap-3 sm:flex-row sm:items-start sm:justify-between">
          <h1 className="text-foreground text-2xl font-bold">Create Company</h1>

          <div className="flex flex-col items-start gap-2 sm:flex-row sm:items-center">
            <div className="flex items-center gap-2">
              <Checkbox
                id="redirectAfter"
                checked={goToAddPatient}
                onCheckedChange={(v) => setGoToAddPatient(Boolean(v))}
                style={{ cursor: 'pointer' }}
              />
              <Label htmlFor="redirectAfter" className="text-sm">
                Go to <span className="font-semibold">Add Patient</span> after saving
              </Label>
            </div>

            <Button type="button" variant="outline" onClick={fillGeneric} style={{ cursor: 'pointer' }}>
              Quick Fill (generic)
            </Button>
          </div>
        </div>

        <form onSubmit={handleSubmit} className="space-y-8">
          <Card className="shadow-sm">
            <CardHeader>
              <CardTitle>Company Details</CardTitle>
            </CardHeader>
            <CardContent>
              <div className="grid grid-cols-1 gap-6 sm:grid-cols-2 md:grid-cols-3">
                {/* Name */}
                <div className="grid gap-2">
                  <Label htmlFor="name">
                    Name <span className="text-red-500">*</span>
                  </Label>
                  <Input
                    id="name"
                    name="name"
                    value={data.name}
                    onChange={(e) => setData('name', e.target.value)}
                  />
                  <InputError message={errors.name} />
                </div>

                {/* Address */}
                <div className="grid gap-2">
                  <Label htmlFor="address">Address</Label>
                  <Input
                    id="address"
                    name="address"
                    value={data.address}
                    onChange={(e) => setData('address', e.target.value)}
                  />
                  <InputError message={errors.address} />
                </div>

                {/* Site Telephone */}
                <div className="grid gap-2">
                  <Label htmlFor="site_telephone">Site Telephone</Label>
                  <Input
                    id="site_telephone"
                    name="site_telephone"
                    value={data.site_telephone}
                    onChange={(e) => setData('site_telephone', e.target.value)}
                  />
                  <InputError message={errors.site_telephone} />
                </div>

                {/* Company Email with Use System Email button */}
                <div className="grid gap-2">
                  <div className="flex items-center justify-between">
                    <Label htmlFor="company_email">Company Email</Label>
                    <Button type="button" variant="secondary" size="sm" onClick={useSystemEmail} style={{ cursor: 'pointer' }}>
                      Use system email
                    </Button>
                  </div>
                  <Input
                    id="company_email"
                    name="company_email"
                    type="email"
                    value={data.company_email}
                    onChange={(e) => setData('company_email', e.target.value)}
                    placeholder="e.g. admin@yourcompany.co.zw"
                  />
                  <InputError message={errors.company_email} />
                </div>

                {/* Contact Person */}
                <div className="grid gap-2">
                  <Label htmlFor="contact_person">Contact Person</Label>
                  <Input
                    id="contact_person"
                    name="contact_person"
                    value={data.contact_person}
                    onChange={(e) => setData('contact_person', e.target.value)}
                  />
                  <InputError message={errors.contact_person} />
                </div>

                {/* Province */}
                <div className="grid gap-2">
                  <Label htmlFor="province">Province</Label>
                  <Input
                    id="province"
                    name="province"
                    value={data.province}
                    onChange={(e) => setData('province', e.target.value)}
                  />
                  <InputError message={errors.province} />
                </div>

                {/* Designation */}
                <div className="grid gap-2">
                  <Label htmlFor="designation">Designation</Label>
                  <Input
                    id="designation"
                    name="designation"
                    value={data.designation}
                    onChange={(e) => setData('designation', e.target.value)}
                  />
                  <InputError message={errors.designation} />
                </div>

                {/* Contact Number */}
                <div className="grid gap-2">
                  <Label htmlFor="contact_number">Contact Number</Label>
                  <Input
                    id="contact_number"
                    name="contact_number"
                    value={data.contact_number}
                    onChange={(e) => setData('contact_number', e.target.value)}
                  />
                  <InputError message={errors.contact_number} />
                </div>

                {/* Privacy (is_private) */}
                <div className="col-span-1 md:col-span-3">
                  <div className="mt-2 flex items-center gap-3 rounded-md border p-3">
                    <Checkbox
                      id="is_private"
                      checked={!!data.is_private}
                      onCheckedChange={(v) => setData('is_private', Boolean(v))}
                      style={{ cursor: 'pointer' }}
                    />
                    <div>
                      <Label htmlFor="is_private" className="cursor-pointer">
                        Private company (restrict access)
                      </Label>
                      <p className="text-muted-foreground mt-1 text-sm">
                        When enabled, only users explicitly granted access to this company can see consultations for its employees.
                      </p>
                      <InputError message={errors.is_private as string} />
                    </div>
                  </div>
                </div>
              </div>
            </CardContent>
          </Card>

          <div className="flex justify-end gap-3">
            <Button type="button" variant="outline" onClick={() => window.history.back()} style={{ cursor: 'pointer' }}>
              Cancel
            </Button>
            <Button type="submit" disabled={processing} style={{ cursor: 'pointer' }}>
              {processing ? 'Saving...' : 'Save Company'}
            </Button>
          </div>
        </form>
      </div>
    </AppLayout>
  );
}
