import InputError from '@/components/input-error';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Checkbox } from '@/components/ui/checkbox';
import AppLayout from '@/layouts/app-layout';
import { BreadcrumbItem } from '@/types';
import { Head, useForm } from '@inertiajs/react';
import { FormEvent } from 'react';
import * as Yup from 'yup';

interface CompanyData {
  id: number;
  name: string;
  address: string;
  site_telephone: string;
  company_email: string;
  contact_person: string;
  province: string;
  designation: string;
  contact_number: string;
  is_private: boolean; // ← NEW
  [key: string]: any;
}

export default function EditCompany({ company }: { company: CompanyData }) {
  const { data, setData, put, processing, errors } = useForm<CompanyData>({
    ...company,
    is_private: Boolean(company.is_private), // normalize to boolean
  });

  const schema = Yup.object().shape({
    name: Yup.string().required('Company name is required'),
    company_email: Yup.string().email('Invalid email').nullable(),
    is_private: Yup.boolean().nullable(), // ← NEW
  });

  const handleSubmit = async (e: FormEvent) => {
    e.preventDefault();
    try {
      await schema.validate(data, { abortEarly: false });
      put(route('companies.update', company.id), {
        preserveScroll: true,
        data: { ...data, is_private: Boolean(data.is_private) }, // ensure boolean
      });
    } catch (validationError: any) {
      // Optional: map client-side errors if you want to display them before server response
      // Server-side errors are already shown via <InputError />
    }
  };

  const breadcrumbs: BreadcrumbItem[] = [
    { title: 'Companies', href: route('companies.index') },
    { title: 'Edit Company', href: route('companies.edit', company.id) },
  ];

  return (
    <AppLayout breadcrumbs={breadcrumbs}>
      <Head title="Edit Company" />

      <div className="w-full space-y-8 px-4 py-10 sm:px-6">
        <h1 className="text-foreground text-2xl font-bold">Edit Company</h1>

        <form onSubmit={handleSubmit} className="space-y-8">
          <div className="grid grid-cols-1 gap-6 sm:grid-cols-2">
            {['name', 'address', 'site_telephone', 'company_email', 'contact_person', 'province', 'designation', 'contact_number'].map(
              (field) => (
                <div key={field} className="grid gap-2">
                  <Label htmlFor={field}>
                    {field.replace(/_/g, ' ').replace(/\b\w/g, (l) => l.toUpperCase())}
                    {field === 'name' && <span className="text-red-500">*</span>}
                  </Label>
                  <Input
                    id={field}
                    name={field}
                    type={field === 'company_email' ? 'email' : 'text'}
                    value={data[field] ?? ''}
                    onChange={(e) => setData(field, e.target.value)}
                  />
                  <InputError message={errors[field]} />
                </div>
              ),
            )}

            {/* Privacy (is_private) */}
            <div className="col-span-1 sm:col-span-2">
              <div className="mt-2 flex items-center gap-3 rounded-md border p-3">
                <Checkbox
                  id="is_private"
                  checked={!!data.is_private}
                  onCheckedChange={(v) => setData('is_private', Boolean(v))}
                  style={{ cursor: 'pointer' }}
                />
                <div>
                  <Label htmlFor="is_private" className="cursor-pointer">
                    Private company (restrict access)
                  </Label>
                  <p className="text-muted-foreground mt-1 text-sm">
                    When enabled, only users explicitly granted access to this company can see consultations for its employees.
                  </p>
                  <InputError message={errors.is_private as string} />
                </div>
              </div>
            </div>
          </div>

          <div className="flex justify-end gap-3">
            <Button type="button" variant="outline" onClick={() => window.history.back()}>
              Cancel
            </Button>
            <Button type="submit" disabled={processing}>
              {processing ? 'Updating...' : 'Update Company'}
            </Button>
          </div>
        </form>
      </div>
    </AppLayout>
  );
}
