// resources/js/pages/Companies/Index.tsx
import { API } from '@/config';
import AppLayout from '@/layouts/app-layout';
import { Head, Link, router, usePage } from '@inertiajs/react';
import React, { useEffect, useRef, useState } from 'react';
import ReactPaginate from 'react-paginate';
import { Pencil, Plus, Search, Trash2, Users } from 'lucide-react';

// shadcn/ui
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow
} from '@/components/ui/table';
import { Badge } from '@/components/ui/badge';

type Company = {
  id: number;
  name: string;
  address?: string | null;
  company_email?: string | null;
  site_telephone?: string | null;
  contact_person?: string | null;
  // NEW from backend:
  people_count?: number;
};

type PageProps = {
  companies: {
    data: Company[];
    current_page: number;
    last_page: number;
    per_page: number;
    total: number;
  };
  filters: {
    search?: string;
  };
};

// 🔹 Simple debounce hook (no lodash)
function useDebouncedValue<T>(value: T, delay = 450) {
  const [debounced, setDebounced] = useState(value);
  useEffect(() => {
    const id = setTimeout(() => setDebounced(value), delay);
    return () => clearTimeout(id);
  }, [value, delay]);
  return debounced;
}

export default function CompanyIndex() {
  const { companies, filters } = usePage<PageProps>().props as any;

  const [search, setSearch] = useState<string>(filters.search || '');
  const debouncedSearch = useDebouncedValue(search, 450);

  const firstRun = useRef(true);

  const applySearch = (page = 1, override?: string) => {
    router.get(
      `${API}/companies`,
      { page, search: override ?? search },
      { preserveState: true, replace: true, preserveScroll: true }
    );
  };

  const handleManualSearch = () => applySearch(1);
  const handleReset = () => {
    setSearch('');
    router.get(`${API}/companies`, {}, { preserveState: true, replace: true, preserveScroll: true });
  };

  const handleDelete = (id: number) => {
    if (confirm('Are you sure you want to delete this company?')) {
      router.delete(route('companies.destroy', id), { preserveScroll: true });
    }
  };

  const handlePageChange = (selected: { selected: number }) => {
    applySearch(selected.selected + 1);
  };

  // 🔹 Auto-search when typing stops
  useEffect(() => {
    if (firstRun.current) {
      firstRun.current = false;
      return;
    }
    applySearch(1, debouncedSearch);
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [debouncedSearch]);

  return (
    <AppLayout breadcrumbs={[{ title: 'Companies', href: `${API}/companies` }]}>
      <Head title="Companies" />

      <div className="bg-background mx-8 my-6 rounded-xl p-6 shadow-sm">
        <div className="mb-4 flex items-center justify-between">
          <h1 className="text-2xl font-bold">Companies</h1>

          <Button asChild>
            <Link href={route('companies.create')}>
              <Plus className="mr-2 h-4 w-4" />
              Add Company
            </Link>
          </Button>
        </div>

        {/* Filters */}
        <div className="mb-4 flex flex-wrap items-center gap-2">
          <div className="relative w-72">
            <Search className="text-muted-foreground absolute left-3 top-2.5 h-5 w-5" />
            <Input
              value={search}
              onChange={(e) => setSearch(e.target.value)} // 🔹 debounced
              onKeyDown={(e) => e.key === 'Enter' && handleManualSearch()}
              placeholder="Search companies..."
              className="pl-10"
            />
          </div>

          <Button onClick={handleManualSearch}>
            Search
          </Button>

          <Button variant="outline" onClick={handleReset}>
            Reset
          </Button>
        </div>

        {/* Table */}
        <div className="overflow-x-auto rounded-xl border">
          <Table className="min-w-full">
            <TableHeader>
              <TableRow>
                <TableHead className="uppercase">Name</TableHead>
                {/* NEW col */}
                <TableHead className="uppercase text-center">People</TableHead>
                <TableHead className="uppercase">Address</TableHead>
                <TableHead className="uppercase">Email</TableHead>
                <TableHead className="uppercase">Phone Number</TableHead>
                <TableHead className="uppercase">Contact Person</TableHead>
                <TableHead className="uppercase">Actions</TableHead>
              </TableRow>
            </TableHeader>
            <TableBody>
              {companies.data.map((company: Company) => (
                <TableRow key={company.id} className="transition hover:bg-muted/50">
                  <TableCell className="py-2 font-medium">
                    <Link
                      href={route('companies.show', company.id)}
                      className="hover:underline"
                      title="View company"
                    >
                      {company.name}
                    </Link>
                  </TableCell>

                  {/* NEW: People count pill */}
                  <TableCell className="py-2 text-center">
                    <Badge variant="secondary" className="inline-flex items-center gap-1">
                      <Users className="h-3.5 w-3.5" />
                      {company.people_count ?? 0}
                    </Badge>
                  </TableCell>

                  <TableCell className="py-2">{company.address ?? '-'}</TableCell>
                  <TableCell className="py-2">{company.company_email ?? '-'}</TableCell>
                  <TableCell className="py-2">{company.site_telephone ?? '-'}</TableCell>
                  <TableCell className="py-2">{company.contact_person ?? '-'}</TableCell>
                  <TableCell className="py-2">
                    <div className="flex items-center gap-2">
                      <Button asChild size="icon" variant="secondary" title="Edit">
                        <Link href={route('companies.edit', company.id)}>
                          <Pencil className="h-4 w-4" />
                        </Link>
                      </Button>

                      <Button
                        size="icon"
                        variant="destructive"
                        onClick={() => handleDelete(company.id)}
                        title="Delete"
                      >
                        <Trash2 className="h-4 w-4" />
                      </Button>
                    </div>
                  </TableCell>
                </TableRow>
              ))}
            </TableBody>
          </Table>
        </div>

        {/* Pagination */}
        <div className="mt-4 flex justify-center">
          <ReactPaginate
            pageCount={companies.last_page}
            forcePage={companies.current_page - 1}
            onPageChange={handlePageChange}
            marginPagesDisplayed={1}
            pageRangeDisplayed={3}
            previousLabel="← Prev"
            nextLabel="Next →"
            breakLabel="..."
            containerClassName="flex items-center gap-2 text-sm"
            pageClassName="px-3 py-1 rounded border border-border hover:bg-muted"
            activeClassName="bg-blue-600 text-white"
            previousClassName="px-3 py-1 rounded border border-border hover:bg-muted"
            nextClassName="px-3 py-1 rounded border border-border hover:bg-muted"
            breakClassName="px-2"
          />
        </div>
      </div>
    </AppLayout>
  );
}
