// resources/js/pages/Companies/Show.tsx
import AppLayout from '@/layouts/app-layout';
import { Head, Link, usePage } from '@inertiajs/react';
import {
  Users,
  UserRound,
  Baby,
  Mail,
  Phone,
  MapPin,
  Shield,
  Filter,
  Search,
  ChevronDown,
  User2,
  BadgePercent,
} from 'lucide-react';
import { useMemo, useState } from 'react';

// shadcn/ui
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Table, TableHeader, TableRow, TableHead, TableBody, TableCell } from '@/components/ui/table';
import { Input } from '@/components/ui/input';
import { Separator } from '@/components/ui/separator';
import { DropdownMenu, DropdownMenuContent, DropdownMenuItem, DropdownMenuTrigger, DropdownMenuCheckboxItem } from '@/components/ui/dropdown-menu';
import { cn } from '@/lib/utils';

type Person = {
  id: number;
  first_name: string;
  middle_name?: string | null;
  surname: string;
  gender?: string | null;
  id_number?: string | null;
  phone?: string | null;
  date_of_birth?: string | null;
  employee_number?: string | null;
  relationship?: string | null;
  parent_patient_id?: number | null;
  parent?: { id: number; first_name: string; surname: string } | null;
};

type PageProps = {
  company: {
    id: number;
    name: string;
    address?: string | null;
    company_email?: string | null;
    site_telephone?: string | null;
    contact_person?: string | null;
    province?: string | null;
    designation?: string | null;
    contact_number?: string | null;
    is_private?: boolean;
  };
  stats: {
    employees: number;
    dependents: number;
    total: number;
    gender: { male: number; female: number; other: number };
  };
  employees: Person[];
  dependents: Person[];
};

/** Utility */
const fullName = (p: Person) => [p.first_name, p.middle_name, p.surname].filter(Boolean).join(' ');

/** Hue helpers for pretty cards */
function statHue(total: number, part: number) {
  // 0..120 (red to green) scaled by ratio
  if (total <= 0) return 210; // blue-ish fallback
  const ratio = Math.min(1, Math.max(0, part / total));
  // emphasize middle range by easing
  const eased = Math.pow(ratio, 0.7);
  return Math.round(0 + (120 - 0) * eased);
}

function hueBg(h: number) {
  return `linear-gradient(135deg, hsl(${h} 85% 96%), hsl(${h} 85% 90%))`;
}
function hueFg(h: number) {
  return `hsl(${h} 55% 24%)`;
}
function pct(n: number, d: number) {
  if (!d) return '0%';
  return `${Math.round((n / d) * 100)}%`;
}

export default function CompanyShow() {
  const { company, stats, employees, dependents } = usePage<PageProps>().props;

  // view state
  const [tab, setTab] = useState<'all' | 'employees' | 'dependents'>('all');
  const [query, setQuery] = useState('');
  const [genderFilters, setGenderFilters] = useState<{ Male: boolean; Female: boolean; Other: boolean }>({
    Male: true, Female: true, Other: true,
  });
  const [relationFilters, setRelationFilters] = useState<Record<string, boolean>>({}); // dynamic for dependents

  // build working dataset
  const allPeople = useMemo<Person[]>(() => {
    if (tab === 'employees') return employees;
    if (tab === 'dependents') return dependents;
    return [...employees, ...dependents];
  }, [tab, employees, dependents]);

  // build relationships set for chips
  const allRels = useMemo(() => {
    const rels = new Set<string>();
    dependents.forEach((d) => { if (d.relationship) rels.add(d.relationship); });
    return Array.from(rels).sort();
  }, [dependents]);

  // filtered dataset
  const filtered = useMemo(() => {
    const q = query.trim().toLowerCase();
    return allPeople.filter((p) => {
      // gender filter
      const g = (p.gender ?? 'Other') as 'Male' | 'Female' | 'Other';
      if (!genderFilters[g]) return false;

      // relationship filter (only applies if in dependents or "all")
      if (tab !== 'employees' && Object.keys(relationFilters).length > 0) {
        const activeRels = Object.keys(relationFilters).filter((k) => relationFilters[k]);
        if (activeRels.length && !activeRels.includes(p.relationship ?? '')) return false;
      }

      if (!q) return true;
      const hay = [
        fullName(p),
        p.gender ?? '',
        p.id_number ?? '',
        p.phone ?? '',
        p.employee_number ?? '',
        p.relationship ?? '',
        p.parent ? `${p.parent.first_name} ${p.parent.surname}` : '',
      ]
        .join(' ')
        .toLowerCase();
      return hay.includes(q);
    });
  }, [allPeople, query, genderFilters, relationFilters, tab]);

  // Card hues
  const hueTotal = statHue(stats.total, stats.total);
  const hueEmp = statHue(stats.total, stats.employees);
  const hueDep = statHue(stats.total, stats.dependents);

  const male = stats.gender.male;
  const female = stats.gender.female;
  const other = stats.gender.other;
  const genders = [
    { key: 'Male', val: male, hue: 210 },    // blue-ish
    { key: 'Female', val: female, hue: 330 },// pink-ish
    { key: 'Other', val: other, hue: 30 },   // orange-ish
  ];

  return (
    <AppLayout breadcrumbs={[
      { title: 'Companies', href: route('companies.index') },
      { title: company.name, href: route('companies.show', company.id) }
    ]}>
      <Head title={`Company • ${company.name}`} />

      <div className="mx-8 my-6 space-y-6">
        {/* Header */}
        <div className="flex flex-wrap items-start justify-between gap-3">
          <div>
            <div className="flex items-center gap-2">
              <h1 className="text-2xl font-bold">{company.name}</h1>
              {company.is_private && (
                <Badge variant="secondary" className="gap-1">
                  <Shield className="h-3.5 w-3.5" />
                  Private
                </Badge>
              )}
            </div>
            <div className="mt-1 text-sm text-muted-foreground flex flex-wrap items-center gap-3">
              {company.address && (
                <span className="inline-flex items-center gap-1">
                  <MapPin className="h-4 w-4" /> {company.address}
                </span>
              )}
              {company.company_email && (
                <span className="inline-flex items-center gap-1">
                  <Mail className="h-4 w-4" /> {company.company_email}
                </span>
              )}
              {company.site_telephone && (
                <span className="inline-flex items-center gap-1">
                  <Phone className="h-4 w-4" /> {company.site_telephone}
                </span>
              )}
            </div>
          </div>

          <div className="flex gap-2">
            <Button asChild variant="secondary">
              <Link href={route('companies.edit', company.id)}>Edit Company</Link>
            </Button>
            <Button asChild>
              <Link href={route('patients.create')}>Create Patient</Link>
            </Button>
          </div>
        </div>

        {/* Stat cards with color-adaptive backgrounds */}
        <div className="grid grid-cols-1 gap-4 md:grid-cols-3">
          <Card style={{ background: hueBg(hueTotal), borderColor: 'transparent' }}>
            <CardHeader className="pb-1">
              <CardTitle className="text-sm font-medium" style={{ color: hueFg(hueTotal) }}>Total People</CardTitle>
            </CardHeader>
            <CardContent className="flex items-center justify-between">
              <div className="text-3xl font-bold" style={{ color: hueFg(hueTotal) }}>{stats.total}</div>
              <Users className="h-8 w-8 opacity-70" style={{ color: hueFg(hueTotal) }} />
            </CardContent>
          </Card>

          <Card style={{ background: hueBg(hueEmp), borderColor: 'transparent' }}>
            <CardHeader className="pb-1">
              <CardTitle className="text-sm font-medium" style={{ color: hueFg(hueEmp) }}>
                Employees <span className="opacity-60">({pct(stats.employees, stats.total)})</span>
              </CardTitle>
            </CardHeader>
            <CardContent className="flex items-center justify-between">
              <div className="text-3xl font-bold" style={{ color: hueFg(hueEmp) }}>{stats.employees}</div>
              <UserRound className="h-8 w-8 opacity-70" style={{ color: hueFg(hueEmp) }} />
            </CardContent>
          </Card>

          <Card style={{ background: hueBg(hueDep), borderColor: 'transparent' }}>
            <CardHeader className="pb-1">
              <CardTitle className="text-sm font-medium" style={{ color: hueFg(hueDep) }}>
                Dependents <span className="opacity-60">({pct(stats.dependents, stats.total)})</span>
              </CardTitle>
            </CardHeader>
            <CardContent className="flex items-center justify-between">
              <div className="text-3xl font-bold" style={{ color: hueFg(hueDep) }}>{stats.dependents}</div>
              <Baby className="h-8 w-8 opacity-70" style={{ color: hueFg(hueDep) }} />
            </CardContent>
          </Card>
        </div>

        {/* Gender distribution mini-cards */}
        <div className="grid grid-cols-1 gap-4 md:grid-cols-3">
          {genders.map((g) => {
            const perc = pct(g.val, stats.total);
            const fg = `hsl(${g.hue} 55% 30%)`;
            const bg = `linear-gradient(135deg, hsl(${g.hue} 90% 96%), hsl(${g.hue} 90% 90%))`;
            return (
              <Card key={g.key} style={{ background: bg, borderColor: 'transparent' }}>
                <CardHeader className="pb-1">
                  <CardTitle className="text-sm font-medium" style={{ color: fg }}>
                    {g.key} <span className="opacity-60">({perc})</span>
                  </CardTitle>
                </CardHeader>
                <CardContent className="flex items-center justify-between">
                  <div className="text-3xl font-bold" style={{ color: fg }}>{g.val}</div>
                  <User2 className="h-8 w-8 opacity-70" style={{ color: fg }} />
                </CardContent>
              </Card>
            );
          })}
        </div>

        {/* Controls */}
        <Card className="border">
          <CardContent className="pt-6">
            <div className="flex flex-wrap items-center gap-3">
              <div className="inline-flex rounded-md border bg-muted/40 p-1">
                {(['all','employees','dependents'] as const).map((k) => (
                  <button
                    key={k}
                    onClick={() => setTab(k)}
                    className={cn(
                      'px-3 py-1.5 text-sm rounded-md transition',
                      tab === k ? 'bg-background shadow text-foreground' : 'text-muted-foreground hover:text-foreground'
                    )}
                  >
                    {k === 'all' ? 'All' : k === 'employees' ? 'Employees' : 'Dependents'}
                  </button>
                ))}
              </div>

              <div className="relative flex-1 min-w-[220px] max-w-[420px]">
                <Search className="absolute left-3 top-2.5 h-4 w-4 text-muted-foreground" />
                <Input
                  value={query}
                  onChange={(e) => setQuery(e.target.value)}
                  placeholder="Search by name, ID, phone, employee no., relationship..."
                  className="pl-9"
                />
              </div>

              <DropdownMenu>
                <DropdownMenuTrigger asChild>
                  <Button variant="outline" className="gap-2">
                    <Filter className="h-4 w-4" />
                    Filters
                    <ChevronDown className="h-4 w-4 opacity-70" />
                  </Button>
                </DropdownMenuTrigger>
                <DropdownMenuContent align="end" className="w-56">
                  <div className="px-2 py-1.5 text-xs font-semibold text-muted-foreground">Gender</div>
                  {(['Male','Female','Other'] as const).map((g) => (
                    <DropdownMenuCheckboxItem
                      key={g}
                      checked={genderFilters[g]}
                      onCheckedChange={(v) => setGenderFilters((prev) => ({ ...prev, [g]: Boolean(v) }))}
                    >
                      {g}
                    </DropdownMenuCheckboxItem>
                  ))}
                  {tab !== 'employees' && (
                    <>
                      <Separator className="my-1" />
                      <div className="px-2 py-1.5 text-xs font-semibold text-muted-foreground">Relationship</div>
                      {allRels.length === 0 ? (
                        <DropdownMenuItem disabled className="opacity-70">No relationships</DropdownMenuItem>
                      ) : (
                        allRels.map((r) => (
                          <DropdownMenuCheckboxItem
                            key={r}
                            checked={!!relationFilters[r]}
                            onCheckedChange={(v) =>
                              setRelationFilters((prev) => ({ ...prev, [r]: Boolean(v) }))
                            }
                          >
                            {r}
                          </DropdownMenuCheckboxItem>
                        ))
                      )}
                    </>
                  )}
                </DropdownMenuContent>
              </DropdownMenu>
            </div>
          </CardContent>
        </Card>

        {/* Quick stats row */}
        <div className="flex flex-wrap items-center gap-2 text-xs text-muted-foreground">
          <Badge variant="outline" className="gap-1">
            <Users className="h-3.5 w-3.5" /> Showing: <b className="ml-1 text-foreground">{filtered.length}</b> / {allPeople.length}
          </Badge>
          <Badge variant="outline" className="gap-1">
            <BadgePercent className="h-3.5 w-3.5" /> {tab === 'employees' ? 'Employees' : tab === 'dependents' ? 'Dependents' : 'All'}
          </Badge>
          {tab !== 'employees' && allRels.length > 0 && (
            <div className="flex flex-wrap items-center gap-1">
              {Object.entries(relationFilters)
                .filter(([_, v]) => v)
                .map(([rel]) => (
                  <Badge key={rel} variant="secondary">{rel}</Badge>
                ))
              }
            </div>
          )}
        </div>

        {/* Data table */}
        <div className="overflow-x-auto rounded-xl border">
          <Table className="min-w-full">
            <TableHeader className="sticky top-0 z-10 bg-background/95 backdrop-blur">
              <TableRow>
                <TableHead>Name</TableHead>
                <TableHead className="hidden md:table-cell">Gender</TableHead>
                {tab !== 'dependents' && <TableHead className="hidden md:table-cell">Employee No.</TableHead>}
                <TableHead className="hidden md:table-cell">{tab === 'employees' ? 'National ID' : 'Relationship'}</TableHead>
                <TableHead className="hidden md:table-cell">Phone</TableHead>
                {tab !== 'dependents' && <TableHead className="hidden md:table-cell">DOB</TableHead>}
                {tab !== 'employees' && <TableHead className="hidden lg:table-cell">Parent</TableHead>}
                <TableHead>Actions</TableHead>
              </TableRow>
            </TableHeader>
            <TableBody>
              {filtered.length === 0 ? (
                <TableRow>
                  <TableCell colSpan={8} className="py-6 text-center text-sm text-muted-foreground">
                    No people match your filters.
                  </TableCell>
                </TableRow>
              ) : (
                filtered.map((p) => (
                  <TableRow key={p.id} className="hover:bg-muted/50">
                    <TableCell className="font-medium">{fullName(p)}</TableCell>
                    <TableCell className="hidden md:table-cell">
                      {p.gender ? (
                        <Badge
                          className={cn(
                            'rounded-full',
                            p.gender === 'Male' && 'bg-blue-100 text-blue-800',
                            p.gender === 'Female' && 'bg-pink-100 text-pink-800',
                            p.gender !== 'Male' && p.gender !== 'Female' && 'bg-amber-100 text-amber-900',
                          )}
                          variant="secondary"
                        >
                          {p.gender}
                        </Badge>
                      ) : '-'}
                    </TableCell>

                    {tab !== 'dependents' && (
                      <TableCell className="hidden md:table-cell">{p.employee_number ?? '-'}</TableCell>
                    )}

                    <TableCell className="hidden md:table-cell">
                      {tab === 'employees' ? (p.id_number ?? '-') : (p.relationship ?? '-')}
                    </TableCell>

                    <TableCell className="hidden md:table-cell">{p.phone ?? '-'}</TableCell>

                    {tab !== 'dependents' && (
                      <TableCell className="hidden md:table-cell">{p.date_of_birth ?? '-'}</TableCell>
                    )}

                    {tab !== 'employees' && (
                      <TableCell className="hidden lg:table-cell">
                        {p.parent ? `${p.parent.first_name} ${p.parent.surname}` : '-'}
                      </TableCell>
                    )}

                    <TableCell>
                      <Button asChild size="sm" variant="secondary">
                        <Link href={route('patients.show', p.id)}>View</Link>
                      </Button>
                    </TableCell>
                  </TableRow>
                ))
              )}
            </TableBody>
          </Table>
        </div>
      </div>
    </AppLayout>
  );
}
