import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import AppLayout from '@/layouts/app-layout';
import { BreadcrumbItem } from '@/types';
import { Head, Link, useForm, usePage, router } from '@inertiajs/react';
import { Eye, FlaskConical, Microscope, Pencil, Syringe, AlertTriangle } from 'lucide-react';

import moment from 'moment';
import { useEffect, useMemo, useState } from 'react';
import Select from 'react-select';
import { type SharedData } from '@/types';
import Swal from 'sweetalert2';

const nowForDatetimeLocal = () => {
  const d = new Date();
  const pad = (n: number) => String(n).padStart(2, '0');
  return `${d.getFullYear()}-${pad(d.getMonth() + 1)}-${pad(d.getDate())}T${pad(d.getHours())}:${pad(d.getMinutes())}`;
};

// ensure select values are strings (avoid 5 vs "5" mismatches)
const asStr = (v: unknown) => (v == null ? '' : String(v));

// helpers to keep "diagnosis" as a single string while using multi-select UI
const splitDiagnoses = (s: string) =>
  String(s || '')
    .split(/[,;\n]+/g)
    .map(t => t.trim())
    .filter(Boolean);

const joinDiagnoses = (arr: string[]) =>
  arr.map(s => s.trim()).filter(Boolean).join('; ');

interface ConsultationFormData {
  [key: string]: any;
  patient_id: string;
  doctor_id: string;
  clinic_id: string;
  consultation_date: string;

  // CHANGED: allow undecided state until user picks Yes or No
  injury_on_duty: boolean | null;

  systolic: string;
  diastolic: string;
  temperature: string;
  heart_rate: string;
  respiratory_rate: string;
  oxygen_saturation: string;
  weight: string;
  height: string;
  bmi: string;

  blood_sugar: string;
  blood_sugar_unit: 'mmol/L' | 'mg/dL';
  blood_sugar_context: 'random' | 'fasting' | 'postprandial' | 'ogtt' | 'other';
  blood_sugar_measured_at: string;

  urine_test_result: string;
  hiv_status: string;
  pregnancy_test_result: string;
  presenting_complaint: string;
  clinical_notes: string;
  diagnosis: string; // kept as single string (multi-select builds it)
  treatment_plan: string;
  prescription: string;

  follow_up_required: boolean;
  follow_up_date: string;
  referral_notes: string;

  sick_leave: boolean;
  number_of_days: string;
  reason_for_leave: string;
  sick_leave_notes: string;

  // NEW: restricted duty
  restricted_duty: boolean;

  refer_for_radiology: boolean;
  refer_for_lab: boolean;
  refer_to_specialist: boolean;
  specialist_name: string;
  refer_to_casualty: boolean;
  refer_out_patient: boolean;

  // NEW: hospital to which referred (mandatory if any referral ticked)
  referred_hospital: string;

  // NEW: where to go next after successful save (set via SweetAlert)
  next?: 'dispense' | 'imaging' | 'lab' | 'notes' | '';
}

type UserRole =
  | 'superadmin'
  | 'receptionist'
  | 'nurse'
  | 'doctor'
  | 'lab_technician'
  | 'radiologist'
  | 'pharmacist'
  | 'nurse_aid'
  | 'admin'
  | 'specialist'
  | 'casualty'
  | 'user'
  | 'patient';

interface PageProps {
  patients: any[];
  clinics: any[];
  doctors: any[];
  patient_id?: string | number;
  current_user: {
    id: string | number;
    clinic_id: string | number;
  };
  consultation_history?: any[];
  diagnosis_presets: Array<{ name: string }>;
}

const STORAGE_KEY = 'consultation_form_draft_v1';

export default function ConsultationCreate() {
  const { patients, clinics, doctors, patient_id, current_user, consultation_history = [], diagnosis_presets } =
    usePage<PageProps>().props as PageProps;

  const page = usePage<SharedData>();
  const { auth } = page.props;
  const userRole = auth.user?.role as UserRole | undefined;

  const today = new Date().toISOString().split('T')[0];
  const [isDarkMode, setIsDarkMode] = useState(false);

  // Toggle vitals validation (default ON)
  const [enforceVitals, setEnforceVitals] = useState<boolean>(true);

  const initialData: ConsultationFormData = useMemo(
    () => ({
      patient_id: asStr(patient_id || ''),
      doctor_id: asStr(current_user.id),        // lock to authenticated user
      clinic_id: asStr(current_user.clinic_id), // lock to authenticated clinic
      consultation_date: today,

      // IMPORTANT: no default selection
      injury_on_duty: null,

      systolic: '',
      diastolic: '',
      temperature: '',
      heart_rate: '',
      respiratory_rate: '',
      oxygen_saturation: '',
      weight: '',
      height: '',
      bmi: '',

      blood_sugar: '',
      blood_sugar_unit: 'mmol/L',
      blood_sugar_context: 'random',
      blood_sugar_measured_at: nowForDatetimeLocal(),

      urine_test_result: '',
      hiv_status: '',
      pregnancy_test_result: '',
      presenting_complaint: '',
      clinical_notes: '',
      diagnosis: '',

      treatment_plan: '',
      prescription: '',

      follow_up_required: false,
      follow_up_date: '',
      referral_notes: '',

      sick_leave: false,
      number_of_days: '',
      reason_for_leave: '',
      sick_leave_notes: '',

      // NEW
      restricted_duty: false,

      refer_for_radiology: false,
      refer_for_lab: false,
      refer_to_specialist: false,
      specialist_name: '',
      refer_to_casualty: false,
      refer_out_patient: false,

      // NEW
      referred_hospital: '',

      // NEW
      next: '',
    }),
    [current_user.id, current_user.clinic_id, patient_id, today]
  );

  const { data, setData, errors, processing, reset } = useForm<ConsultationFormData>(initialData);

  // lock doctor/clinic
  useEffect(() => {
    setData('doctor_id', asStr(current_user.id));
    setData('clinic_id', asStr(current_user.clinic_id));
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [current_user.id, current_user.clinic_id]);

  const [oxygenError, setOxygenError] = useState<string | null>(null);
  const [bloodSugarError, setBloodSugarError] = useState<string | null>(null);
  const [diagnosisError, setDiagnosisError] = useState<string | null>(null);

  // dark mode tracker (for react-select styles)
  useEffect(() => {
    const checkDarkMode = () => setIsDarkMode(document.documentElement.classList.contains('dark'));
    checkDarkMode();
    const observer = new MutationObserver(checkDarkMode);
    observer.observe(document.documentElement, { attributes: true, attributeFilter: ['class'] });
    return () => observer.disconnect();
  }, []);

  // ensure a patient is selected
  useEffect(() => {
    if (!asStr(data.patient_id)) {
      if (patient_id) setData('patient_id', asStr(patient_id));
      else if (Array.isArray(patients) && patients.length > 0) {
        setData('patient_id', asStr(patients[0].id));
      }
    }
    if (patient_id && asStr(data.patient_id) !== asStr(patient_id)) {
      setData('patient_id', asStr(patient_id));
    }
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [patient_id, patients]);

  // load draft
  useEffect(() => {
    try {
      const raw = localStorage.getItem(STORAGE_KEY);
      if (raw) {
        const draft = JSON.parse(raw) as Partial<ConsultationFormData>;
        setData((prev) => ({
          ...prev,
          ...draft,
          doctor_id: asStr(current_user.id),
          clinic_id: asStr(current_user.clinic_id),
          blood_sugar_measured_at:
            draft?.blood_sugar_measured_at && String(draft.blood_sugar_measured_at).trim() !== ''
              ? draft.blood_sugar_measured_at
              : prev.blood_sugar_measured_at,
        }));
      }
    } catch { /* ignore */ }
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, []);

  // save draft
  useEffect(() => {
    try {
      localStorage.setItem(STORAGE_KEY, JSON.stringify(data));
    } catch { /* ignore */ }
  }, [data]);

  // auto-calc BMI
  useEffect(() => {
    const w = parseFloat(String(data.weight));
    const h = parseFloat(String(data.height));
    if (Number.isFinite(w) && w > 0 && Number.isFinite(h) && h > 0) {
      const bmi = w / Math.pow(h / 100, 2);
      const formatted = bmi.toFixed(2);
      if (formatted !== data.bmi) setData('bmi', formatted);
    } else {
      if (data.bmi !== '') setData('bmi', '');
    }
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [data.weight, data.height]);

  const handleSickLeaveChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const isChecked = e.target.checked;
    setData('sick_leave', isChecked);

    // Clear the shared fields only if BOTH sick_leave and restricted_duty are OFF
    const willHaveNoCategory = !isChecked && !data.restricted_duty;
    if (willHaveNoCategory) {
      setData('number_of_days', '');
      setData('reason_for_leave', '');
      setData('sick_leave_notes', '');
    }
  };

  const handleRestrictedDutyChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const isChecked = e.target.checked;
    setData('restricted_duty', isChecked);

    // Clear the shared fields only if BOTH sick_leave and restricted_duty are OFF
    const willHaveNoCategory = !isChecked && !data.sick_leave;
    if (willHaveNoCategory) {
      setData('number_of_days', '');
      setData('reason_for_leave', '');
      setData('sick_leave_notes', '');
    }
  };

  // Multi-select Diagnoses (still stored as one string in data.diagnosis)
  const diagnosisOptions = useMemo(
    () =>
      Array.from(new Set((diagnosis_presets ?? []).map(p => p?.name).filter(Boolean))).map(name => ({
        value: name as string,
        label: name as string,
      })),
    [diagnosis_presets]
  );

  const selectedDiagnosisOptions = useMemo(
    () => splitDiagnoses(data.diagnosis).map(d => ({ value: d, label: d })),
    [data.diagnosis]
  );

  const handleDiagnosesChange = (selected: Array<{ value: string; label: string }> | null) => {
    const values = (selected ?? []).map(o => o.value);
    setData('diagnosis', joinDiagnoses(values));
    if (values.length > 0) setDiagnosisError(null);
  };

  // oxygen live validation
  const handleOxygenChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const value = e.target.value;
    setData('oxygen_saturation', value);
    if (value === '' || value === null) {
      setOxygenError(null);
      return;
    }
    const num = Number(value);
    if (Number.isNaN(num) || num < 0 || num > 100) setOxygenError('Oxygen saturation must be between 0 and 100%');
    else setOxygenError(null);
  };

  // blood sugar validation helper
  const validateBloodSugar = (val: string, unit: 'mmol/L' | 'mg/dL'): string | null => {
    if (val === '' || val === null || typeof val === 'undefined') return null;
    const num = Number(val);
    if (Number.isNaN(num) || num < 0) return 'Blood sugar must be a non-negative number';
    if (unit === 'mmol/L' && num > 40) return 'Unusually high (mmol/L). Please confirm value or unit.';
    if (unit === 'mg/dL' && num > 700) return 'Unusually high (mg/dL). Please confirm value or unit.';
    return null;
  };

  const handleBloodSugarChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const v = e.target.value;
    setData('blood_sugar', v);
    setBloodSugarError(validateBloodSugar(v, data.blood_sugar_unit));
  };

  const handleBloodSugarUnitChange = (e: React.ChangeEvent<HTMLSelectElement>) => {
    const newUnit = e.target.value as 'mmol/L' | 'mg/dL';
    setData('blood_sugar_unit', newUnit);
    setBloodSugarError(validateBloodSugar(data.blood_sugar, newUnit));
  };

  const handleBloodSugarContextChange = (e: React.ChangeEvent<HTMLSelectElement>) => {
    setData('blood_sugar_context', e.target.value);
  };

  const handleBloodSugarMeasuredAtChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    setData('blood_sugar_measured_at', e.target.value);
  };

  const customStyles = {
    control: (base: any) => ({
      ...base,
      backgroundColor: isDarkMode ? '#000' : '#fff',
      borderColor: isDarkMode ? '#374151' : '#d1d5db',
      color: isDarkMode ? '#fff' : '#000',
      '&:hover': { borderColor: isDarkMode ? '#4b5563' : '#9ca3af' },
    }),
    menu: (base: any) => ({ ...base, backgroundColor: isDarkMode ? '#000' : '#fff', color: isDarkMode ? '#fff' : '#000' }),
    option: (base: any, { isFocused }: any) => ({
      ...base,
      backgroundColor: isFocused ? (isDarkMode ? '#1e40af' : '#3b82f6') : 'transparent',
      color: isFocused ? '#fff' : isDarkMode ? '#fff' : '#000',
      ':active': { backgroundColor: isDarkMode ? '#1e40af' : '#3b82f6' },
    }),
    multiValue: (base: any) => ({ ...base, backgroundColor: isDarkMode ? '#111827' : '#e5e7eb' }),
    multiValueLabel: (base: any) => ({ ...base, color: isDarkMode ? '#e5e7eb' : '#111827' }),
    multiValueRemove: (base: any) => ({
      ...base,
      ':hover': { backgroundColor: isDarkMode ? '#1f2937' : '#d1d5db', color: isDarkMode ? '#fff' : '#111827' },
    }),
    singleValue: (base: any) => ({ ...base, color: isDarkMode ? '#fff' : '#000' }),
    input: (base: any) => ({ ...base, color: isDarkMode ? '#fff' : '#000' }),
    placeholder: (base: any) => ({ ...base, color: isDarkMode ? '#9ca3af' : '#6b7280' }),
  };

  const anyReferralSelected =
    data.refer_for_radiology ||
    data.refer_for_lab ||
    data.refer_to_specialist ||
    data.refer_to_casualty ||
    data.refer_out_patient;

  // Show leave details if either Sick Leave OR Restricted Duty is on
  const requireLeaveDetails = data.sick_leave || data.restricted_duty;

  const handleResetDraft = async () => {
    const res = await Swal.fire({
      title: 'Clear form & draft?',
      text: 'This will clear the form and remove the saved draft from this browser.',
      icon: 'warning',
      showCancelButton: true,
      confirmButtonText: 'Clear form',
      cancelButtonText: 'Cancel',
      confirmButtonColor: '#dc2626',
    });
    if (res.isConfirmed) {
      reset();
      setOxygenError(null);
      setBloodSugarError(null);
      setDiagnosisError(null);
      localStorage.removeItem(STORAGE_KEY);
      // re-lock
      setData('doctor_id', asStr(current_user.id));
      setData('clinic_id', asStr(current_user.clinic_id));
      // keep IOD undecided after reset
      setData('injury_on_duty', null);
      await Swal.fire({ icon: 'success', title: 'Cleared', timer: 1200, showConfirmButton: false });
    }
  };

  // Handlers for the two IOD checkboxes (mutually exclusive, no default)
  const toggleIODYes = (checked: boolean) => {
    setData('injury_on_duty', checked ? true : (data.injury_on_duty === true ? null : data.injury_on_duty));
  };
  const toggleIODNo = (checked: boolean) => {
    setData('injury_on_duty', checked ? false : (data.injury_on_duty === false ? null : data.injury_on_duty));
  };

  // 🔑 Submit with SweetAlert destination chooser
  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();

    // Require IOD selection
    if (data.injury_on_duty === null) {
      await Swal.fire({
        icon: 'error',
        title: 'Injury on Duty (IOD) selection required',
        html: `
          <div style="text-align:left">
            <p>Please confirm whether this case is an <b>Injury on Duty</b>.</p>
            <ol style="margin-left:1rem;margin-top:.5rem;">
              <li>Tick <b>Yes</b> if the condition is work-related (IOD).</li>
              <li>Tick <b>No</b> if it is not work-related.</li>
            </ol>
            <p style="margin-top:.5rem;"><b>You must choose one option before saving.</b></p>
          </div>
        `,
        confirmButtonText: 'OK, I’ll choose one',
      });
      return;
    }

    // patient required
    if (!asStr(data.patient_id)) {
      await Swal.fire({ icon: 'error', title: 'Select a patient', text: 'Please select a patient before saving the consultation.' });
      return;
    }

    // diagnosis required
    if (String(data.diagnosis || '').trim() === '') {
      setDiagnosisError('Diagnosis is required.');
      await Swal.fire({ icon: 'error', title: 'Diagnosis required', text: 'Please add at least one diagnosis.' });
      return;
    } else {
      setDiagnosisError(null);
    }

    // Enforced vitals validation
    if (enforceVitals) {
      const errs: string[] = [];
      const systolic = Number(data.systolic);
      const diastolic = Number(data.diastolic);
      const height = Number(data.height);
      const weight = Number(data.weight);

      if (String(data.systolic).trim() === '') errs.push('Systolic (mmHg)');
      if (String(data.diastolic).trim() === '') errs.push('Diastolic (mmHg)');
      if (String(data.height).trim() === '') errs.push('Height (cm)');
      if (String(data.weight).trim() === '') errs.push('Weight (kg)');

      if (errs.length) {
        await Swal.fire({
          icon: 'error',
          title: 'Vitals required',
          html: `<div style="text-align:left">
                   <p>Please fill the following fields:</p>
                   <ul style="margin-left:1rem; list-style:disc;">
                     ${errs.map((m) => `<li><b>${m}</b></li>`).join('')}
                   </ul>
                 </div>`,
        });
        return;
      }

      const bad: string[] = [];
      if (!Number.isFinite(systolic) || systolic < 70 || systolic > 250) bad.push('Systolic must be 70–250 mmHg');
      if (!Number.isFinite(diastolic) || diastolic < 40 || diastolic > 150) bad.push('Diastolic must be 40–150 mmHg');
      if (!Number.isFinite(height) || height < 30 || height > 300) bad.push('Height must be 30–300 cm');
      if (!Number.isFinite(weight) || weight <= 0 || weight > 500) bad.push('Weight must be 0–500 kg');

      if (bad.length) {
        await Swal.fire({
          icon: 'error',
          title: 'Check vitals',
          html: `<ul style="text-align:left; margin-left:1rem; list-style:disc;">${bad.map(b => `<li>${b}</li>`).join('')}</ul>`,
        });
        return;
      }
    }

    // oxygen sanity
    const oxygen = data.oxygen_saturation === '' ? null : Number(data.oxygen_saturation);
    if (oxygen !== null && (Number.isNaN(oxygen) || oxygen < 0 || oxygen > 100)) {
      await Swal.fire({ icon: 'error', title: 'Invalid Input', text: 'Oxygen saturation must be between 0 and 100 percent.' });
      return;
    }

    // blood sugar sanity
    const bsErr = validateBloodSugar(data.blood_sugar, data.blood_sugar_unit);
    if (bsErr) {
      setBloodSugarError(bsErr);
      await Swal.fire({ icon: 'error', title: 'Invalid Blood Sugar', text: bsErr });
      return;
    }

    // 🔒 Leave/Restricted Duty strict validation
    if (requireLeaveDetails) {
      const slErrs: string[] = [];
      if (String(data.number_of_days).trim() === '') slErrs.push('Number of Days');
      if (String(data.reason_for_leave).trim() === '') slErrs.push('Reason for Leave');
      if (String(data.sick_leave_notes).trim() === '') slErrs.push('Notes');

      if (slErrs.length) {
        await Swal.fire({
          icon: 'error',
          title: 'Leave/Restricted duty details required',
          html: `<div style="text-align:left">
                   <p>Because <b>${data.sick_leave ? 'Sick Leave' : 'Restricted Duty'}</b> is checked, the following fields are required:</p>
                   <ul style="margin-left:1rem; list-style:disc;">
                     ${slErrs.map((m) => `<li><b>${m}</b></li>`).join('')}
                   </ul>
                 </div>`,
        });
        return;
      }
    }

    // 🔒 Referral strict validation
    if (anyReferralSelected) {
      const refErrs: string[] = [];
      if (String(data.referred_hospital).trim() === '') refErrs.push('Referred Hospital');

      if (data.refer_to_specialist && String(data.specialist_name).trim() === '') {
        refErrs.push('Specialist Name');
      }

      if (refErrs.length) {
        await Swal.fire({
          icon: 'error',
          title: 'Referral details required',
          html: `<div style="text-align:left">
                   <p>Since you selected a referral option, please fill:</p>
                   <ul style="margin-left:1rem; list-style:disc;">
                     ${refErrs.map((m) => `<li><b>${m}</b></li>`).join('')}
                   </ul>
                 </div>`,
        });
        return;
      }
    }

    // recompute BMI as safety
    const weight = parseFloat(data.weight);
    const height = parseFloat(data.height);
    if (weight && height) {
      const heightInMeters = height / 100;
      const bmi = weight / (heightInMeters * heightInMeters);
      setData('bmi', bmi.toFixed(2));
    }

    // re-lock
    setData('doctor_id', asStr(current_user.id));
    setData('clinic_id', asStr(current_user.clinic_id));

    // ✅ "Where to next?" chooser
    const defaultChoice =
      data.refer_for_lab
        ? 'lab'
        : data.refer_for_radiology
          ? 'imaging'
          : 'dispense';

    const { value: nextChoice, isConfirmed } = await Swal.fire({
      title: 'Save consultation — next step',
      width: '58rem',
      html: `
        <div style="text-align:left">
          <p>Consultation will be saved. Choose where you want to go next:</p>
          <ul style="margin-left:1rem; list-style: disc;">
            <li><b>Dispense</b> — go to medication dispensing.</li>
            <li><b>Imaging referral</b> — open the imaging referral form.</li>
            <li><b>Lab referral</b> — open the laboratory referral form.</li>
          </ul>
        </div>
      `,
      input: 'radio',
      inputOptions: {
        dispense: 'Dispense medications',
        imaging: 'Create Imaging Referral',
        lab: 'Create Lab Referral',
        notes: 'Doctor Notes',
      },
      inputValue: defaultChoice,
      inputValidator: (v) => (!v ? 'Please select where to go next.' : null),
      showCancelButton: true,
      confirmButtonText: 'Save & go',
      cancelButtonText: 'Cancel',
      reverseButtons: true,
      focusConfirm: false,
    });

    if (!isConfirmed || !nextChoice) return;

    // Build payload explicitly to avoid any async state race
    const payload = { ...data, next: nextChoice as ConsultationFormData['next'] };

    router.post(route('consultations.store'), payload, {
      onSuccess: () => {
        localStorage.removeItem(STORAGE_KEY);
      },
    });
  };

  const breadcrumbs: BreadcrumbItem[] = [
    { title: 'Patients', href: route('patients.index') },
    { title: 'Consultation', href: '#' },
  ];

  return (
    <AppLayout breadcrumbs={breadcrumbs}>
      <Head title="Start Consultation" />
      <div className="w-full px-4 py-6 sm:px-6">
        {/* Header Actions */}
        <div className="mb-4 flex flex-wrap items-center justify-between gap-3">
          <h1 className="text-2xl font-bold">Start Consultation</h1>
          <div className="flex gap-2">
            <Button variant="destructive" onClick={handleResetDraft} style={{ cursor: 'pointer' }}>
              Reset Form (clear draft)
            </Button>
          </div>
        </div>

        <div className="grid grid-cols-12 gap-8">
          <div className="col-span-8 space-y-6">
            <form onSubmit={handleSubmit} className="space-y-8">
              {/* Patient Details */}
              <div className="bg-card rounded-lg border p-6 shadow-sm dark:bg-neutral-900">
                <h2 className="mb-4 text-xl font-semibold">Patient Details</h2>
                <div className="grid grid-cols-1 gap-4 sm:grid-cols-3">
                  <div>
                    <Label htmlFor="patient_id">Patient</Label>
                    <select
                      id="patient_id"
                      required
                      value={asStr(data.patient_id)}
                      onChange={(e) => setData('patient_id', e.target.value)}
                      disabled={Boolean(patient_id)}
                      className="w-full rounded-md border border-gray-300 bg-white px-3 py-2 text-sm text-gray-900 shadow-sm focus:border-blue-500 focus:ring-1 focus:ring-blue-500 focus:outline-none dark:border-gray-700 dark:bg-gray-800 dark:text-white dark:focus:border-blue-500 dark:focus:ring-blue-500"
                    >
                      {!patient_id && <option value="">Select Patient</option>}
                      {patients.map((patient) => (
                        <option key={patient.id} value={String(patient.id)}>
                          {patient.full_name ?? `${patient.first_name} ${patient.surname}`}
                        </option>
                      ))}
                    </select>
                    {Boolean(patient_id) && (
                      <p className="mt-1 text-xs text-muted-foreground">
                        Selected from URL parameter (patient_id={asStr(patient_id)}).
                      </p>
                    )}
                  </div>

                  <div>
                    <Label htmlFor="clinic_id">Clinic</Label>
                    <select
                      id="clinic_id"
                      value={asStr(data.clinic_id)}
                      onChange={(e) => setData('clinic_id', e.target.value)}
                      disabled
                      className="w-full rounded-md border border-gray-300 bg-white px-3 py-2 text-sm text-gray-900 shadow-sm focus:border-blue-500 focus:ring-1 focus:ring-blue-500 focus:outline-none dark:border-gray-700 dark:bg-gray-800 dark:text-white dark:focus:border-blue-500 dark:focus:ring-blue-500"
                    >
                      <option value="">Select Clinic</option>
                      {clinics.map((clinic) => (
                        <option key={clinic.id} value={String(clinic.id)}>
                          {clinic.name}
                        </option>
                      ))}
                    </select>
                  </div>

                  <div>
                    <Label htmlFor="doctor_id">Name Of Consulting Nurse/Dr</Label>
                    <select
                      id="doctor_id"
                      value={asStr(data.doctor_id)}
                      onChange={(e) => setData('doctor_id', e.target.value)}
                      disabled
                      className="w-full rounded-md border border-gray-300 bg-white px-3 py-2 text-sm text-gray-900 shadow-sm focus:border-blue-500 focus:ring-1 focus:ring-blue-500 focus:outline-none dark:border-gray-700 dark:bg-gray-800 dark:text-white dark:focus:border-blue-500 dark:focus:ring-blue-500"
                    >
                      <option value="">Select Doctor</option>
                      {doctors.map((doc) => (
                        <option key={doc.id} value={String(doc.id)}>
                          {doc.name}
                        </option>
                      ))}
                    </select>
                  </div>

                  <div>
                    <Label htmlFor="consultation_date">Consultation Date</Label>
                    <Input
                      id="consultation_date"
                      type="date"
                      value={data.consultation_date}
                      onChange={(e) => setData('consultation_date', e.target.value)}
                      className="w-full rounded-md border border-gray-300 bg-white px-3 py-2 text-sm text-gray-900 shadow-sm focus:border-blue-500 focus:ring-1 focus:ring-blue-500 focus:outline-none dark:border-gray-700 dark:bg-gray-800 dark:text-white dark:focus:border-blue-500 dark:focus:ring-blue-500"
                    />
                  </div>
                </div>
              </div>

              {/* Injury on Duty (two checkboxes, mutually exclusive, no default) */}
              <div className="bg-card rounded-lg border p-6 shadow-sm dark:bg-neutral-900">
                <h2 className="mb-3 text-xl font-semibold flex items-center gap-2">
                  <AlertTriangle className="h-5 w-5 text-amber-600 dark:text-amber-400" />
                  Injury on Duty (IOD)
                </h2>

                <p className="mb-3 text-xs text-muted-foreground">
                  Please confirm whether this case is work-related.
                </p>

                <div className="flex items-center gap-8">
                  <label className="inline-flex items-center gap-2">
                    <input
                      type="checkbox"
                      checked={data.injury_on_duty === true}
                      onChange={(e) => toggleIODYes(e.target.checked)}
                      className="h-5 w-5 accent-amber-600 cursor-pointer rounded"
                    />
                    <span className="font-medium">Yes (work-related)</span>
                  </label>

                  <label className="inline-flex items-center gap-2">
                    <input
                      type="checkbox"
                      checked={data.injury_on_duty === false}
                      onChange={(e) => toggleIODNo(e.target.checked)}
                      className="h-5 w-5 accent-amber-600 cursor-pointer rounded"
                    />
                    <span className="font-medium">No (not work-related)</span>
                  </label>
                </div>

                {data.injury_on_duty === null && (
                  <p className="mt-2 text-xs text-amber-700 dark:text-amber-300">
                    You must choose either <b>Yes</b> or <b>No</b> before saving.
                  </p>
                )}
              </div>

              {/* Vitals & Measurements */}
              <div className="bg-card rounded-lg border p-6 shadow-sm dark:bg-neutral-900 overflow-x-auto">
                <div className="mb-2 flex flex-col gap-2 sm:flex-row sm:items-center sm:justify-between">
                  <h2 className="text-xl font-semibold">Vitals & Measurements</h2>

                  {/* Enforce vitals toggle (default ON) */}
                  <label className="inline-flex select-none items-center gap-2 text-sm">
                    <input
                      type="checkbox"
                      className="h-4 w-4 rounded border-gray-300 text-blue-600 focus:ring-blue-500"
                      checked={enforceVitals}
                      onChange={(e) => setEnforceVitals(e.target.checked)}
                    />
                    <span className="font-medium">Enforce BP/Height/Weight validation</span>
                  </label>
                </div>
                <p className="mb-4 text-xs text-muted-foreground">
                  {enforceVitals
                    ? 'Required: enter valid Systolic & Diastolic blood pressure, Height (cm) and Weight (kg). Uncheck to skip this requirement.'
                    : 'Vitals validation is relaxed. You can submit without BP/Height/Weight.'}
                </p>

                <div className="flex flex-wrap gap-4 min-w-full">
                  {(
                    [
                      ['systolic', 'Systolic (mmHg)'],
                      ['diastolic', 'Diastolic (mmHg)'],
                      ['temperature', 'Temperature (°C)'],
                      ['heart_rate', 'Heart Rate (bpm)'],
                      ['respiratory_rate', 'Respiratory Rate (bpm)'],
                      ['weight', 'Weight (kg)'],
                      ['height', 'Height (cm)'],
                      ['bmi', 'BMI (auto-calculated)'],
                    ] as const
                  ).map(([key, label]) => (
                    <div key={key} className="flex flex-col min-w-[180px]">
                      <Label htmlFor={key}>
                        {label}
                        {enforceVitals && (key === 'systolic' || key === 'diastolic' || key === 'height' || key === 'weight') ? (
                          <span className="text-red-600"> *</span>
                        ) : null}
                      </Label>
                      <Input
                        id={key}
                        type={key === 'bmi' ? 'text' : 'number'}
                        step="any"
                        {...(key === 'weight' ? { min: 0 } : {})}
                        {...(key === 'height' ? { min: 0 } : {})}
                        value={data[key] || ''}
                        onChange={(e) => setData(key, e.target.value)}
                        readOnly={key === 'bmi'}
                        placeholder={key === 'bmi' ? 'Auto-calculated' : undefined}
                      />
                    </div>
                  ))}

                  {/* Oxygen saturation with live validation */}
                  <div className="flex flex-col min-w-[180px]">
                    <Label htmlFor="oxygen_saturation">Oxygen Saturation (%)</Label>
                    <Input
                      id="oxygen_saturation"
                      type="number"
                      step="any"
                      min={0}
                      max={100}
                      value={data.oxygen_saturation}
                      onChange={handleOxygenChange}
                      aria-invalid={!!oxygenError}
                      aria-describedby="oxygen_saturation_error"
                    />
                    {oxygenError && (
                      <p id="oxygen_saturation_error" className="mt-1 text-sm text-red-500">
                        {oxygenError}
                      </p>
                    )}
                  </div>

                  {/* Blood Sugar inline row */}
                  <div className="flex w-full items-center gap-2">
                    <Label htmlFor="blood_sugar" className="sr-only">
                      Blood Sugar
                    </Label>
                    <Input
                      id="blood_sugar"
                      type="number"
                      step="0.1"
                      min={0}
                      value={data.blood_sugar}
                      onChange={handleBloodSugarChange}
                      aria-invalid={!!bloodSugarError}
                      aria-describedby="blood_sugar_error"
                      placeholder={data.blood_sugar_unit === 'mmol/L' ? 'Blood sugar (e.g. 5.6)' : 'Blood sugar (e.g. 100)'}
                      className="w-32"
                    />

                    <Label htmlFor="blood_sugar_unit" className="sr-only">
                      Unit
                    </Label>
                    <select
                      id="blood_sugar_unit"
                      value={data.blood_sugar_unit}
                      onChange={handleBloodSugarUnitChange}
                      className="w-28 rounded-md border border-gray-300 bg-white px-2 py-2 text-sm dark:border-gray-700 dark:bg-gray-800 dark:text-white"
                      aria-label="Blood Sugar Unit"
                    >
                      <option value="mmol/L">mmol/L</option>
                      <option value="mg/dL">mg/dL</option>
                    </select>

                    <Label htmlFor="blood_sugar_context" className="sr-only">
                      Context
                    </Label>
                    <select
                      id="blood_sugar_context"
                      value={data.blood_sugar_context}
                      onChange={handleBloodSugarContextChange}
                      className="w-40 rounded-md border border-gray-300 bg-white px-2 py-2 text-sm dark:border-gray-700 dark:bg-gray-800 dark:text-white"
                      aria-label="Blood Sugar Context"
                    >
                      <option value="random">Random</option>
                      <option value="fasting">Fasting</option>
                      <option value="postprandial">Postprandial</option>
                      <option value="ogtt">OGTT</option>
                      <option value="other">Other</option>
                    </select>

                    <Label htmlFor="blood_sugar_measured_at" className="sr-only">
                      Measured At
                    </Label>
                    <Input
                      id="blood_sugar_measured_at"
                      type="datetime-local"
                      value={data.blood_sugar_measured_at}
                      onChange={handleBloodSugarMeasuredAtChange}
                      className="w-64"
                    />
                  </div>

                  <p className="mt-1 text-xs text-gray-500 dark:text-gray-400">
                    Enter the value in the correct unit; system stores canonical mmol/L.
                  </p>

                  {bloodSugarError && (
                    <p id="blood_sugar_error" className="mt-1 text-sm text-red-500">
                      {bloodSugarError}
                    </p>
                  )}
                </div>
              </div>

              {/* Clinical Information */}
              <div className="bg-card rounded-lg border p-6 shadow-sm dark:bg-neutral-900">
                <h2 className="mb-4 text-xl font-semibold">Clinical Information</h2>

                {(
                  [
                    ['presenting_complaint', 'Presenting Complaint'],
                    ['clinical_notes', 'Clinical Notes'],
                  ] as const
                ).map(([key, label]) => (
                  <div className="mb-4" key={key}>
                    <Label htmlFor={key}>{label}</Label>
                    <textarea
                      id={key}
                      className="w-full rounded-md border border-gray-300 p-2 dark:border-gray-700 dark:bg-black"
                      rows={3}
                      value={String(data[key])}
                      onChange={(e) => setData(key, e.target.value)}
                    />
                  </div>
                ))}

                {/* Multi-select Diagnoses */}
                <div className="mb-2">
                  <Label htmlFor="diagnosis_multi">Diagnoses (select one or more)</Label>
                  <Select
                    inputId="diagnosis_multi"
                    isMulti
                    options={diagnosisOptions}
                    value={selectedDiagnosisOptions}
                    onChange={handleDiagnosesChange}
                    styles={customStyles}
                    placeholder="Select diagnosis…"
                    className="w-full"
                  />
                  {diagnosisError && <p className="mt-1 text-sm text-red-500">{diagnosisError}</p>}
                </div>

                {/* Canonical diagnosis text */}
                <div className="mb-4">
                  <Label htmlFor="diagnosis">Diagnosis (combined)</Label>
                  <textarea
                    id="diagnosis"
                    className="w-full rounded-md border border-gray-300 p-2 dark:border-gray-700 dark:bg-black"
                    rows={3}
                    value={String(data.diagnosis)}
                    onChange={(e) => setData('diagnosis', e.target.value)}
                    placeholder="You can also type or edit diagnoses here (e.g. Hypertension; Diabetes Mellitus)"
                  />
                </div>

                {([['treatment_plan', 'Treatment Plan']] as const).map(([key, label]) => (
                  <div className="mb-4" key={key}>
                    <Label htmlFor={key}>{label}</Label>
                    <textarea
                      id={key}
                      className="w-full rounded-md border border-gray-300 p-2 dark:border-gray-700 dark:bg-black"
                      rows={3}
                      value={String(data[key])}
                      onChange={(e) => setData(key, e.target.value)}
                    />
                  </div>
                ))}
              </div>

              {/* Follow-up */}
              <div className="bg-card rounded-lg border p-6 shadow-sm dark:bg-neutral-900">
                <h2 className="mb-4 text-xl font-semibold">Follow-up</h2>
                <div className="grid grid-cols-1 gap-4 sm:grid-cols-2">
                  <div>
                    <Label htmlFor="follow_up_required">Follow-up Required</Label>
                    <input
                      type="checkbox"
                      id="follow_up_required"
                      checked={data.follow_up_required}
                      onChange={(e) => setData('follow_up_required', e.target.checked)}
                      className="ml-2"
                    />
                  </div>
                  <div>
                    <Label htmlFor="follow_up_date">Follow-up Date</Label>
                    <Input
                      id="follow_up_date"
                      type="date"
                      value={data.follow_up_date}
                      onChange={(e) => setData('follow_up_date', e.target.value)}
                    />
                  </div>
                </div>
              </div>

              {/* Sick Leave / Restricted Duty (shared detail fields) */}
              <div className="bg-card rounded-lg border p-6 shadow-sm dark:bg-neutral-900">
                <h2 className="mb-4 text-xl font-semibold">Leave / Duty</h2>
                <div className="grid grid-cols-1 gap-4 sm:grid-cols-2">
                  <div className="flex items-center gap-2">
                    <Label htmlFor="sick_leave">Sick Leave</Label>
                    <input
                      type="checkbox"
                      id="sick_leave"
                      checked={data.sick_leave}
                      onChange={handleSickLeaveChange}
                      className="ml-2"
                    />
                  </div>

                  {/* NEW: Restricted Duty checkbox */}
                  <div className="flex items-center gap-2">
                    <Label htmlFor="restricted_duty">Restricted Duty</Label>
                    <input
                      type="checkbox"
                      id="restricted_duty"
                      checked={data.restricted_duty}
                      onChange={handleRestrictedDutyChange}
                      className="ml-2"
                    />
                  </div>

                  {/* Show the three shared fields if EITHER is selected */}
                  {requireLeaveDetails && (
                    <>
                      <div>
                        <Label htmlFor="number_of_days">Number of Days</Label>
                        <Input
                          id="number_of_days"
                          type="number"
                          value={data.number_of_days}
                          onChange={(e) => setData('number_of_days', e.target.value)}
                        />
                      </div>

                      <div className="sm:col-span-2">
                        <Label htmlFor="reason_for_leave">
                          {data.sick_leave ? 'Reason for Leave' : 'Reason / Instruction (Restricted Duty)'}
                        </Label>
                        <textarea
                          id="reason_for_leave"
                          className="w-full rounded-md border border-gray-300 p-2 dark:border-gray-700 dark:bg-black"
                          rows={3}
                          value={data.reason_for_leave}
                          onChange={(e) => setData('reason_for_leave', e.target.value)}
                        />
                      </div>

                      <div className="sm:col-span-2">
                        <Label htmlFor="sick_leave_notes">
                          {data.sick_leave ? 'Sick Leave Notes' : 'Restricted Duty Notes'}
                        </Label>
                        <textarea
                          id="sick_leave_notes"
                          className="w-full rounded-md border border-gray-300 p-2 dark:border-gray-700 dark:bg-black"
                          rows={3}
                          value={data.sick_leave_notes}
                          onChange={(e) => setData('sick_leave_notes', e.target.value)}
                        />
                      </div>
                    </>
                  )}
                </div>
              </div>

              {/* Referrals */}
              <div className="bg-card rounded-lg border p-6 shadow-sm dark:bg-neutral-900">
                <h2 className="mb-4 text-xl font-semibold">Referrals</h2>
                <div className="grid grid-cols-1 gap-4 sm:grid-cols-2">
                  <div>
                    <Label htmlFor="refer_for_radiology">Refer for Radiology</Label>
                    <input
                      type="checkbox"
                      id="refer_for_radiology"
                      checked={data.refer_for_radiology}
                      onChange={(e) => setData('refer_for_radiology', e.target.checked)}
                      className="ml-2"
                    />
                  </div>
                  <div>
                    <Label htmlFor="refer_for_lab">Refer for Lab</Label>
                    <input
                      type="checkbox"
                      id="refer_for_lab"
                      checked={data.refer_for_lab}
                      onChange={(e) => setData('refer_for_lab', e.target.checked)}
                      className="ml-2"
                    />
                  </div>
                  <div>
                    <Label htmlFor="refer_to_specialist">Refer to Specialist</Label>
                    <input
                      type="checkbox"
                      id="refer_to_specialist"
                      checked={data.refer_to_specialist}
                      onChange={(e) => setData('refer_to_specialist', e.target.checked)}
                      className="ml-2"
                    />
                  </div>
                  {data.refer_to_specialist && (
                    <div>
                      <Label htmlFor="specialist_name">Specialist Name</Label>
                      <Input
                        id="specialist_name"
                        value={data.specialist_name}
                        onChange={(e) => setData('specialist_name', e.target.value)}
                      />
                    </div>
                  )}
                  <div>
                    <Label htmlFor="refer_to_casualty">Refer to Casualty</Label>
                    <input
                      type="checkbox"
                      id="refer_to_casualty"
                      checked={data.refer_to_casualty}
                      onChange={(e) => setData('refer_to_casualty', e.target.checked)}
                      className="ml-2"
                    />
                  </div>
                  <div>
                    <Label htmlFor="refer_out_patient">Refer Out Patient</Label>
                    <input
                      type="checkbox"
                      id="refer_out_patient"
                      checked={data.refer_out_patient}
                      onChange={(e) => setData('refer_out_patient', e.target.checked)}
                      className="ml-2"
                    />
                  </div>
                </div>

                {/* Referred Hospital + Notes when any referral selected */}
                {anyReferralSelected && (
                  <div className="mt-4 grid grid-cols-1 gap-4 sm:grid-cols-2">
                    <div className="sm:col-span-2">
                      <Label htmlFor="referred_hospital">Referred Hospital</Label>
                      <Input
                        id="referred_hospital"
                        value={data.referred_hospital}
                        onChange={(e) => setData('referred_hospital', e.target.value)}
                        placeholder="Type the hospital/clinic to which the patient is being referred"
                      />
                    </div>

                    <div className="sm:col-span-2">
                      <Label htmlFor="referral_notes">Referral Notes</Label>
                      <textarea
                        id="referral_notes"
                        className="w-full rounded-md border border-gray-300 p-2 dark:border-gray-700 dark:bg-black"
                        rows={3}
                        value={data.referral_notes}
                        onChange={(e) => setData('referral_notes', e.target.value)}
                      />
                    </div>
                  </div>
                )}
              </div>

              <div className="flex justify-end gap-4">
                <Button type="button" variant="outline" onClick={() => window.history.back()}>
                  Cancel
                </Button>
                <Button
                  type="submit"
                  disabled={processing || !!oxygenError || !!bloodSugarError}
                  style={{ cursor: 'pointer' }}
                >
                  {processing ? 'Saving...' : 'Save Consultation'}
                </Button>
              </div>
            </form>
          </div>

          {/* Right column: Consultation History */}
          <div className="col-span-4 space-y-4">
            <h2 className="text-lg font-semibold">Consultation History</h2>

            {patient_id && consultation_history?.length > 0 ? (
              consultation_history.map((consultation: any) => (
                <div key={consultation.id} className="bg-card rounded-xl border p-4 shadow-sm dark:bg-neutral-900">
                  <div className="space-y-1 text-sm text-gray-700 dark:text-gray-100">
                    <div>
                      <strong>Date:</strong>{' '}
                      {consultation.consultation_date
                        ? moment(consultation.consultation_date).format('dddd, D MMMM YYYY')
                        : 'N/A'}
                    </div>

                    <div>
                      <strong>Clinic:</strong>{' '}
                      <span className="inline-block rounded border px-2 py-0.5 text-xs">
                        {consultation.clinic || 'N/A'}
                      </span>
                    </div>

                    <div>
                      <strong>Consulted By:</strong>{' '}
                      <span className="inline-block rounded border px-2 py-0.5 text-xs">
                        {consultation.doctor || 'N/A'}
                      </span>
                    </div>

                    <div>
                      <strong>Diagnosis:</strong> {consultation.diagnosis || 'N/A'}
                    </div>
                  </div>

                  <div className="mt-4 flex flex-wrap gap-2">
                    <Button asChild size="sm" className="gap-1">
                      <Link href={route('consultations.show', consultation.id)}>
                        <Eye className="h-4 w-4" />
                        View
                        </Link>
                    </Button>

                    <Button asChild variant="secondary" size="sm" className="gap-1">
                      <Link href={route('consultations.edit', consultation.id)}>
                        <Pencil className="h-4 w-4" />
                        Edit
                      </Link>
                    </Button>

                    <Button asChild variant="outline" size="sm" className="gap-1">
                      <Link href={route('consultations.dispense', consultation.id)}>
                        <Syringe className="h-4 w-4" />
                        Dispense
                      </Link>
                    </Button>

                    {userRole && ['radiologist'].includes(userRole) && (
                      <Button asChild variant="outline" size="sm" className="gap-1">
                        <Link href={route('radiology-orders.create', { consultation: consultation.id })}>
                          <Microscope className="h-4 w-4" />
                          Radiology
                        </Link>
                      </Button>
                    )}

                    {userRole && ['specialist'].includes(userRole) && (
                      <Button asChild variant="outline" size="sm" className="gap-1">
                        <Link href={route('lab-tests.create', { consultation: consultation.id })}>
                          <FlaskConical className="h-4 w-4" />
                          Laboratory
                        </Link>
                      </Button>
                    )}
                  </div>
                </div>
              ))
            ) : (
              <p className="text-sm text-gray-500 dark:text-gray-300">No past consultations found.</p>
            )}
          </div>
        </div>
      </div>
    </AppLayout>
  );
}
