import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import AppLayout from '@/layouts/app-layout';
import { BreadcrumbItem } from '@/types';
import { PageProps as InertiaPageProps } from '@inertiajs/core';
import { Head, useForm, usePage, Link } from '@inertiajs/react';
import moment from 'moment';
import Swal from 'sweetalert2';
import { useState } from 'react';

interface DispensedItem {
  id: number;
  origin: 'clinic' | 'nurse' | null;
  medication: { name: string; dosage?: string | null; form?: string | null; unit?: string | null } | null;
  batch_number?: string | null;
  quantity: number;
  frequency?: string | null;
  route?: string | null;
  duration?: number | null;
  notes?: string | null;
  dispensed_at?: string | null;
  dispensed_by?: { id: number; name: string } | null;
}

interface ConsultationFormData {
  [key: string]: any;
  patient_id: string;
  doctor_id: string;
  clinic_id: string;
  consultation_date: string;
  injury_on_duty: boolean;

  // Vitals
  systolic: string;
  diastolic: string;
  temperature: string;
  heart_rate: string;
  respiratory_rate: string;
  oxygen_saturation: string;
  weight: string;
  height: string;
  bmi: string;

  // Labs & tests
  blood_sugar: string;
  blood_sugar_unit?: 'mmol/L' | 'mg/dL';
  blood_sugar_context?: 'random' | 'fasting' | 'postprandial' | 'ogtt' | 'other';
  blood_sugar_measured_at?: string;
  urine_test_result: string;
  hiv_status: string;
  pregnancy_test_result: string;

  // Notes
  presenting_complaint: string;
  clinical_notes: string;
  diagnosis: string;
  treatment_plan: string;
  prescription: string;

  // Follow-up & referral meta
  follow_up_required: boolean;
  follow_up_date: string;
  referral_notes: string;

  // Leave / duty
  sick_leave: boolean;
  restricted_duty: boolean;
  number_of_days: string;
  reason_for_leave: string;
  sick_leave_notes: string;

  // Referrals
  refer_for_radiology: boolean;
  refer_for_lab: boolean;
  refer_to_specialist: boolean;
  specialist_name: string;
  refer_to_casualty: boolean;
  refer_out_patient: boolean;
  referred_hospital: string;
}

interface PageProps extends InertiaPageProps {
  consultation: ConsultationFormData & {
    id: string | number;
    updated_at?: string;
    patient: { full_name: string };
    doctor: { name: string };
    clinic: { name: string };
  };
  // NEW:
  dispensations: DispensedItem[];
  patients: any[];
  clinics: any[];
  doctors: any[];
  current_user: {
    id: string | number;
    clinic_id: string | number;
  };
}

const nowForDatetimeLocal = () => {
  const d = new Date();
  const pad = (n: number) => String(n).padStart(2, '0');
  return `${d.getFullYear()}-${pad(d.getMonth() + 1)}-${pad(d.getDate())}T${pad(d.getHours())}:${pad(d.getMinutes())}`;
};

// Soft validation bounds for blood sugar inputs
const validateBloodSugar = (val: string, unit: 'mmol/L' | 'mg/dL'): string | null => {
  if (val === '' || val === null || typeof val === 'undefined') return null;
  const num = Number(val);
  if (Number.isNaN(num) || num < 0) return 'Blood sugar must be a non-negative number';
  if (unit === 'mmol/L' && num > 40) return 'Unusually high (mmol/L). Please confirm value or unit.';
  if (unit === 'mg/dL' && num > 700) return 'Unusually high (mg/dL). Please confirm value or unit.';
  return null;
};

export default function ConsultationEdit() {
  const { consultation, patients, clinics, doctors, dispensations } = usePage<PageProps>().props;

  // Defaults for the new blood sugar metadata (if missing from payload)
  const initialUnit = (consultation.blood_sugar_unit as 'mmol/L' | 'mg/dL') ?? 'mmol/L';
  const initialContext =
    (consultation.blood_sugar_context as 'random' | 'fasting' | 'postprandial' | 'ogtt' | 'other') ?? 'random';
  const initialMeasuredAt = consultation.blood_sugar_measured_at || nowForDatetimeLocal();

  const { data, setData, put, errors, processing } = useForm<ConsultationFormData>({
    ...consultation,
    restricted_duty: Boolean(consultation.restricted_duty ?? false),
    referred_hospital: consultation.referred_hospital ?? '',
    consultation_date: consultation.consultation_date ? moment(consultation.consultation_date).format('YYYY-MM-DD') : '',
    follow_up_date: consultation.follow_up_date ? moment(consultation.follow_up_date).format('YYYY-MM-DD') : '',
    blood_sugar_unit: initialUnit,
    blood_sugar_context: initialContext,
    blood_sugar_measured_at: initialMeasuredAt,
  });

  const [bloodSugarError, setBloodSugarError] = useState<string | null>(
    validateBloodSugar(String(data.blood_sugar ?? ''), data.blood_sugar_unit!)
  );

  // Handlers: blood sugar
  const handleBloodSugarChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const v = e.target.value;
    setData('blood_sugar', v);
    setBloodSugarError(validateBloodSugar(v, data.blood_sugar_unit!));
  };
  const handleBloodSugarUnitChange = (e: React.ChangeEvent<HTMLSelectElement>) => {
    const newUnit = e.target.value as 'mmol/L' | 'mg/dL';
    setData('blood_sugar_unit', newUnit);
    setBloodSugarError(validateBloodSugar(String(data.blood_sugar ?? ''), newUnit));
  };
  const handleBloodSugarContextChange = (e: React.ChangeEvent<HTMLSelectElement>) => {
    setData('blood_sugar_context', e.target.value);
  };
  const handleBloodSugarMeasuredAtChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    setData('blood_sugar_measured_at', e.target.value);
  };

  // Handlers: leave / duty
  const handleSickLeaveChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const isChecked = e.target.checked;
    setData('sick_leave', isChecked);
    if (!isChecked && !data.restricted_duty) {
      setData('number_of_days', '');
      setData('reason_for_leave', '');
      setData('sick_leave_notes', '');
    }
  };
  const handleRestrictedDutyChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const isChecked = e.target.checked;
    setData('restricted_duty', isChecked);
    if (!isChecked && !data.sick_leave) {
      setData('number_of_days', '');
      setData('reason_for_leave', '');
      setData('sick_leave_notes', '');
    }
  };

  const anyReferralSelected =
    !!data.refer_for_radiology ||
    !!data.refer_for_lab ||
    !!data.refer_to_specialist ||
    !!data.refer_to_casualty ||
    !!data.refer_out_patient;

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();

    const weight = parseFloat(data.weight);
    const height = parseFloat(data.height);
    if (weight && height) {
      const heightInMeters = height / 100;
      const bmi = weight / (heightInMeters * heightInMeters);
      setData('bmi', bmi.toFixed(2));
    }

    const bsErr = validateBloodSugar(String(data.blood_sugar ?? ''), data.blood_sugar_unit!);
    if (bsErr) {
      setBloodSugarError(bsErr);
      Swal.fire({ icon: 'error', title: 'Invalid Blood Sugar', text: bsErr });
      return;
    }

    const missing: string[] = [];

    if (data.sick_leave || data.restricted_duty) {
      if (!String(data.number_of_days || '').trim()) missing.push('Number of Days (leave/restricted duty)');
      if (!String(data.reason_for_leave || '').trim()) missing.push('Reason for Leave / Restricted Duty');
      if (!String(data.sick_leave_notes || '').trim()) missing.push('Leave / Restricted Duty Notes');
    }

    if (anyReferralSelected) {
      if (!String(data.referred_hospital || '').trim()) missing.push('Referred Hospital');
    }

    if (data.refer_to_specialist) {
      if (!String(data.specialist_name || '').trim()) missing.push('Specialist Name');
    }

    if (missing.length > 0) {
      Swal.fire({
        icon: 'error',
        title: 'Missing Required Information',
        html: `<div style="text-align:left">
                 <p>Please provide the following:</p>
                 <ul style="margin-left:1rem; list-style:disc;">
                   ${missing.map((m) => `<li><b>${m}</b></li>`).join('')}
                 </ul>
               </div>`,
      });
      return;
    }

    Swal.fire({
      title: 'Saving…',
      allowOutsideClick: false,
      didOpen: () => Swal.showLoading(),
    });

    put(route('consultations.update', consultation.id), {
      preserveScroll: true,
      onSuccess: () => {
        Swal.close();
        Swal.fire({ icon: 'success', title: 'Updated', timer: 1200, showConfirmButton: false });
      },
      onError: () => {
        Swal.close();
        Swal.fire({ icon: 'error', title: 'Error', text: 'Failed to save changes.' });
      },
    });
  };

  const breadcrumbs: BreadcrumbItem[] = [
    { title: 'Patients', href: route('patients.index') },
    { title: 'Consultations', href: '' },
    { title: 'Edit Consultation', href: '#' },
  ];

  return (
    <AppLayout breadcrumbs={breadcrumbs}>
      <Head title="Edit Consultation" />
      <div className="w-full px-4 py-6 sm:px-6">
        <div className="mb-6">
          <h1 className="text-2xl font-bold">Edit Consultation</h1>
          <p className="text-sm text-gray-500">
            Patient: {consultation.patient.full_name} | Doctor: {consultation.doctor.name} | Clinic: {consultation.clinic.name}
          </p>
        </div>

        <div className="grid grid-cols-12 gap-8">
          {/* LEFT: form */}
          <div className="col-span-12 lg:col-span-8 space-y-6">
            <form onSubmit={handleSubmit} className="space-y-8">
              {/* Patient Details */}
              <div className="bg-card rounded-lg border p-6 shadow-sm">
                <h2 className="mb-4 text-xl font-semibold">Patient Details</h2>
                <div className="grid grid-cols-1 gap-4 sm:grid-cols-3">
                  <div>
                    <Label htmlFor="patient_id">Patient</Label>
                    <select
                      id="patient_id"
                      value={data.patient_id}
                      onChange={(e) => setData('patient_id', e.target.value)}
                      className="w-full rounded-md border border-gray-300 bg-white px-3 py-2 text-sm text-gray-900 shadow-sm focus:border-blue-500 focus:ring-1 focus:ring-blue-500 focus:outline-none dark:border-gray-700 dark:bg-gray-800 dark:text-white dark:focus:border-blue-500 dark:focus:ring-blue-500"
                    >
                      <option value="">Select Patient</option>
                      {patients.map((patient) => (
                        <option key={patient.id} value={patient.id}>
                          {patient.full_name}
                        </option>
                      ))}
                    </select>
                    {errors.patient_id && <p className="text-sm text-red-500">{errors.patient_id}</p>}
                  </div>

                  <div>
                    <Label htmlFor="clinic_id">Clinic</Label>
                    <select
                      id="clinic_id"
                      value={data.clinic_id}
                      onChange={(e) => setData('clinic_id', e.target.value)}
                      className="w-full rounded-md border border-gray-300 bg-white px-3 py-2 text-sm text-gray-900 shadow-sm focus:border-blue-500 focus:ring-1 focus:ring-blue-500 focus:outline-none dark:border-gray-700 dark:bg-gray-800 dark:text-white dark:focus:border-blue-500 dark:focus:ring-blue-500"
                    >
                      <option value="">Select Clinic</option>
                      {clinics.map((clinic) => (
                        <option key={clinic.id} value={clinic.id}>
                          {clinic.name}
                        </option>
                      ))}
                    </select>
                  </div>

                  <div>
                    <Label htmlFor="doctor_id">Doctor</Label>
                    <select
                      id="doctor_id"
                      value={data.doctor_id}
                      onChange={(e) => setData('doctor_id', e.target.value)}
                      className="w-full rounded-md border border-gray-300 bg-white px-3 py-2 text-sm text-gray-900 shadow-sm focus:border-blue-500 focus:ring-1 focus:ring-blue-500 focus:outline-none dark:border-gray-700 dark:bg-gray-800 dark:text-white dark:focus:border-blue-500 dark:focus:ring-blue-500"
                    >
                      <option value="">Select Doctor</option>
                      {doctors.map((doc) => (
                        <option key={doc.id} value={doc.id}>
                          {doc.name}
                        </option>
                      ))}
                    </select>
                  </div>

                  <div>
                    <Label htmlFor="consultation_date">Consultation Date</Label>
                    <Input
                      id="consultation_date"
                      type="date"
                      value={data.consultation_date}
                      onChange={(e) => setData('consultation_date', e.target.value)}
                    />
                    {errors.consultation_date && <p className="text-sm text-red-500">{errors.consultation_date}</p>}
                  </div>

                  <div className="mt-4 flex items-center gap-2 sm:col-span-3">
                    <input
                      type="checkbox"
                      id="injury_on_duty"
                      checked={data.injury_on_duty}
                      onChange={(e) => setData('injury_on_duty', e.target.checked)}
                    />
                    <Label htmlFor="injury_on_duty">Injury on Duty</Label>
                  </div>
                </div>
              </div>

              {/* Vitals & Measurements */}
              <div className="bg-card rounded-lg border p-6 shadow-sm overflow-x-auto">
                <h2 className="mb-4 text-xl font-semibold">Vitals & Measurements</h2>
                <div className="flex flex-wrap gap-4 min-w-full">
                  {(
                    [
                      ['systolic', 'Systolic (mmHg)'],
                      ['diastolic', 'Diastolic (mmHg)'],
                      ['temperature', 'Temperature (°C)'],
                      ['heart_rate', 'Heart Rate (bpm)'],
                      ['respiratory_rate', 'Respiratory Rate (bpm)'],
                      ['oxygen_saturation', 'Oxygen Saturation (%)'],
                      ['weight', 'Weight (kg)'],
                      ['height', 'Height (cm)'],
                      ['bmi', 'BMI (auto-calculated)'],
                    ] as const
                  ).map(([key, label]) => (
                    <div key={key} className="flex flex-col min-w-[180px]">
                      <Label htmlFor={key}>{label}</Label>
                      <Input
                        id={key}
                        type="number"
                        step="any"
                        value={data[key] || ''}
                        onChange={(e) => setData(key, e.target.value)}
                        disabled={key === 'bmi'}
                      />
                      {errors[key] && <p className="text-sm text-red-500">{errors[key]}</p>}
                    </div>
                  ))}

                  {/* Blood Sugar inline row */}
                  <div className="flex w-full items-end gap-2 flex-wrap">
                    <div className="flex flex-col">
                      <Label htmlFor="blood_sugar">Blood Sugar</Label>
                      <Input
                        id="blood_sugar"
                        type="number"
                        step="0.1"
                        min={0}
                        value={data.blood_sugar ?? ''}
                        onChange={handleBloodSugarChange}
                        aria-invalid={!!bloodSugarError}
                        aria-describedby="blood_sugar_error"
                        placeholder={data.blood_sugar_unit === 'mmol/L' ? 'e.g. 5.6' : 'e.g. 100'}
                        className="w-40"
                      />
                    </div>

                    <div className="flex flex-col">
                      <Label htmlFor="blood_sugar_unit">Unit</Label>
                      <select
                        id="blood_sugar_unit"
                        value={data.blood_sugar_unit}
                        onChange={handleBloodSugarUnitChange}
                        className="w-28 rounded-md border border-gray-300 bg-white px-2 py-2 text-sm dark:border-gray-700 dark:bg-gray-800 dark:text-white"
                        aria-label="Blood Sugar Unit"
                      >
                        <option value="mmol/L">mmol/L</option>
                        <option value="mg/dL">mg/dL</option>
                      </select>
                    </div>

                    <div className="flex flex-col">
                      <Label htmlFor="blood_sugar_context">Context</Label>
                      <select
                        id="blood_sugar_context"
                        value={data.blood_sugar_context}
                        onChange={handleBloodSugarContextChange}
                        className="w-44 rounded-md border border-gray-300 bg-white px-2 py-2 text-sm dark:border-gray-700 dark:bg-gray-800 dark:text-white"
                        aria-label="Blood Sugar Context"
                      >
                        <option value="random">Random</option>
                        <option value="fasting">Fasting</option>
                        <option value="postprandial">Postprandial</option>
                        <option value="ogtt">OGTT</option>
                        <option value="other">Other</option>
                      </select>
                    </div>

                    <div className="flex flex-col">
                      <Label htmlFor="blood_sugar_measured_at">Measured At</Label>
                      <Input
                        id="blood_sugar_measured_at"
                        type="datetime-local"
                        value={data.blood_sugar_measured_at || nowForDatetimeLocal()}
                        onChange={handleBloodSugarMeasuredAtChange}
                        className="w-64"
                      />
                    </div>

                    {bloodSugarError && (
                      <p id="blood_sugar_error" className="mt-1 text-sm text-red-500 w-full">
                        {bloodSugarError}
                      </p>
                    )}
                    {errors.blood_sugar && <p className="text-sm text-red-500 w-full">{errors.blood_sugar}</p>}
                    {errors.blood_sugar_measured_at && (
                      <p className="text-sm text-red-500 w-full">{errors.blood_sugar_measured_at}</p>
                    )}
                  </div>

                  <p className="mt-1 text-xs text-gray-500 dark:text-gray-400 w-full">
                    Enter the value in the correct unit; the system stores canonical mmol/L server-side.
                  </p>
                </div>
              </div>

              {/* Clinical Information */}
              <div className="bg-card rounded-lg border p-6 shadow-sm">
                <h2 className="mb-4 text-xl font-semibold">Clinical Information</h2>
                {(
                  [
                    ['presenting_complaint', 'Presenting Complaint'],
                    ['clinical_notes', 'Clinical Notes'],
                    ['diagnosis', 'Diagnosis'],
                    ['treatment_plan', 'Treatment Plan'],
                  ] as const
                ).map(([key, label]) => (
                  <div className="mb-4" key={key}>
                    <Label htmlFor={key}>{label}</Label>
                    <textarea
                      id={key}
                      className="w-full rounded-md border border-gray-300 p-2 dark:border-gray-700 dark:bg-black"
                      rows={3}
                      value={String(data[key] ?? '')}
                      onChange={(e) => setData(key, e.target.value)}
                    />
                    {errors[key] && <p className="text-sm text-red-500">{errors[key]}</p>}
                  </div>
                ))}
              </div>

              {/* Follow-up */}
              <div className="bg-card rounded-lg border p-6 shadow-sm">
                <h2 className="mb-4 text-xl font-semibold">Follow-up</h2>
                <div className="grid grid-cols-1 gap-4 sm:grid-cols-2">
                  <div>
                    <Label htmlFor="follow_up_required">Follow-up Required</Label>
                    <input
                      type="checkbox"
                      id="follow_up_required"
                      checked={data.follow_up_required}
                      onChange={(e) => setData('follow_up_required', e.target.checked)}
                      className="ml-2"
                    />
                  </div>
                  <div>
                    <Label htmlFor="follow_up_date">Follow-up Date</Label>
                    <Input
                      id="follow_up_date"
                      type="date"
                      value={data.follow_up_date || ''}
                      onChange={(e) => setData('follow_up_date', e.target.value)}
                    />
                    {errors.follow_up_date && <p className="text-sm text-red-500">{errors.follow_up_date}</p>}
                  </div>
                  <div className="sm:col-span-2">
                    <Label htmlFor="referral_notes">Referral Notes</Label>
                    <textarea
                      id="referral_notes"
                      className="w-full rounded-md border border-gray-300 p-2 dark:border-gray-700 dark:bg-black"
                      rows={3}
                      value={data.referral_notes || ''}
                      onChange={(e) => setData('referral_notes', e.target.value)}
                    />
                    {errors.referral_notes && <p className="text-sm text-red-500">{errors.referral_notes}</p>}
                  </div>
                </div>
              </div>

              {/* Sick Leave / Restricted Duty */}
              <div className="bg-card rounded-lg border p-6 shadow-sm dark:bg-neutral-900">
                <h2 className="mb-4 text-xl font-semibold">Sick Leave / Restricted Duty</h2>
                <div className="grid grid-cols-1 gap-4 sm:grid-cols-2">
                  <div className="flex items-center gap-2">
                    <input
                      type="checkbox"
                      id="sick_leave"
                      checked={data.sick_leave}
                      onChange={handleSickLeaveChange}
                      className="ml-0"
                    />
                    <Label htmlFor="sick_leave">Sick Leave</Label>
                  </div>

                  <div className="flex items-center gap-2">
                    <input
                      type="checkbox"
                      id="restricted_duty"
                      checked={data.restricted_duty}
                      onChange={handleRestrictedDutyChange}
                      className="ml-0"
                    />
                    <Label htmlFor="restricted_duty">Restricted Duty</Label>
                  </div>

                  {(data.sick_leave || data.restricted_duty) && (
                    <>
                      <div>
                        <Label htmlFor="number_of_days">Number of Days</Label>
                        <Input
                          id="number_of_days"
                          type="number"
                          value={data.number_of_days || ''}
                          onChange={(e) => setData('number_of_days', e.target.value)}
                        />
                      </div>

                      <div className="sm:col-span-2">
                        <Label htmlFor="reason_for_leave">Reason</Label>
                        <textarea
                          id="reason_for_leave"
                          className="w-full rounded-md border border-gray-300 p-2 dark:border-gray-700 dark:bg-black"
                          rows={3}
                          value={data.reason_for_leave || ''}
                          onChange={(e) => setData('reason_for_leave', e.target.value)}
                        />
                      </div>

                      <div className="sm:col-span-2">
                        <Label htmlFor="sick_leave_notes">Notes</Label>
                        <textarea
                          id="sick_leave_notes"
                          className="w-full rounded-md border border-gray-300 p-2 dark:border-gray-700 dark:bg-black"
                          rows={3}
                          value={data.sick_leave_notes || ''}
                          onChange={(e) => setData('sick_leave_notes', e.target.value)}
                        />
                      </div>
                    </>
                  )}
                </div>
              </div>

              {/* Referrals */}
              <div className="bg-card rounded-lg border p-6 shadow-sm dark:bg-neutral-900">
                <h2 className="mb-4 text-xl font-semibold">Referrals</h2>
                <div className="grid grid-cols-1 gap-4 sm:grid-cols-2">
                  <div>
                    <Label htmlFor="refer_for_radiology">Refer for Radiology</Label>
                    <input
                      type="checkbox"
                      id="refer_for_radiology"
                      checked={data.refer_for_radiology}
                      onChange={(e) => setData('refer_for_radiology', e.target.checked)}
                      className="ml-2"
                    />
                  </div>

                  <div>
                    <Label htmlFor="refer_for_lab">Refer for Lab</Label>
                    <input
                      type="checkbox"
                      id="refer_for_lab"
                      checked={data.refer_for_lab}
                      onChange={(e) => setData('refer_for_lab', e.target.checked)}
                      className="ml-2"
                    />
                  </div>

                  <div>
                    <Label htmlFor="refer_to_specialist">Refer to Specialist</Label>
                    <input
                      type="checkbox"
                      id="refer_to_specialist"
                      checked={data.refer_to_specialist}
                      onChange={(e) => setData('refer_to_specialist', e.target.checked)}
                      className="ml-2"
                    />
                  </div>

                  {data.refer_to_specialist && (
                    <div>
                      <Label htmlFor="specialist_name">Specialist Name</Label>
                      <Input
                        id="specialist_name"
                        value={data.specialist_name || ''}
                        onChange={(e) => setData('specialist_name', e.target.value)}
                      />
                    </div>
                  )}

                  <div>
                    <Label htmlFor="refer_to_casualty">Refer to Casualty</Label>
                    <input
                      type="checkbox"
                      id="refer_to_casualty"
                      checked={data.refer_to_casualty}
                      onChange={(e) => setData('refer_to_casualty', e.target.checked)}
                      className="ml-2"
                    />
                  </div>

                  <div>
                    <Label htmlFor="refer_out_patient">Refer Out Patient</Label>
                    <input
                      type="checkbox"
                      id="refer_out_patient"
                      checked={data.refer_out_patient}
                      onChange={(e) => setData('refer_out_patient', e.target.checked)}
                      className="ml-2"
                    />
                  </div>

                  {anyReferralSelected && (
                    <div className="sm:col-span-2">
                      <Label htmlFor="referred_hospital">Referred Hospital</Label>
                      <Input
                        id="referred_hospital"
                        value={data.referred_hospital || ''}
                        onChange={(e) => setData('referred_hospital', e.target.value)}
                        placeholder="Enter the hospital/centre the patient is being referred to"
                      />
                    </div>
                  )}
                </div>
              </div>

              <div className="flex justify-end gap-4">
                <Button type="button" variant="outline" onClick={() => window.history.back()}>
                  Cancel
                </Button>
                <Button type="submit" disabled={processing || !!bloodSugarError}>
                  {processing ? 'Updating...' : 'Update Consultation'}
                </Button>
              </div>
            </form>
          </div>

          {/* RIGHT: summary + actions + DISPENSED */}
          <div className="col-span-12 lg:col-span-4 space-y-4">
            <div className="bg-card rounded-lg border p-6 shadow-sm">
              <h2 className="mb-4 text-xl font-semibold">Consultation Summary</h2>
              <div className="space-y-2">
                <div>
                  <h3 className="text-sm font-medium text-gray-500">Patient</h3>
                  <p>{consultation.patient.full_name}</p>
                </div>
                <div>
                  <h3 className="text-sm font-medium text-gray-500">Consultation Date</h3>
                  <p>{data.consultation_date ? moment(data.consultation_date).format('MMMM Do YYYY') : '—'}</p>
                </div>
                <div>
                  <h3 className="text-sm font-medium text-gray-500">Last Updated</h3>
                  <p>{consultation.updated_at ? moment(consultation.updated_at).format('MMMM Do YYYY, h:mm a') : '—'}</p>
                </div>
              </div>
            </div>

            <div className="bg-card rounded-lg border p-6 shadow-sm">
              <h2 className="mb-4 text-xl font-semibold">Actions</h2>
              <div className="space-y-2">
                <Button variant="outline" className="w-full" asChild>
                  <a href={route('consultations.show', consultation.id)}>View Consultation</a>
                </Button>
                <Button variant="outline" className="w-full" asChild>
                  <a href={route('patients.show', data.patient_id)}>View Patient</a>
                </Button>
              </div>
            </div>

            {/* NEW: Dispensed Medications */}
            <div className="bg-card rounded-lg border p-6 shadow-sm">
              <div className="mb-4 flex items-center justify-between">
                <h2 className="text-xl font-semibold">Dispensed Medications</h2>
                <Button size="sm" asChild>
                  {/* This goes to your existing dispensing page */}
                  <a href={route('consultations.dispense', consultation.id)}>Edit Dispensed</a>
                </Button>
              </div>

              {dispensations.length === 0 ? (
                <p className="text-sm text-gray-500">No medications dispensed for this consultation yet.</p>
              ) : (
                <ul className="space-y-3">
                  {dispensations.map((d) => {
                    const name = d.medication?.name ?? '—';
                    const meta = [
                      d.frequency ? `Freq: ${d.frequency}` : '',
                      d.route ? `Route: ${d.route}` : '',
                      d.duration ? `Duration: ${d.duration}d` : '',
                    ].filter(Boolean).join(' • ');
                    return (
                      <li key={d.id} className="rounded-md border p-3">
                        <div className="flex items-center justify-between">
                          <div className="font-medium">{name}</div>
                          <div className="text-sm">Qty: {d.quantity}</div>
                        </div>
                        {meta && <div className="mt-1 text-xs text-gray-600">{meta}</div>}
                        <div className="mt-1 text-xs text-gray-500">
                          {d.dispensed_by?.name ? `By ${d.dispensed_by.name}` : '—'}
                          {d.dispensed_at ? ` • ${moment(d.dispensed_at).format('MMM D, YYYY h:mm a')}` : ''}
                          {d.batch_number ? ` • Batch: ${d.batch_number}` : ''}
                        </div>
                        {d.notes && <div className="mt-1 text-xs text-gray-500 italic">“{d.notes}”</div>}

                        {/* OPTIONAL: If you want per-row correction shortcut, wire when your route exists
                        <div className="mt-2">
                          <Button variant="ghost" size="sm" asChild>
                            <a href={route('dispensation-corrections.create', d.id)}>
                              Request Correction
                            </a>
                          </Button>
                        </div>
                        */}


                      </li>
                    );
                  })}
                </ul>
              )}
            </div>
          </div>
        </div>
      </div>
    </AppLayout>
  );
}
