import React, { useEffect, useMemo, useRef, useState } from 'react';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { API } from '@/config';
import AppLayout from '@/layouts/app-layout';
import { BreadcrumbItem } from '@/types';
import { PageProps } from '@inertiajs/core';
import { Head, Link, router, usePage } from '@inertiajs/react';
import {
  Eye,
  Lock,
  Pencil,
  Plus,
  Search as SearchIcon,
  Trash,
  Settings,
  X,
  BarChart3,
} from 'lucide-react';
import moment from 'moment';
import ReactPaginate from 'react-paginate';
import Swal from 'sweetalert2';
import { useReactToPrint } from 'react-to-print';
import PrintReferralNote from './components/PrintReferralNote';
import PrintSickLeaveNote from './components/PrintSickLeaveNote';

// Recharts
import {
  ResponsiveContainer,
  BarChart,
  Bar,
  XAxis,
  YAxis,
  Tooltip,
  CartesianGrid,
  LabelList,
} from 'recharts';

/** Breadcrumbs */
const breadcrumbs: BreadcrumbItem[] = [{ title: 'Consultations', href: `${API}/consultations` }];

/** SAST (Africa/Johannesburg) = UTC+02:00, no DST */
const SAST_OFFSET_MINUTES = 120;

/** Props */
interface CompanyBucket {
  company_name: string;
  total: number;
}

interface PagePropsWithConsultations extends PageProps {
  consultations: {
    data: any[];
    total: number;
    per_page: number;
    current_page: number;
    last_page: number;
  };
  clinics: Array<{ id: number; name: string }>;
  doctors: Array<{ id: number; name: string }>;
  company_breakdown?: CompanyBucket[];
  filters: {
    search?: string;
    patient_id?: string;
    start_date?: string;
    end_date?: string;
    injury_on_duty?: boolean;
    follow_up_required?: boolean;

    refer_for_radiology?: boolean;
    refer_for_lab?: boolean;
    refer_to_specialist?: boolean;
    refer_to_casualty?: boolean;
    refer_out_patient?: boolean;

    clinic_id?: number;
    doctor_id?: number;
  };
}

type AuthUser = {
  id: number;
  name: string;
  role?: string;
  roles?: string[];
  can?: Record<string, boolean>;
};

/** Debounce hook (no lodash) */
function useDebouncedValue<T>(value: T, delay = 450) {
  const [debounced, setDebounced] = useState(value);
  useEffect(() => {
    const id = setTimeout(() => setDebounced(value), delay);
    return () => clearTimeout(id);
  }, [value, delay]);
  return debounced;
}

/** Column types */
type ColumnId =
  | 'date'
  | 'first_name'
  | 'surname'
  | 'access'
  | 'company'
  | 'clinic'
  | 'doctor'
  | 'iod'
  | 'diagnosis'
  | 'actions';

type ColumnDef = {
  id: ColumnId;
  label: string;
  defaultOn: boolean;
  /** Only include/allow this column if user can see clinical info */
  requiresClinical?: boolean;
};

export default function ConsultationIndex() {
  const { consultations, clinics, doctors, filters, company_breakdown = [] } =
    usePage<PagePropsWithConsultations>().props;

  /** Filters state */
  const [searchQuery, setSearchQuery] = useState(filters.search || '');
  const [startDate, setStartDate] = useState(filters.start_date || '');
  const [endDate, setEndDate] = useState(filters.end_date || '');
  const [injuryOnDuty, setInjuryOnDuty] = useState<boolean | null>(filters.injury_on_duty ?? null);
  const [followUpRequired, setFollowUpRequired] = useState<boolean | null>(filters.follow_up_required ?? null);

  const [referToRadiology, setReferToRadiology] = useState<boolean | null>(filters.refer_for_radiology ?? null);
  const [referToLab, setReferToLab] = useState<boolean | null>(filters.refer_for_lab ?? null);
  const [referToSpecialist, setReferToSpecialist] = useState<boolean | null>(filters.refer_to_specialist ?? null);
  const [referToCasualty, setReferToCasualty] = useState<boolean | null>(filters.refer_to_casualty ?? null);
  const [referOutPatient, setReferOutPatient] = useState<boolean | null>(filters.refer_out_patient ?? null);

  const [clinicId, setClinicId] = useState<string | number>('' + (filters.clinic_id ?? ''));
  const [doctorId, setDoctorId] = useState<string | number>('' + (filters.doctor_id ?? ''));

  /** Debounced search */
  const debouncedSearch = useDebouncedValue(searchQuery, 450);

  /** Printing */
  const contentRef = useRef<HTMLDivElement>(null);
  const reactToPrintFn = useReactToPrint({ contentRef });

  /** Auth / role logic */
  const { auth } = usePage<{ auth: { user: AuthUser } }>().props;

  const canDelete =
    auth?.user?.role === 'superadmin' ||
    auth?.user?.roles?.includes?.('superadmin') ||
    auth?.user?.can?.delete_consultations === true;

  const userRole = auth?.user?.role || '';
  const userRolesArr = Array.isArray(auth?.user?.roles) ? auth?.user?.roles : [];
  const canSeeDiagnosis =
    userRole === 'nurse' ||
    userRole === 'doctor' ||
    userRolesArr.includes('nurse') ||
    userRolesArr.includes('doctor');

  /** Column configuration */
  const allColumns: ColumnDef[] = useMemo(
    () => [
      { id: 'date', label: 'Date • Time', defaultOn: true },
      { id: 'first_name', label: 'Patient Name', defaultOn: true },
      { id: 'surname', label: 'Last_Name', defaultOn: true },
      { id: 'access', label: 'Access', defaultOn: true },
      { id: 'company', label: 'Company', defaultOn: true },
      { id: 'clinic', label: 'Clinic', defaultOn: true },
      { id: 'doctor', label: 'Consulting RGN/Doctor', defaultOn: true },
      { id: 'iod', label: 'Injury On Duty', defaultOn: true },
      { id: 'diagnosis', label: 'Diagnosis', defaultOn: canSeeDiagnosis, requiresClinical: true },
      { id: 'actions', label: 'Actions', defaultOn: true },
    ],
    [canSeeDiagnosis]
  );

  /** Load saved columns (persisted) */
  const storageKey = 'consultation_index_columns_v1';
  const [selectedColumns, setSelectedColumns] = useState<Record<ColumnId, boolean>>(() => {
    try {
      const fromStorage = localStorage.getItem(storageKey);
      if (fromStorage) {
        const parsed = JSON.parse(fromStorage) as Record<string, boolean>;
        const withDefaults: Record<ColumnId, boolean> = {} as any;
        allColumns.forEach((c) => {
          const stored = parsed[c.id];
          withDefaults[c.id] =
            typeof stored === 'boolean'
              ? stored
              : c.requiresClinical
              ? canSeeDiagnosis
                ? c.defaultOn
                : false
              : c.defaultOn;
        });
        return withDefaults;
      }
    } catch {}
    const initial: Record<ColumnId, boolean> = {} as any;
    allColumns.forEach((c) => {
      initial[c.id] = c.requiresClinical ? (canSeeDiagnosis ? c.defaultOn : false) : c.defaultOn;
    });
    return initial;
  });

  /** Persist selection */
  useEffect(() => {
    localStorage.setItem(storageKey, JSON.stringify(selectedColumns));
  }, [selectedColumns]);

  /** Visible columns */
  const visibleColumns = allColumns.filter((c) => selectedColumns[c.id]);

  /** Column picker popover toggle */
  const [showPicker, setShowPicker] = useState(false);
  const pickerRef = useRef<HTMLDivElement>(null);

  // Click outside to close
  useEffect(() => {
    function onDocClick(e: MouseEvent) {
      if (!showPicker) return;
      const target = e.target as Node;
      if (pickerRef.current && !pickerRef.current.contains(target)) {
        setShowPicker(false);
      }
    }
    document.addEventListener('mousedown', onDocClick);
    return () => document.removeEventListener('mousedown', onDocClick);
  }, [showPicker]);

  /** Filter apply/reset */
  const applyFilters = (page = 1, overrideSearch?: string) => {
    router.get(
      `${API}/consultations`,
      {
        page,
        search: overrideSearch ?? searchQuery,
        start_date: startDate,
        end_date: endDate,
        injury_on_duty: injuryOnDuty,
        follow_up_required: followUpRequired,

        refer_for_radiology: referToRadiology,
        refer_for_lab: referToLab,
        refer_to_specialist: referToSpecialist,
        refer_to_casualty: referToCasualty,
        refer_out_patient: referOutPatient,

        clinic_id: clinicId || '',
        doctor_id: doctorId || '',
      },
      { preserveState: true, replace: true }
    );
  };

  const handleFilter = () => applyFilters(1);

  const handleResetFilters = () => {
    setSearchQuery('');
    setStartDate('');
    setEndDate('');
    setInjuryOnDuty(null);
    setReferToRadiology(null);
    setReferToLab(null);
    setReferToSpecialist(null);
    setReferToCasualty(null);
    setReferOutPatient(null);
    setFollowUpRequired(null);
    setClinicId('');
    setDoctorId('');
    router.get(`${API}/consultations`, {}, { preserveState: true, replace: true });
  };

  const handleDelete = (id: number) => {
    Swal.fire({
      title: 'Are you sure?',
      text: 'This consultation will be permanently deleted.',
      icon: 'warning',
      showCancelButton: true,
      confirmButtonColor: '#e3342f',
      cancelButtonColor: '#6c757d',
      confirmButtonText: 'Yes, delete it!',
      width: '350px',
    }).then((result) => {
      if (result.isConfirmed) {
        router.delete(route('consultations.destroy', id), {
          onSuccess: () => {
            Swal.fire({
              title: 'Deleted!',
              text: 'The consultation has been deleted.',
              icon: 'success',
              timer: 1500,
              showConfirmButton: false,
            });
          },
          onError: () => {
            Swal.fire({
              title: 'Error!',
              text: 'Something went wrong.',
              icon: 'error',
            });
          },
        });
      }
    });
  };

  const handlePageChange = (selected: { selected: number }) => {
    applyFilters(selected.selected + 1);
  };

  const handlePrintSickLeave = (consultationId: number) => {
    reactToPrintFn();
    console.log('Printing Sick Leave for Consultation ID:', consultationId);
  };

  /** Auto-apply debounced search */
  const firstRun = useRef(true);
  useEffect(() => {
    if (firstRun.current) {
      firstRun.current = false;
      return;
    }
    applyFilters(1, debouncedSearch);
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [debouncedSearch]);

  /** Time formatting: STRICTLY use created_at */
  const formatCreatedAtSAST = (c: any) => {
    const created = moment(c.created_at).utcOffset(SAST_OFFSET_MINUTES);
    return `${created.format('DD MMM YYYY')} • ${created.format('HH:mm')}`;
  };

  /** Render a cell by column id */
  const renderCell = (colId: ColumnId, c: any) => {
    const hasReferral =
      Boolean(c?.refer_for_radiology) ||
      Boolean(c?.refer_for_lab) ||
      Boolean(c?.refer_to_specialist) ||
      Boolean(c?.refer_to_casualty) ||
      Boolean(c?.refer_out_patient);

    const canView = c?.can_view !== false; // default true
    const canEdit = c?.can_edit === true;

    switch (colId) {
      case 'date':
        return <span>{formatCreatedAtSAST(c)}</span>;

      case 'first_name':
        return (
          <Link href={route('consultations.show', c.id)} className="font-medium">
            {c.patient?.first_name}
          </Link>
        );

      case 'surname':
        return (
          <Link href={route('consultations.show', c.id)}>
            {c.patient?.surname}
          </Link>
        );

      case 'access':
        return canView ? (
          <Badge className="inline-flex items-center gap-1 bg-emerald-100 text-emerald-800 dark:bg-emerald-900/30 dark:text-emerald-300">
            <Eye className="h-3 w-3" />
            Accessible
          </Badge>
        ) : (
          <Badge variant="destructive" className="inline-flex items-center gap-1">
            <Lock className="h-3 w-3" />
            Private File
          </Badge>
        );

      case 'company':
        return <span>{c?.patient?.company?.name ?? '—'}</span>;

      case 'clinic':
        return <span>{c.clinic?.name}</span>;

      case 'doctor':
        return <span>{c.doctor?.name}</span>;

      case 'iod':
        return (
          <span
            className={`inline-flex items-center rounded-full px-2.5 py-0.5 text-xs font-medium ${
              c.injury_on_duty ? 'bg-green-100 text-green-800' : 'bg-red-100 text-red-800'
            }`}
          >
            {c.injury_on_duty ? 'Yes' : 'No'}
          </span>
        );

      case 'diagnosis':
        if (!canSeeDiagnosis) return null;
        if (!canView) {
          return (
            <Badge variant="destructive" className="inline-flex items-center gap-1">
              <Lock className="h-3 w-3" />
              Private Diagnosis
            </Badge>
          );
        }
        return <span>{c.diagnosis}</span>;

      case 'actions':
        return (
          <div className="flex flex-wrap items-center gap-2">
            {canView ? (
              <Link
                href={route('consultations.show', c.id)}
                className="inline-flex items-center justify-center rounded-full bg-blue-600 p-2 text-white hover:bg-blue-700"
                title="View"
              >
                <Eye className="h-4 w-4" />
              </Link>
            ) : (
              <Link
                href={route('consultations.show', c.id)}
                className="inline-flex items-center justify-center rounded-full border border-gray-300 p-2 text-gray-700 hover:bg-gray-50"
                title="You cannot access this file"
              >
                <Lock className="h-4 w-4" />
              </Link>
            )}

            {canEdit && (
              <Link
                href={route('consultations.edit', c.id)}
                className="inline-flex items-center justify-center rounded-full bg-yellow-500 p-2 text-white hover:bg-yellow-600"
                title="Edit"
              >
                <Pencil className="h-4 w-4" />
              </Link>
            )}

            {canDelete && (
              <button
                onClick={() => handleDelete(c.id)}
                className="inline-flex items-center justify-center rounded-full bg-red-600 p-2 text-white hover:bg-red-700"
                title="Delete"
                type="button"
              >
                <Trash className="h-4 w-4" />
              </button>
            )}

            {hasReferral && (
              <PrintReferralNote
                consultation={c}
                buttonLabel="Print Referral"
                buttonVariant="outline"
                buttonSize="sm"
                buttonClassName="shrink-0"
              />
            )}

            {c.sick_leave ? (
              <PrintSickLeaveNote consultation={c} />
            ) : (
              <Button
                variant="outline"
                size="sm"
                disabled
                className="shrink-0"
                title="No Sick Leave Available"
                style={{ cursor: 'not-allowed' }}
                type="button"
              >
                No Sick Leave
              </Button>
            )}
          </div>
        );

      default:
        return null;
    }
  };

  /** Column Picker Popover */
  const ColumnPickerButton = () => (
    <div className="relative" ref={pickerRef}>
      <Button
        type="button"
        variant="outline"
        className="inline-flex items-center gap-2"
        onClick={() => setShowPicker((s) => !s)}
      >
        <Settings className="h-4 w-4" />
        Columns
      </Button>

      {showPicker && (
        <div className="absolute right-0 z-20 mt-2 w-[320px] rounded-lg border bg-background p-3 shadow-lg">
          <div className="mb-2 flex items-center justify-between">
            <div className="flex items-center gap-2">
              <Settings className="h-4 w-4" />
              <span className="text-sm font-semibold">Select Columns</span>
            </div>
            <button
              type="button"
              onClick={() => setShowPicker(false)}
              className="rounded p-1 hover:bg-muted"
              title="Close"
            >
              <X className="h-4 w-4" />
            </button>
          </div>

          <div className="grid grid-cols-2 gap-2">
            {allColumns.map((col) => {
              if (col.requiresClinical && !canSeeDiagnosis) return null;
              return (
                <label key={col.id} className="flex cursor-pointer items-center gap-2 text-sm">
                  <input
                    type="checkbox"
                    checked={Boolean(selectedColumns[col.id])}
                    onChange={() =>
                      setSelectedColumns((prev) => ({
                        ...prev,
                        [col.id]: !prev[col.id],
                      }))
                    }
                    className="h-4 w-4 rounded border-gray-300 text-blue-600 focus:ring-blue-500"
                  />
                  {col.label}
                </label>
              );
            })}
          </div>

          <div className="mt-3 flex flex-wrap gap-2">
            <Button
              variant="outline"
              size="sm"
              type="button"
              onClick={() => {
                const allOn: Record<ColumnId, boolean> = {} as any;
                allColumns.forEach((c) => {
                  allOn[c.id] = c.requiresClinical ? (canSeeDiagnosis ? true : false) : true;
                });
                setSelectedColumns(allOn);
              }}
            >
              Select All
            </Button>
            <Button
              variant="outline"
              size="sm"
              type="button"
              onClick={() => {
                const none: Record<ColumnId, boolean> = {} as any;
                allColumns.forEach((c) => {
                  none[c.id] = false;
                });
                // keep Actions visible
                none.actions = true;
                setSelectedColumns(none);
              }}
            >
              Clear
            </Button>
            <Button
              size="sm"
              type="button"
              onClick={() => {
                const defaults: Record<ColumnId, boolean> = {} as any;
                allColumns.forEach((c) => {
                  defaults[c.id] = c.requiresClinical ? (canSeeDiagnosis ? c.defaultOn : false) : c.defaultOn;
                });
                setSelectedColumns(defaults);
              }}
            >
              Defaults
            </Button>
          </div>
        </div>
      )}
    </div>
  );

  /** Company Breakdown (Bar Chart) — open/close on demand */
  const [showCompanyGraph, setShowCompanyGraph] = useState(false);

  // Reduce label noise: show top 12 buckets (still pass all if <=12)
  const companyChartData = useMemo<CompanyBucket[]>(() => {
    const sorted = [...company_breakdown].sort((a, b) => b.total - a.total);
    return sorted.slice(0, 12);
  }, [company_breakdown]);

  const CompanyGraphPanel = () => {
    if (!showCompanyGraph) return null;

    return (
      <div className="rounded-xl border p-4 shadow-sm">
        <div className="mb-3 flex items-center justify-between">
          <div className="flex items-center gap-2">
            <BarChart3 className="h-5 w-5" />
            <h3 className="text-base font-semibold">Consultations by Company</h3>
          </div>
        <Button variant="outline" size="sm" onClick={() => setShowCompanyGraph(false)}>
            <X className="mr-1 h-4 w-4" />
            Close
          </Button>
        </div>

        <p className="mb-4 text-sm text-muted-foreground">
          Overview of people visiting the clinic by company. For dependents, the parent&apos;s company is used.
        </p>

        <div className="h-[380px] w-full">
          <ResponsiveContainer width="100%" height="100%">
            <BarChart data={companyChartData} margin={{ top: 16, right: 24, bottom: 8, left: 8 }}>
              <CartesianGrid vertical={false} />
              <XAxis
                dataKey="company_name"
                tick={{ fontSize: 12 }}
                interval={0}
                angle={-20}
                textAnchor="end"
                height={60}
              />
              <YAxis allowDecimals={false} />
              <Tooltip cursor={{ fillOpacity: 0.075 }} />
              <Bar dataKey="total" fill="#3b82f6" stroke="#2563eb" radius={[6, 6, 0, 0]}>
                <LabelList dataKey="total" position="top" fill="#1f2937" />
              </Bar>
            </BarChart>
          </ResponsiveContainer>
        </div>
      </div>
    );
  };

  // Enter key handler for big search
  const onSearchEnter = (e: React.KeyboardEvent<HTMLInputElement>) => {
    if (e.key === 'Enter') applyFilters(1);
  };

  return (
    <AppLayout breadcrumbs={breadcrumbs}>
      <Head title="Consultations" />

      <div className="bg-background text-foreground mx-8 my-6 flex flex-1 flex-col gap-4 rounded-xl p-6 shadow-sm">
        <div className="mb-4 flex flex-wrap items-center justify-between gap-4">
          <h1 className="text-2xl font-bold">Consultations</h1>
          <div className="flex items-center gap-2">
            <Button
              type="button"
              variant={showCompanyGraph ? 'default' : 'outline'}
              onClick={() => setShowCompanyGraph((s) => !s)}
              className="inline-flex items-center gap-2"
            >
              <BarChart3 className="h-4 w-4" />
              {showCompanyGraph ? 'Hide Company Overview' : 'Show Company Overview'}
            </Button>
            <ColumnPickerButton />
            <Link
              href={route('consultations.create')}
              className="inline-flex items-center gap-2 rounded-lg bg-blue-600 px-5 py-2 text-sm font-medium text-white hover:bg-blue-700"
              style={{ cursor: 'pointer' }}
            >
              <Plus className="h-4 w-4" />
              Add Consultation
            </Link>
          </div>
        </div>

        {/* BIG Search (same design as Patients) */}
        <div className="rounded-xl border bg-white/70 p-2 shadow-sm ring-1 ring-border dark:bg-neutral-900/60">
          <div className="relative">
            <SearchIcon className="text-muted-foreground absolute left-4 top-1/2 h-5 w-5 -translate-y-1/2" />
            <input
              value={searchQuery}
              onChange={(e) => setSearchQuery(e.target.value)}
              onKeyDown={onSearchEnter}
              placeholder="Search by patient surname (e.g Gashira)…"
              className="h-12 w-full rounded-xl bg-background px-4 pl-12 pr-[8.5rem] text-base outline-none"
            />
            {searchQuery && (
              <Button
                type="button"
                variant="ghost"
                className="absolute right-28 top-1/2 -translate-y-1/2 h-10 w-10 rounded-full"
                onClick={() => setSearchQuery('')}
                title="Clear"
              >
                <X className="h-4 w-4" />
              </Button>
            )}
            <Button
              type="button"
              className="absolute right-2 top-1/2 -translate-y-1/2 h-10 rounded-lg px-4"
              onClick={() => applyFilters(1)}
              title="Search"
            >
              <SearchIcon className="mr-2 h-4 w-4" />
              Search
            </Button>
          </div>
          <div className="mt-1 px-1 text-xs text-muted-foreground">
            Tip: Press <kbd className="rounded border px-1">Enter</kbd> to search • Hyphens/spaces in IDs are ignored.
          </div>
        </div>

        {/* Filters */}
        <div className="rounded-lg border p-4">
          <div className="grid grid-cols-1 gap-4 md:grid-cols-2 lg:grid-cols-4">
            {/* Date Range */}
            <div>
              <label className="mb-1 block text-sm font-medium">Date Range</label>
              <div className="grid grid-cols-2 gap-2">
                <input
                  type="date"
                  value={startDate}
                  onChange={(e) => setStartDate(e.target.value)}
                  className="border-border bg-background rounded-lg border p-2 text-sm"
                />
                <input
                  type="date"
                  value={endDate}
                  onChange={(e) => setEndDate(e.target.value)}
                  className="border-border bg-background rounded-lg border p-2 text-sm"
                />
              </div>
            </div>

            {/* Clinic */}
            <div>
              <label className="mb-1 block text sm font-medium">Clinic</label>
              <select
                value={clinicId}
                onChange={(e) => setClinicId(e.target.value)}
                className="border-border bg-background w-full rounded-lg border p-2 text-sm"
              >
                <option value="">All Clinics</option>
                {clinics.map((clinic) => (
                  <option key={clinic.id} value={clinic.id}>
                    {clinic.name}
                  </option>
                ))}
              </select>
            </div>

            {/* Doctor */}
            <div>
              <label className="mb-1 block text-sm font-medium">Doctor</label>
              <select
                value={doctorId}
                onChange={(e) => setDoctorId(e.target.value)}
                className="border-border bg-background w-full rounded-lg border p-2 text-sm"
              >
                <option value="">All Doctors</option>
                {doctors.map((doctor) => (
                  <option key={doctor.id} value={doctor.id}>
                    {doctor.name}
                  </option>
                ))}
              </select>
            </div>
          </div>

          {/* Binary filters */}
          <div className="mt-6 flex flex-wrap items-center justify-start gap-4">
            {[
              { id: 'injuryOnDuty', label: 'Injury on Duty', val: injuryOnDuty, setter: setInjuryOnDuty },
              { id: 'followUpRequired', label: 'Follow Up Required', val: followUpRequired, setter: setFollowUpRequired },
              { id: 'referToRadiology', label: 'Radiology Referral', val: referToRadiology, setter: setReferToRadiology },
              { id: 'referToLab', label: 'Laboratory Referral', val: referToLab, setter: setReferToLab },
              { id: 'referToSpecialist', label: 'Specialist Referral', val: referToSpecialist, setter: setReferToSpecialist },
              { id: 'referToCasualty', label: 'Casualty Referral', val: referToCasualty, setter: setReferToCasualty },
              { id: 'referOutPatient', label: 'Outpatient Referral', val: referOutPatient, setter: setReferOutPatient },
            ].map((f) => (
              <label key={f.id} className="flex items-center space-x-2">
                <input
                  type="checkbox"
                  checked={f.val === true}
                  onChange={(e) => f.setter(e.target.checked ? true : null)}
                  className="h-4 w-4 rounded border-gray-300 text-blue-600 focus:ring-blue-500"
                />
                <span className="text-sm font-medium">{f.label}</span>
              </label>
            ))}
          </div>

          <div className="mt-4 flex justify-end gap-2">
            <button
              onClick={handleResetFilters}
              className="rounded-lg bg-gray-200 px-4 py-2 text-sm font-medium text-gray-700 hover:bg-gray-300"
              style={{ cursor: 'pointer' }}
              type="button"
            >
              Reset Filters
            </button>
            <button
              onClick={handleFilter}
              className="rounded-lg bg-blue-600 px-4 py-2 text-sm font-medium text-white hover:bg-blue-700"
              style={{ cursor: 'pointer' }}
              type="button"
            >
              Apply Filters
            </button>
          </div>
        </div>

        {/* Results */}
        <div className="overflow-x-auto rounded-xl border">
          <table className="min-w-full divide-y overflow-hidden rounded-xl bg-white text-sm dark:bg-neutral-900">
            <thead className="bg-muted">
              <tr>
                {visibleColumns.map((c) => (
                  <th
                    key={c.id}
                    className="text-muted-foreground px-4 py-3 text-left text-xs font-semibold uppercase tracking-wider"
                  >
                    {c.label}
                  </th>
                ))}
              </tr>
            </thead>

            <tbody className="divide-y">
              {consultations.data.map((c) => (
                <tr key={c.id} className="transition hover:bg-muted/50">
                  {visibleColumns.map((col) => (
                    <td key={col.id} className="px-4 py-3">
                      {renderCell(col.id, c)}
                    </td>
                  ))}
                </tr>
              ))}
            </tbody>
          </table>
        </div>

        {/* Pagination */}
        <div className="mt-6 flex justify-center">
          <ReactPaginate
            pageCount={consultations.last_page}
            forcePage={consultations.current_page - 1}
            onPageChange={handlePageChange}
            marginPagesDisplayed={1}
            pageRangeDisplayed={3}
            previousLabel="← Prev"
            nextLabel="Next →"
            breakLabel="..."
            containerClassName="flex items-center gap-2 text-sm"
            pageClassName="px-3 py-1 rounded border border-border hover:bg-muted"
            activeClassName="bg-blue-600 text-white"
            previousClassName="px-3 py-1 rounded border border-border hover:bg-muted"
            nextClassName="px-3 py-1 rounded border border-border hover:bg-muted"
            breakClassName="px-2"
          />
        </div>
      </div>
    </AppLayout>
  );
}
