import moment from 'moment';
import { forwardRef } from 'react';

const logoUrl = 'https://providencehumancapital.com/headoffice/assets/images/PHC_Logo.png';

interface ConsultationPrintProps {
    consultation: any;
}

// 👇 Explicitly define ref as HTMLDivElement
const ConsultationPrint = forwardRef<HTMLDivElement, ConsultationPrintProps>(({ consultation }, ref) => {
    return (
        <div ref={ref} className="p-8 font-sans text-sm text-black">
            <div className="mb-6 flex items-center">
                <img src={logoUrl} alt="PHC Logo" className="mr-4 h-16 w-auto" />
                <div>
                    <h1 className="text-xl font-bold">Consultation Summary</h1>
                    <p>{moment(consultation.consultation_date).format('MMMM Do YYYY, h:mm a')}</p>
                </div>
            </div>

            <hr className="my-4" />

            <section className="mb-6">
                <h2 className="text-lg font-semibold">Patient Information</h2>
                <p>
                    <strong>Patient:</strong> {consultation.patient.full_name}
                </p>
                <p>
                    <strong>Clinic:</strong> {consultation.clinic.name}
                </p>
                <p>
                    <strong>Doctor:</strong> {consultation.doctor.name}
                </p>
                <p>
                    <strong>Injury on Duty:</strong> {consultation.injury_on_duty ? 'Yes' : 'No'}
                </p>
            </section>

            <section className="mb-6">
                <h2 className="text-lg font-semibold">Vitals & Measurements</h2>
                <p>
                    <strong>BP:</strong> {consultation.systolic}/{consultation.diastolic} mmHg
                </p>
                <p>
                    <strong>Temp:</strong> {consultation.temperature} °C
                </p>
                <p>
                    <strong>Heart Rate:</strong> {consultation.heart_rate} bpm
                </p>
                <p>
                    <strong>Respiratory Rate:</strong> {consultation.respiratory_rate} bpm
                </p>
                <p>
                    <strong>O₂ Saturation:</strong> {consultation.oxygen_saturation}%
                </p>
                <p>
                    <strong>Weight:</strong> {consultation.weight} kg
                </p>
                <p>
                    <strong>Height:</strong> {consultation.height} cm
                </p>
                <p>
                    <strong>BMI:</strong> {consultation.bmi}
                </p>
            </section>

            <section className="mb-6">
                <h2 className="text-lg font-semibold">Tests</h2>
                <p>
                    <strong>Blood Sugar:</strong> {consultation.blood_sugar}
                </p>
                <p>
                    <strong>Urine Test:</strong> {consultation.urine_test_result}
                </p>
                <p>
                    <strong>HIV Status:</strong> {consultation.hiv_status}
                </p>
                <p>
                    <strong>Pregnancy Test:</strong> {consultation.pregnancy_test_result}
                </p>
            </section>

            <section className="mb-6">
                <h2 className="text-lg font-semibold">Clinical</h2>
                <p>
                    <strong>Complaint:</strong> {consultation.presenting_complaint}
                </p>
                <p>
                    <strong>Notes:</strong> {consultation.clinical_notes}
                </p>
                <p>
                    <strong>Diagnosis:</strong> {consultation.diagnosis}
                </p>
                <p>
                    <strong>Treatment Plan:</strong> {consultation.treatment_plan}
                </p>
                <p>
                    <strong>Prescription:</strong> {consultation.prescription}
                </p>
            </section>

            {consultation.follow_up_required || consultation.follow_up_date || consultation.referral_notes ? (
                <section className="mb-6">
                    <h2 className="text-lg font-semibold">Follow-up</h2>
                    {consultation.follow_up_required && (
                        <p>
                            <strong>Required:</strong> Yes
                        </p>
                    )}
                    {consultation.follow_up_date && (
                        <p>
                            <strong>Date:</strong> {moment(consultation.follow_up_date).format('MMMM Do YYYY')}
                        </p>
                    )}
                    {consultation.referral_notes && (
                        <p>
                            <strong>Referral Notes:</strong> {consultation.referral_notes}
                        </p>
                    )}
                </section>
            ) : null}
        </div>
    );
});

export default ConsultationPrint;
