// resources/js/Pages/Consultations/ConsultationReferralsForm.tsx
import { Head, useForm, usePage, router } from '@inertiajs/react';
import AppLayout from '@/layouts/app-layout';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { BreadcrumbItem } from '@/types';
import type { PageProps as InertiaPageProps } from '@inertiajs/core';
import {
  CalendarDays,
  Hospital,
  ClipboardList,
  FileText,
  ArrowLeft,
} from 'lucide-react';
import { useMemo } from 'react';
import Swal from 'sweetalert2';

// ⬇️ Print components you already have
import PrintSickLeaveNote from './components/PrintSickLeaveNote';
import PrintReferralNote from './components/PrintReferralNote';
import PrintAttendanceSlip from '@/components/PrintAttendanceSlip';

interface PageProps extends InertiaPageProps {
  consultation: {
    id: number | string;
    patient_id: number | string;
    doctor_id: number | string;
    clinic_id: number | string;
    consultation_date: string | null;
    injury_on_duty: boolean;

    systolic: string | null;
    diastolic: string | null;
    temperature: string | null;
    heart_rate: string | null;
    respiratory_rate: string | null;
    oxygen_saturation: string | null;
    weight: string | null;
    height: string | null;
    bmi: string | null;

    blood_sugar?: string | null;
    blood_sugar_unit?: 'mmol/L' | 'mg/dL' | null;
    blood_sugar_context?: 'random' | 'fasting' | 'postprandial' | 'ogtt' | 'other' | null;
    blood_sugar_measured_at?: string | null;

    urine_test_result: string | null;
    hiv_status: string | null;
    pregnancy_test_result: string | null;

    presenting_complaint: string | null;
    clinical_notes: string | null;
    diagnosis: string | null;
    treatment_plan: string | null;
    prescription: string | null;

    follow_up_required: boolean;
    follow_up_date: string | null;
    referral_notes: string | null;

    refer_for_radiology: boolean;
    refer_for_lab: boolean;
    refer_to_specialist: boolean;
    specialist_name: string | null;
    refer_to_casualty: boolean;
    refer_out_patient: boolean;

    // NEW
    restricted_duty: boolean;
    referred_hospital: string | null;

    sick_leave: boolean;
    reason_for_leave: string | null;
    sick_leave_notes: string | null;
    number_of_days: number | string | null;

    created_at?: string;
    updated_at?: string;

    patient: {
      id: number | string;
      full_name: string;
      first_name?: string;
      surname?: string;
      employee_number?: string | null;
      home_address?: string | null;
      date_of_birth?: string | null;
      email?: string | null;
      gender?: string | null;
      company?: string | null;
      id_number?: string | null;
    };
    doctor: {
      name?: string | null;
      email?: string | null;
      signature?: string | null;
      qualifications?: string | null;
    };
    clinic: {
      name?: string | null;
      address?: string | null;
      contact_person?: string | null;
      email_address?: string | null;
      city?: string | null;
    };

    doctors_note?: {
      id: number;
      title: string | null;
      body: string | null;
      note_type?: string | null;
      private: boolean;
      created_at?: string | null;
      doctor?: { id?: number; name?: string | null } | null;
    } | null;
  };
}

export default function ConsultationReferralsForm() {
  const { consultation } = usePage<PageProps>().props;

  const breadcrumbs: BreadcrumbItem[] = useMemo(
    () => [
      { title: 'Patients', href: route('patients.index') },
      { title: 'Consultations', href: route('consultations.index') },
      { title: 'Edit Referrals', href: '#' },
    ],
    []
  );

  const { data, setData, patch, processing, errors } = useForm({
    follow_up_required: consultation.follow_up_required ?? false,
    follow_up_date: consultation.follow_up_date ?? '',
    referral_notes: consultation.referral_notes ?? '',

    refer_for_radiology: consultation.refer_for_radiology ?? false,
    refer_for_lab: consultation.refer_for_lab ?? false,
    refer_to_specialist: consultation.refer_to_specialist ?? false,
    specialist_name: consultation.specialist_name ?? '',
    refer_to_casualty: consultation.refer_to_casualty ?? false,
    refer_out_patient: consultation.refer_out_patient ?? false,

    // NEW
    restricted_duty: consultation.restricted_duty ?? false,
    referred_hospital: consultation.referred_hospital ?? '',

    sick_leave: consultation.sick_leave ?? false,
    number_of_days: consultation.number_of_days ?? '',
    reason_for_leave: consultation.reason_for_leave ?? '',
    sick_leave_notes: consultation.sick_leave_notes ?? '',
  });

  const anyReferralSelected =
    data.refer_for_radiology ||
    data.refer_for_lab ||
    data.refer_to_specialist ||
    data.refer_to_casualty ||
    data.refer_out_patient;

  // Show leave details if either Sick Leave OR Restricted Duty is on
  const requireLeaveDetails = data.sick_leave || data.restricted_duty;

  // --- Mutually exclusive toggles (like radios) ---
  const onToggleSickLeave = (checked: boolean) => {
    if (checked) {
      setData('sick_leave', true);
      setData('restricted_duty', false);
    } else {
      // turning off; keep restricted as-is, and if both now false, clear detail fields
      const willHaveNone = !data.restricted_duty;
      setData('sick_leave', false);
      if (willHaveNone) {
        setData('number_of_days', '');
        setData('reason_for_leave', '');
        setData('sick_leave_notes', '');
      }
    }
  };

  const onToggleRestrictedDuty = (checked: boolean) => {
    if (checked) {
      setData('restricted_duty', true);
      setData('sick_leave', false);
    } else {
      // turning off; keep sick_leave as-is, and if both now false, clear detail fields
      const willHaveNone = !data.sick_leave;
      setData('restricted_duty', false);
      if (willHaveNone) {
        setData('number_of_days', '');
        setData('reason_for_leave', '');
        setData('sick_leave_notes', '');
      }
    }
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();

    // Enforce mutual exclusivity (safety net)
    if (data.sick_leave && data.restricted_duty) {
      await Swal.fire({
        icon: 'error',
        title: 'Choose only one',
        text: 'Please select either Sick Leave or Restricted Duty, not both.',
      });
      return;
    }

    // 🔒 Strict validation aligned with Create form
    if (requireLeaveDetails) {
      const slErrs: string[] = [];
      if (String(data.number_of_days || '').trim() === '') slErrs.push('Number of Days');
      if (String(data.reason_for_leave || '').trim() === '') slErrs.push('Reason for Leave');
      if (String(data.sick_leave_notes || '').trim() === '') slErrs.push('Notes');

      if (slErrs.length) {
        await Swal.fire({
          icon: 'error',
          title: `${data.sick_leave ? 'Sick Leave' : 'Restricted Duty'} details required`,
          html: `<div style="text-align:left">
                   <p>Because <b>${data.sick_leave ? 'Sick Leave' : 'Restricted Duty'}</b> is checked, the following fields are required:</p>
                   <ul style="margin-left:1rem; list-style:disc;">
                     ${slErrs.map((m) => `<li><b>${m}</b></li>`).join('')}
                   </ul>
                 </div>`,
        });
        return;
      }
    }

    if (anyReferralSelected) {
      const refErrs: string[] = [];
      if (String(data.referred_hospital || '').trim() === '') refErrs.push('Referred Hospital');
      if (data.refer_to_specialist && String(data.specialist_name || '').trim() === '') {
        refErrs.push('Specialist Name');
      }

      if (refErrs.length) {
        await Swal.fire({
          icon: 'error',
          title: 'Referral details required',
          html: `<div style="text-align:left">
                   <p>Since you selected a referral option, please fill:</p>
                   <ul style="margin-left:1rem; list-style:disc;">
                     ${refErrs.map((m) => `<li><b>${m}</b></li>`).join('')}
                   </ul>
                 </div>`,
        });
        return;
      }
    }

    patch(route('consultations.referrals.update', consultation.id));
  };

  const handleCloseConsultation = async (consultationId: number | string) => {
    const result = await Swal.fire({
      title: 'Close Consultation?',
      text: 'Are you sure you want to close this consultation?',
      icon: 'warning',
      showCancelButton: true,
      confirmButtonText: 'Yes, close it',
      cancelButtonText: 'Cancel',
    });
    if (!result.isConfirmed) return;

    Swal.fire({
      title: 'Closing Consultation...',
      text: 'Please wait',
      allowOutsideClick: false,
      didOpen: () => Swal.showLoading(),
    });

    router.post(route('consultations.close', consultationId), {}, {
      onSuccess: () => Swal.fire('Closed!', 'Consultation closed successfully.', 'success'),
      onError: () => Swal.fire('Error', 'Failed to close consultation.', 'error'),
    });
  };

  const handleBackToEdit = () => {
    router.visit(route('consultations.edit', consultation.id));
  };

  return (
    <AppLayout breadcrumbs={breadcrumbs}>
      <Head title="Edit Referrals" />

      <div className="w-full px-4 py-6 sm:px-6">
        <div className="mb-6 flex items-center justify-between">
          <div>
            <h1 className="text-2xl font-bold">Referral & Follow-up</h1>
            <p className="text-sm text-muted-foreground">
              Update referrals, follow-up and leave/duty for{' '}
              <span className="font-semibold">{consultation.patient.full_name}</span>
            </p>
          </div>

          <Button variant="ghost" className="border border-gray-300" onClick={handleBackToEdit}>
            <ArrowLeft className="mr-2 h-4 w-4" />
            Back to Edit Consultation
          </Button>
        </div>

        {/* Two-column layout */}
        <div className="grid grid-cols-1 gap-6 md:grid-cols-3">
          {/* LEFT: form */}
          <form onSubmit={handleSubmit} className="md:col-span-2 grid grid-cols-1 gap-6">
            {/* Follow-up */}
            <div className="bg-card rounded-lg border p-6 shadow-sm">
              <h2 className="mb-4 flex items-center gap-2 text-xl font-semibold">
                <CalendarDays className="h-5 w-5 text-blue-600" />
                Follow-up
              </h2>

              <div className="space-y-4">
                <div className="flex items-center gap-2">
                  <input
                    id="follow_up_required"
                    type="checkbox"
                    checked={data.follow_up_required}
                    onChange={(e) => setData('follow_up_required', e.target.checked)}
                  />
                  <Label htmlFor="follow_up_required">Follow-up Required</Label>
                </div>

                <div>
                  <Label htmlFor="follow_up_date">Follow-up Date</Label>
                  <Input
                    id="follow_up_date"
                    type="date"
                    value={data.follow_up_date || ''}
                    onChange={(e) => setData('follow_up_date', e.target.value)}
                  />
                  {errors.follow_up_date && (
                    <p className="mt-1 text-sm text-red-500">{errors.follow_up_date}</p>
                  )}
                </div>

                <div>
                  <Label htmlFor="referral_notes">Referral Notes</Label>
                  <textarea
                    id="referral_notes"
                    rows={4}
                    className="mt-1 w-full rounded-md border border-gray-300 p-2 dark:border-gray-700 dark:bg-black"
                    value={data.referral_notes || ''}
                    onChange={(e) => setData('referral_notes', e.target.value)}
                  />
                  {errors.referral_notes && (
                    <p className="mt-1 text-sm text-red-500">{errors.referral_notes}</p>
                  )}
                </div>
              </div>
            </div>

            {/* Referrals */}
            <div className="bg-card rounded-lg border p-6 shadow-sm">
              <h2 className="mb-4 flex items-center gap-2 text-xl font-semibold">
                <Hospital className="h-5 w-5 text-blue-600" />
                Referrals
              </h2>

              <div className="space-y-4">
                {([
                  ['refer_for_radiology', 'Refer for Radiology'],
                  ['refer_for_lab', 'Refer for Lab'],
                  ['refer_to_specialist', 'Refer to Specialist'],
                  ['refer_to_casualty', 'Refer to Casualty'],
                  ['refer_out_patient', 'Refer Out Patient'],
                ] as const).map(([key, label]) => (
                  <div key={key} className="flex items-center gap-2">
                    <input
                      id={key}
                      type="checkbox"
                      checked={Boolean((data as any)[key])}
                      onChange={(e) => setData(key, e.target.checked)}
                    />
                    <Label htmlFor={key}>{label}</Label>
                  </div>
                ))}

                {data.refer_to_specialist && (
                  <div>
                    <Label htmlFor="specialist_name">Specialist Name</Label>
                    <Input
                      id="specialist_name"
                      value={data.specialist_name || ''}
                      onChange={(e) => setData('specialist_name', e.target.value)}
                    />
                    {errors.specialist_name && (
                      <p className="mt-1 text-sm text-red-500">{errors.specialist_name}</p>
                    )}
                  </div>
                )}
              </div>

              {/* NEW: Referred Hospital when any referral selected */}
              {anyReferralSelected && (
                <div className="mt-4 grid grid-cols-1 gap-4">
                  <div>
                    <Label htmlFor="referred_hospital">Referred Hospital</Label>
                    <Input
                      id="referred_hospital"
                      value={data.referred_hospital || ''}
                      onChange={(e) => setData('referred_hospital', e.target.value)}
                      placeholder="Type the hospital/clinic to which the patient is being referred"
                    />
                    {errors.referred_hospital && (
                      <p className="mt-1 text-sm text-red-500">{errors.referred_hospital}</p>
                    )}
                  </div>
                </div>
              )}
            </div>

            {/* Sick Leave / Restricted Duty (mutually exclusive) */}
            <div className="bg-card rounded-lg border p-6 shadow-sm">
              <h2 className="mb-4 flex items-center gap-2 text-xl font-semibold">
                <ClipboardList className="h-5 w-5 text-blue-600" />
                Leave / Duty
              </h2>

              <div className="grid grid-cols-1 gap-4 sm:grid-cols-3">
                <div className="flex items-center gap-2">
                  <input
                    id="sick_leave"
                    type="checkbox"
                    checked={data.sick_leave}
                    onChange={(e) => onToggleSickLeave(e.target.checked)}
                  />
                  <Label htmlFor="sick_leave">Sick Leave</Label>
                </div>

                <div className="flex items-center gap-2">
                  <input
                    id="restricted_duty"
                    type="checkbox"
                    checked={data.restricted_duty}
                    onChange={(e) => onToggleRestrictedDuty(e.target.checked)}
                  />
                  <Label htmlFor="restricted_duty">Restricted Duty</Label>
                </div>

                {requireLeaveDetails && (
                  <>
                    <div>
                      <Label htmlFor="number_of_days">Number of Days</Label>
                      <Input
                        id="number_of_days"
                        type="number"
                        value={String(data.number_of_days || '')}
                        onChange={(e) => setData('number_of_days', e.target.value)}
                      />
                      {errors.number_of_days && (
                        <p className="mt-1 text-sm text-red-500">{errors.number_of_days}</p>
                      )}
                    </div>

                    <div className="sm:col-span-3 grid grid-cols-1 gap-4 sm:grid-cols-2">
                      <div>
                        <Label htmlFor="reason_for_leave">
                          {data.sick_leave ? 'Reason for Leave' : 'Reason / Instruction (Restricted Duty)'}
                        </Label>
                        <textarea
                          id="reason_for_leave"
                          rows={3}
                          className="mt-1 w-full rounded-md border border-gray-300 p-2 dark:border-gray-700 dark:bg-black"
                          value={data.reason_for_leave || ''}
                          onChange={(e) => setData('reason_for_leave', e.target.value)}
                        />
                        {errors.reason_for_leave && (
                          <p className="mt-1 text-sm text-red-500">{errors.reason_for_leave}</p>
                        )}
                      </div>

                      <div>
                        <Label htmlFor="sick_leave_notes">
                          {data.sick_leave ? 'Sick Leave Notes' : 'Restricted Duty Notes'}
                        </Label>
                        <textarea
                          id="sick_leave_notes"
                          rows={3}
                          className="mt-1 w-full rounded-md border border-gray-300 p-2 dark:border-gray-700 dark:bg-black"
                          value={data.sick_leave_notes || ''}
                          onChange={(e) => setData('sick_leave_notes', e.target.value)}
                        />
                        {errors.sick_leave_notes && (
                          <p className="mt-1 text-sm text-red-500">{errors.sick_leave_notes}</p>
                        )}
                      </div>
                    </div>
                  </>
                )}
              </div>
            </div>

            {/* Save / Cancel */}
            <div className="flex items-center justify-end gap-2">
              <Button type="button" variant="outline" onClick={() => window.history.back()}>
                Cancel
              </Button>
              <Button type="submit" disabled={processing}>
                {processing ? 'Saving…' : 'Save Referrals'}
              </Button>
            </div>
          </form>

          {/* RIGHT: actions */}
          <div className="md:col-span-1 space-y-6">
            <div className="bg-card rounded-lg border p-6 shadow-sm">
              <h2 className="mb-4 text-xl font-semibold">Actions</h2>

              <div className="space-y-2">
                <Button
                  className="w-full"
                  onClick={() => handleCloseConsultation(consultation.id)}
                  style={{ cursor: 'pointer' }}
                >
                  Close Consultation
                </Button>

                <PrintAttendanceSlip
                  consultation={consultation}
                  buttonVariant="secondary"
                  className="w-full"
                />

                <Button
                  variant="outline"
                  className="w-full"
                  onClick={handleBackToEdit}
                  style={{ cursor: 'pointer' }}
                >
                  <FileText className="mr-2 h-4 w-4" />
                  Back to Edit Consultation
                </Button>
              </div>
            </div>

            {/* Print components (need full consultation object) */}
            <div className="bg-card rounded-lg border p-6 shadow-sm">
              <h2 className="mb-4 text-xl font-semibold">Printable Notes</h2>
              {consultation.sick_leave || consultation.restricted_duty ? (
                <PrintSickLeaveNote consultation={consultation as any} />
              ) : (
                <p className="text-muted-foreground text-sm">No leave/restricted duty note available</p>
              )}

              <div className="mt-3">
                <PrintReferralNote consultation={consultation} />
              </div>
            </div>
          </div>
        </div>
      </div>
    </AppLayout>
  );
}
