import { Button } from '@/components/ui/button';
import AppLayout from '@/layouts/app-layout';
import { BreadcrumbItem } from '@/types';
import { PageProps as InertiaPageProps } from '@inertiajs/core';
import { Head, router, usePage } from '@inertiajs/react';
import { Table, TableHeader, TableRow, TableHead, TableBody, TableCell } from '@/components/ui/table';
import { ScrollArea } from '@/components/ui/scroll-area';
import { Badge } from '@/components/ui/badge';
import { Alert, AlertTitle, AlertDescription } from '@/components/ui/alert';
import {
  AudioWaveform,
  Pencil,
  Pill,
  TestTube,
  ClipboardList,
  Activity,
  Thermometer,
  Wind,
  Droplet,
  Ruler,
  Scale,
  HeartPulse,
  Stethoscope,
  Microscope,
  FlaskConical,
  Hospital,
  User,
  CalendarDays,
  FileText,
  Printer,
  StickyNote,
  ShieldAlert
} from 'lucide-react';
import moment from 'moment';
import { QRCodeSVG } from 'qrcode.react';
import { useRef } from 'react';
import { useReactToPrint } from 'react-to-print';
import { type SharedData } from '@/types';
import PrintSickLeaveNote from './components/PrintSickLeaveNote';
import PrintReferralNote from './components/PrintReferralNote';
import PrintAttendanceSlip from '@/components/PrintAttendanceSlip';

const logoUrl = 'https://providencehumancapital.com/headoffice/assets/images/PHC_Logo.png';

/** Typed dispensations payload **/
interface DispensedBy {
  id: number;
  name: string;
  email?: string | null;
}

interface DispensationItem {
  id: number;
  consultation_id: number;
  patient_id: number;
  origin?: 'nurse' | 'clinic' | null;
  clinic_medication_stock_id?: number | null;
  nurse_medication_stock_id?: number | null;
  medication?: {
    name?: string;
    dosage?: string | null;
    form?: string | null;
    unit?: string | null;
  } | null;
  batch_number?: string | null;
  quantity: number;
  frequency?: string | null;
  route?: string | null;
  duration?: string | number | null;
  notes?: string | null;
  dispensed_at?: string | null;
  dispensed_by?: DispensedBy | null;
}

interface PageProps extends InertiaPageProps {
  consultation: {
    id: string;
    patient_id: string;
    doctor_id: string;
    clinic_id: string;
    consultation_date: string;
    injury_on_duty: boolean;
    systolic: string;
    diastolic: string;
    temperature: string;
    heart_rate: string;
    respiratory_rate: string;
    oxygen_saturation: string;
    weight: string;
    height: string;
    bmi: string;

    blood_sugar?: string;
    blood_sugar_unit?: 'mmol/L' | 'mg/dL';
    blood_sugar_context?: 'random' | 'fasting' | 'postprandial' | 'ogtt' | 'other';
    blood_sugar_measured_at?: string;

    urine_test_result: string;
    hiv_status: string;
    pregnancy_test_result: string;

    presenting_complaint: string;
    clinical_notes: string;
    diagnosis: string;
    treatment_plan: string;
    prescription: string;
    follow_up_required: boolean;

    // Leave / duty
    sick_leave: boolean;
    restricted_duty?: boolean;
    reason_for_leave: string;
    sick_leave_notes: string;
    number_of_days: string;

    // Referrals
    refer_for_radiology: boolean;
    refer_for_lab: boolean;
    refer_to_specialist: boolean;
    specialist_name: string;
    refer_to_casualty: boolean;
    refer_out_patient: boolean;
    referred_hospital?: string;

    follow_up_date: string;
    referral_notes: string;

    // Doctor’s note (optional)
    doctors_note?: {
      id: number;
      title: string | null;
      body: string | null;
      note_type?: string | null;
      private: boolean;
      created_at?: string | null;
      doctor?: { id?: number; name?: string | null } | null;
    } | null;

    patient: {
      id: number;
      full_name: string;
      first_name: string;
      surname: string;
      employee_number: string;
      home_address: string;
      date_of_birth: string;
      email: string;
      gender: string;
      id_number: string;
    };
    doctor: {
      name: string;
      email: string;
      signature: string;
      qualifications: string;
    };
    clinic: {
      name: string;
      address: string;
      contact_person: string;
      email_address: string;
      city: string;
    };

    /** Permissions + dispensations **/
    can_edit?: boolean;
    dispensations?: DispensationItem[];

    /** OPTIONAL: backend can send this to avoid heuristics */
    consulted_by_doctor?: boolean;
  };
}

type UserRole =
  | 'superadmin'
  | 'receptionist'
  | 'nurse'
  | 'doctor'
  | 'lab_technician'
  | 'radiologist'
  | 'pharmacist'
  | 'nurse_aid'
  | 'admin'
  | 'specialist'
  | 'casualty'
  | 'user'
  | 'patient';

export default function ConsultationShow() {
  const { consultation } = usePage<PageProps>().props;

  const page = usePage<SharedData>();
  const { auth } = page.props;
  const userRole: UserRole | undefined = auth.user?.role as UserRole | undefined;
  const rolesArr = Array.isArray((auth.user as any)?.roles) ? ((auth.user as any).roles as string[]) : [];

  const isNurse = userRole === 'nurse' || rolesArr.includes('nurse');
  const canSeeClinical =
    isNurse ||
    userRole === 'doctor' ||
    rolesArr.includes('doctor');

  // Prefer backend flag; fallback to checking clinician name/qualifications for "Dr/Doctor"
  const wasConsultedByDoctor =
    consultation.consulted_by_doctor === true ||
    /(^|\b)(dr\.?|doctor)\b/i.test(
      `${consultation.doctor?.qualifications ?? ''} ${consultation.doctor?.name ?? ''}`
    );

  const contentRef = useRef<HTMLDivElement>(null);
  const reactToPrintFn = useReactToPrint({ contentRef });

  const breadcrumbs: BreadcrumbItem[] = [
    { title: 'Patients', href: route('patients.index') },
    { title: 'Consultations', href: route('consultations.index') },
    { title: 'View Consultation', href: '#' },
  ];

  const renderRow = (label: string, value?: string | number | boolean | null, icon?: JSX.Element) => {
    if (value === null || value === '' || value === undefined) return null;
    return (
      <div className="flex items-center gap-2 border p-2">
        {icon}
        <div>
          <p className="font-semibold">{label}</p>
          <p className="text-gray-700 dark:text-gray-300">
            {typeof value === 'boolean' ? (value ? 'Yes' : 'No') : value}
          </p>
        </div>
      </div>
    );
  };

  const bsUnit = consultation.blood_sugar_unit || 'mmol/L';
  const bsContext = consultation.blood_sugar_context || undefined;
  const bsMeasuredAt = consultation.blood_sugar_measured_at
    ? moment(consultation.blood_sugar_measured_at).format('MMM D, YYYY • HH:mm')
    : undefined;

  const note = consultation.doctors_note || null;

  const anyReferralSelected =
    consultation.refer_for_radiology ||
    consultation.refer_for_lab ||
    consultation.refer_to_specialist ||
    consultation.refer_to_casualty ||
    consultation.refer_out_patient;

  /** Medication helpers **/
  const hasDispensations = Array.isArray(consultation.dispensations) && consultation.dispensations.length > 0;

  const formatMedLine = (d: DispensationItem) => {
    const name = d.medication?.name ?? '—';
    const dosage = d.medication?.dosage ? ` ${d.medication.dosage}` : '';
    const unit = d.medication?.unit ? ` ${d.medication.unit}` : '';
    const form = d.medication?.form ? ` ${d.medication.form}` : '';
    return `${name}${dosage}${unit}${form}`.trim();
  };

  const originBadge = (origin?: 'nurse' | 'clinic' | null) => {
    if (!origin) return null;
    const label = origin === 'nurse' ? 'Nurse Stock' : 'Clinic Stock';
    const color =
      origin === 'nurse'
        ? 'bg-amber-100 text-amber-800 border-amber-200'
        : 'bg-emerald-100 text-emerald-800 border-emerald-200';
    return <span className={`rounded border px-2 py-0.5 text-xs ${color}`}>{label}</span>;
  };

  return (
    <AppLayout breadcrumbs={breadcrumbs}>
      <Head title="View Consultation" />

      {/* Hidden printable content */}
      <div style={{ display: 'none' }}>
        <div ref={contentRef} className="p-6 font-sans text-sm text-gray-800">
          <style>
            {`
              @page { size: auto; margin: 15mm 10mm; }
              @media print {
                body { -webkit-print-color-adjust: exact; print-color-adjust: exact; }
                .section-break { page-break-after: always; }
                .avoid-break { page-break-inside: avoid; }
              }
            `}
          </style>

          {/* Header */}
          <div className="flex items-center justify-between border-b-2 border-gray-200 pb-4">
            <div className="flex items-center">
              <img src={logoUrl} alt="PHC Logo" style={{ height: '4rem' }} />
            </div>
            <div className="text-center text-sm text-gray-800">
              <p className="text-lg font-bold">Consultation Summary</p>
              <p>{moment(consultation.consultation_date).format('MMMM Do YYYY, h:mm a')}</p>
            </div>
            <div className="flex items-center">
              <QRCodeSVG value={`Consultation ID: ${consultation.id}`} size={60} />
            </div>
          </div>

          {/* Patient Info */}
          <div className="mt-4">
            <p className="mb-2 text-lg font-bold">Patient Information</p>
            <div className="grid grid-cols-2 gap-y-2">
              <p className="font-semibold">Patient:</p>
              <p>{consultation.patient.full_name}</p>

              <p className="font-semibold">Clinic:</p>
              <p>{consultation.clinic?.name}</p>

              <p className="font-semibold">Consulting Doctor/RGN:</p>
              <p>{consultation.doctor.name}</p>

              <p className="font-semibold">Injury on Duty:</p>
              <p className={consultation.injury_on_duty ? 'text-red-600' : 'text-green-600'}>
                {consultation.injury_on_duty ? 'Yes' : 'No'}
              </p>
            </div>
          </div>

          {/* Vitals */}
          <div className="mt-6">
            <p className="mb-2 text-lg font-bold">Vitals & Measurements</p>
            <div className="grid grid-cols-2 gap-2 md:grid-cols-4">
              {consultation.systolic && consultation.diastolic && (
                <div className="border p-2">
                  <p className="font-semibold">Blood Pressure</p>
                  <p>{consultation.systolic}/{consultation.diastolic} mmHg</p>
                </div>
              )}
              {consultation.temperature && (
                <div className="border p-2">
                  <p className="font-semibold">Temperature</p>
                  <p>{consultation.temperature} °C</p>
                </div>
              )}
              {consultation.heart_rate && (
                <div className="border p-2">
                  <p className="font-semibold">Heart Rate</p>
                  <p>{consultation.heart_rate} bpm</p>
                </div>
              )}
              {consultation.oxygen_saturation && (
                <div className="border p-2">
                  <p className="font-semibold">O₂ Saturation</p>
                  <p>{consultation.oxygen_saturation} %</p>
                </div>
              )}
              {consultation.weight && (
                <div className="border p-2">
                  <p className="font-semibold">Weight</p>
                  <p>{consultation.weight} kg</p>
                </div>
              )}
              {consultation.height && (
                <div className="border p-2">
                  <p className="font-semibold">Height</p>
                  <p>{consultation.height} cm</p>
                </div>
              )}
              {consultation.bmi && (
                <div className="border p-2">
                  <p className="font-semibold">BMI</p>
                  <p>{consultation.bmi}</p>
                </div>
              )}
            </div>
          </div>

          {/* Tests */}
          <div className="mt-6">
            <p className="mb-2 text-lg font-bold">Test Results</p>
            <table className="w-full border-collapse">
              <thead>
                <tr>
                  <th className="border p-2 text-left font-semibold">Test</th>
                  <th className="border p-2 text-left font-semibold">Result</th>
                </tr>
              </thead>
              <tbody>
                {consultation.blood_sugar && (
                  <tr>
                    <td className="border p-2">Blood Sugar</td>
                    <td className="border p-2">
                      {consultation.blood_sugar} {bsUnit}
                      {bsContext ? ` • ${bsContext}` : ''}
                      {bsMeasuredAt ? ` • ${bsMeasuredAt}` : ''}
                    </td>
                  </tr>
                )}
                {consultation.urine_test_result && (
                  <tr>
                    <td className="border p-2">Urine Test</td>
                    <td className="border p-2">{consultation.urine_test_result}</td>
                  </tr>
                )}
                {consultation.hiv_status && (
                  <tr>
                    <td className="border p-2">HIV Status</td>
                    <td className="border p-2">{consultation.hiv_status}</td>
                  </tr>
                )}
                {consultation.pregnancy_test_result && (
                  <tr>
                    <td className="border p-2">Pregnancy Test</td>
                    <td className="border p-2">{consultation.pregnancy_test_result}</td>
                  </tr>
                )}
              </tbody>
            </table>
          </div>

          {/* Clinical (PRINT) */}
          {canSeeClinical && (
            <div className="mt-6">
              <p className="mb-2 text-lg font-bold">Clinical Information</p>
              {consultation.presenting_complaint && (
                <div className="mb-4">
                  <p className="font-semibold">Presenting Complaint</p>
                  <p className="border p-2">{consultation.presenting_complaint}</p>
                </div>
              )}
              {consultation.clinical_notes && (
                <div className="mb-4">
                  <p className="font-semibold">Clinical Notes</p>
                  <p className="border p-2 whitespace-pre-wrap">{consultation.clinical_notes}</p>
                </div>
              )}
              {consultation.diagnosis && (
                <div className="mb-4">
                  <p className="font-semibold">Diagnosis</p>
                  <p className="border p-2 whitespace-pre-wrap">{consultation.diagnosis}</p>
                </div>
              )}
              {consultation.treatment_plan && (
                <div className="mb-4">
                  <p className="font-semibold">Treatment Plan</p>
                  <p className="border p-2 whitespace-pre-wrap">{consultation.treatment_plan}</p>
                </div>
              )}
              {consultation.prescription && (
                <div className="mb-4">
                  <p className="font-semibold">Prescription</p>
                  <p className="border p-2 whitespace-pre-wrap">{consultation.prescription}</p>
                </div>
              )}
            </div>
          )}

          {/* Medication Dispensed (PRINT) */}
          {hasDispensations && (
            <div className="mt-6 avoid-break">
              <p className="mb-2 text-lg font-bold">Medication Dispensed</p>
              <table className="w-full border-collapse text-sm">
                <thead>
                  <tr>
                    <th className="border p-2 text-left">Medication</th>
                    <th className="border p-2 text-left">Batch</th>
                    <th className="border p-2 text-right">Qty</th>
                    <th className="border p-2 text-left">Frequency</th>
                    <th className="border p-2 text-left">Route</th>
                    <th className="border p-2 text-left">Duration</th>
                    <th className="border p-2 text-left">Origin</th>
                    <th className="border p-2 text-left">Dispensed By</th>
                  </tr>
                </thead>
                <tbody>
                  {consultation.dispensations!.map((d) => (
                    <tr key={d.id}>
                      <td className="border p-2">{formatMedLine(d)}</td>
                      <td className="border p-2">{d.batch_number ?? '—'}</td>
                      <td className="border p-2 text-right">{d.quantity}</td>
                      <td className="border p-2">{d.frequency ?? '—'}</td>
                      <td className="border p-2">{d.route ?? '—'}</td>
                      <td className="border p-2">
                        {d.duration ? `${d.duration} day${Number(d.duration) === 1 ? '' : 's'}` : '—'}
                      </td>
                      <td className="border p-2 capitalize">{d.origin ?? '—'}</td>
                      <td className="border p-2">
                        {d.dispensed_by?.name ?? '—'}
                        {d.dispensed_at ? ` • ${moment(d.dispensed_at).format('MMM D, YYYY HH:mm')}` : ''}
                      </td>
                    </tr>
                  ))}
                </tbody>
              </table>
            </div>
          )}

          {/* Follow-up */}
          {(consultation.follow_up_required || consultation.follow_up_date || consultation.referral_notes) && (
            <div className="mt-6">
              <p className="mb-2 text-lg font-bold">Follow-up Information</p>
              <div className="grid grid-cols-2 gap-2 md:grid-cols-3">
                {consultation.follow_up_required && (
                  <div className="border p-2">
                    <p className="font-semibold">Follow-up Required</p>
                    <p className="text-green-600">Yes</p>
                  </div>
                )}
                {consultation.follow_up_date && (
                  <div className="border p-2">
                    <p className="font-semibold">Follow-up Date</p>
                    <p>{moment(consultation.follow_up_date).format('MMMM Do YYYY')}</p>
                  </div>
                )}
                {consultation.referral_notes && (
                  <div className="border p-2 md:col-span-3">
                    <p className="font-semibold">Referral Notes</p>
                    <p className="mt-1">{consultation.referral_notes}</p>
                  </div>
                )}
              </div>
            </div>
          )}

          {/* Referrals (PRINT) */}
          {(anyReferralSelected || consultation.referred_hospital) && (
            <div className="mt-6">
              <p className="mb-2 text-lg font-bold">Referrals</p>
              <div className="grid grid-cols-2 gap-2 md:grid-cols-3">
                {consultation.refer_for_radiology && (
                  <div className="border p-2">
                    <p className="font-semibold">Radiology / Imaging</p>
                    <p>Yes</p>
                  </div>
                )}
                {consultation.refer_for_lab && (
                  <div className="border p-2">
                    <p className="font-semibold">Laboratory Tests</p>
                    <p>Yes</p>
                  </div>
                )}
                {consultation.refer_to_specialist && (
                  <div className="border p-2">
                    <p className="font-semibold">Specialist</p>
                    <p>{consultation.specialist_name ? consultation.specialist_name : 'Yes'}</p>
                  </div>
                )}
                {consultation.refer_to_casualty && (
                  <div className="border p-2">
                    <p className="font-semibold">Casualty / Emergency</p>
                    <p>Yes</p>
                  </div>
                )}
                {consultation.refer_out_patient && (
                  <div className="border p-2">
                    <p className="font-semibold">Out-Patient Department</p>
                    <p>Yes</p>
                  </div>
                )}
                {consultation.referred_hospital && (
                  <div className="border p-2 md:col-span-3">
                    <p className="font-semibold">Referred Hospital</p>
                    <p>{consultation.referred_hospital}</p>
                  </div>
                )}
              </div>
            </div>
          )}

          {/* Sick Leave / Restricted Duty (PRINT) */}
          {(consultation.sick_leave || consultation.restricted_duty || consultation.number_of_days || consultation.reason_for_leave || consultation.sick_leave_notes) && (
            <div className="mt-6">
              <p className="mb-2 text-lg font-bold">Sick Leave / Restricted Duty</p>
              <div className="grid grid-cols-2 gap-2 md:grid-cols-3">
                <div className="border p-2">
                  <p className="font-semibold">Sick Leave Granted</p>
                  <p>{consultation.sick_leave ? 'Yes' : 'No'}</p>
                </div>
                <div className="border p-2">
                  <p className="font-semibold">Restricted Duty</p>
                  <p>{consultation.restricted_duty ? 'Yes' : 'No'}</p>
                </div>
                {consultation.number_of_days && (
                  <div className="border p-2">
                    <p className="font-semibold">Number of Days</p>
                    <p>{consultation.number_of_days}</p>
                  </div>
                )}
              </div>

              {(consultation.reason_for_leave || consultation.sick_leave_notes) && (
                <div className="mt-3 grid gap-3 md:grid-cols-2">
                  {consultation.reason_for_leave && (
                    <div className="rounded border p-3">
                      <p className="text-xs font-semibold text-gray-500">Reason</p>
                      <p className="text-sm whitespace-pre-wrap">{consultation.reason_for_leave}</p>
                    </div>
                  )}
                  {consultation.sick_leave_notes && (
                    <div className="rounded border p-3">
                      <p className="text-xs font-semibold text-gray-500">Notes</p>
                      <p className="text-sm whitespace-pre-wrap">{consultation.sick_leave_notes}</p>
                    </div>
                  )}
                </div>
              )}
            </div>
          )}

          {/* Footer */}
          <div className="mt-8 border-t pt-4 text-xs text-gray-500">
            <p>© {new Date().getFullYear()} Providence Health • Generated on {moment().format('MMMM Do YYYY, h:mm a')}</p>
          </div>
        </div>
      </div>

      {/* On-screen content */}
      <div className="w-full px-4 py-6 sm:px-6">
        <div className="mb-6 flex items-center justify-between">
          <div>
            <h1 className="flex items-center gap-2 text-2xl font-bold">
              <Stethoscope className="h-5 w-5 text-blue-600" />
              Consultation Details
            </h1>
            <p className="flex items-center gap-1 text-sm text-gray-500">
              <CalendarDays className="h-4 w-4" />
              {moment(consultation.consultation_date).format('MMMM Do YYYY, h:mm a')}
            </p>
          </div>

          <div className="flex flex-wrap gap-2">
            <Button variant="ghost" className="border border-gray-300" onClick={() => window.history.back()}>
              <span className="flex items-center gap-2">
                <svg xmlns="http://www.w3.org/2000/svg" className="h-4 w-4" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M15 19l-7-7 7-7" />
                </svg>
                Back
              </span>
            </Button>

            <Button variant="secondary" className="border border-gray-300" onClick={reactToPrintFn}>
              <Printer className="mr-2 h-4 w-4" />
              Print
            </Button>
          </div>
        </div>

        <div className="grid grid-cols-1 gap-8 md:grid-cols-3">
          <div className="col-span-2 space-y-6">
            {/* Patient Info */}
            <div className="bg-card rounded-lg border p-6 shadow-sm">
              <h2 className="mb-4 flex items-center gap-2 text-xl font-semibold">
                <User className="h-5 w-5 text-blue-600" />
                Patient Information
              </h2>
              <div className="grid grid-cols-1 gap-4 sm:grid-col-3">
                {renderRow('Patient', consultation.patient.full_name)}
                {renderRow('Clinic', consultation.clinic?.name)}
                {renderRow('Doctor / RGN', consultation.doctor.name)}
                {renderRow('Injury on Duty', consultation.injury_on_duty)}
              </div>
            </div>

            {/* Vitals & Measurements */}
            <div className="bg-card rounded-lg border p-6 shadow-sm">
              <h2 className="mb-4 flex items-center gap-2 text-xl font-semibold">
                <Activity className="h-5 w-5 text-blue-600" />
                Vitals & Measurements
              </h2>
              <div className="grid grid-cols-2 gap-4 sm:grid-cols-3">
                {consultation.systolic &&
                  consultation.diastolic &&
                  renderRow(
                    'Blood Pressure',
                    `${consultation.systolic}/${consultation.diastolic} mmHg`,
                    <Activity className="h-4 w-4 text-blue-600" />
                  )}
                {consultation.temperature &&
                  renderRow('Temperature', `${consultation.temperature} °C`, <Thermometer className="h-4 w-4 text-orange-600" />)}
                {consultation.heart_rate &&
                  renderRow('Heart Rate', `${consultation.heart_rate} bpm`, <HeartPulse className="h-4 w-4 text-red-600" />)}
                {consultation.respiratory_rate &&
                  renderRow('Respiratory Rate', `${consultation.respiratory_rate} bpm`, <Wind className="h-4 w-4 text-cyan-600" />)}
                {consultation.oxygen_saturation &&
                  renderRow('Oxygen Saturation', `${consultation.oxygen_saturation}%`, <Droplet className="h-4 w-4 text-sky-600" />)}
                {consultation.weight && renderRow('Weight', `${consultation.weight} kg`, <Scale className="h-4 w-4 text-emerald-600" />)}
                {consultation.height && renderRow('Height', `${consultation.height} cm`, <Ruler className="h-4 w-4 text-violet-600" />)}
                {consultation.bmi && renderRow('BMI', consultation.bmi, <AudioWaveform className="h-4 w-4 text-fuchsia-600" />)}
              </div>
            </div>

            {/* Clinical Information — visible only to nurse/doctor */}
            {canSeeClinical && (
              <div className="bg-card rounded-lg border p-6 shadow-sm">
                <h2 className="mb-4 flex items-center gap-2 text-xl font-semibold">
                  <FileText className="h-5 w-5 text-blue-600" />
                  Clinical Information
                </h2>
                {renderRow('Presenting Complaint', consultation.presenting_complaint)}
                {renderRow('Clinical Notes', consultation.clinical_notes)}
                {renderRow('Diagnosis', consultation.diagnosis)}
                {renderRow('Treatment Plan', consultation.treatment_plan)}
                {renderRow('Prescription', consultation.prescription)}
              </div>
            )}

            {/* Doctor’s Note (screen) */}
            {(() => {
              const note = consultation.doctors_note || null;
              return (
                note && (
                  <div className="bg-card rounded-lg border p-6 shadow-sm">
                    <h2 className="mb-4 flex items-center gap-2 text-xl font-semibold">
                      <StickyNote className="h-5 w-5 text-blue-600" />
                      Doctor’s Note
                    </h2>

                    <div className="mb-2 flex flex-wrap items-center gap-2 text-sm text-gray-600">
                      <span className="font-semibold">{note.title || 'Note'}</span>
                      {note.note_type ? (
                        <span className="rounded bg-gray-100 px-2 py-0.5 text-xs uppercase">{note.note_type}</span>
                      ) : null}
                      {note.private ? (
                        <span className="inline-flex items-center gap-1 rounded bg-amber-100 px-2 py-0.5 text-xs text-amber-800">
                          <ShieldAlert className="h-3 w-3" />
                          Private
                        </span>
                      ) : null}
                    </div>

                    <div className="rounded border p-3 text-sm whitespace-pre-wrap">
                      {note.body}
                    </div>

                    <div className="mt-2 text-xs text-gray-500">
                      {note.doctor?.name ? `Authored by ${note.doctor.name}` : null}
                      {note.created_at ? ` • ${moment(note.created_at).format('MMM D, YYYY • HH:mm')}` : null}
                    </div>
                  </div>
                )
              );
            })()}

            {/* Medication Dispensed (screen) */}
            <div className="bg-card rounded-lg border p-6 shadow-sm">
              <h2 className="mb-4 flex items-center gap-2 text-xl font-semibold">
                <Pill className="h-5 w-5 text-blue-600" />
                Medication Dispensed
              </h2>

              {!hasDispensations ? (
                <p className="text-sm text-muted-foreground">No medication was dispensed for this consultation.</p>
              ) : (
                <div className="rounded-md border">
                  <Table>
                    <TableHeader>
                      <TableRow>
                        <TableHead>Medication</TableHead>
                        <TableHead className="hidden sm:table-cell">Batch</TableHead>
                        <TableHead className="text-right">Qty</TableHead>
                        <TableHead className="hidden md:table-cell">Frequency</TableHead>
                        <TableHead className="hidden lg:table-cell">Route</TableHead>
                        <TableHead className="hidden lg:table-cell">Duration</TableHead>
                        <TableHead className="hidden sm:table-cell">Origin</TableHead>
                        <TableHead className="hidden md:table-cell">Dispensed By</TableHead>
                      </TableRow>
                    </TableHeader>
                    <TableBody>
                      {consultation.dispensations!.map((d) => (
                        <TableRow key={d.id}>
                          <TableCell className="font-medium">
                            <div className="flex items-center gap-2">
                              {formatMedLine(d)}
                              {d.notes ? (
                                <span title={d.notes} className="rounded bg-gray-100 px-1.5 py-0.5 text-[10px] uppercase">
                                  Note
                                </span>
                              ) : null}
                            </div>
                          </TableCell>
                          <TableCell className="hidden sm:table-cell">{d.batch_number ?? '—'}</TableCell>
                          <TableCell className="text-right">{d.quantity}</TableCell>
                          <TableCell className="hidden md:table-cell">{d.frequency ?? '—'}</TableCell>
                          <TableCell className="hidden lg:table-cell">{d.route ?? '—'}</TableCell>
                          <TableCell className="hidden lg:table-cell">
                            {d.duration ? `${d.duration} day${Number(d.duration) === 1 ? '' : 's'}` : '—'}
                          </TableCell>
                          <TableCell className="hidden sm:table-cell">
                            {originBadge(d.origin)}
                          </TableCell>
                          <TableCell className="hidden md:table-cell">
                            {d.dispensed_by?.name ?? '—'}
                            {d.dispensed_at ? (
                              <span className="text-xs text-muted-foreground">
                                {' '}&bull; {moment(d.dispensed_at).format('MMM D, YYYY HH:mm')}
                              </span>
                            ) : null}
                          </TableCell>
                        </TableRow>
                      ))}
                    </TableBody>
                  </Table>
                </div>
              )}
            </div>

            {/* Follow-up */}
            {(consultation.follow_up_required || consultation.follow_up_date || consultation.referral_notes) && (
              <div className="bg-card rounded-lg border p-6 shadow-sm">
                <h2 className="mb-4 flex items-center gap-2 text-xl font-semibold">
                  <CalendarDays className="h-5 w-5 text-blue-600" />
                  Follow-up
                </h2>
                <div className="grid grid-cols-1 gap-4 sm:grid-cols-2">
                  {consultation.follow_up_required && renderRow('Follow-up Required', true)}
                  {consultation.follow_up_date &&
                    renderRow('Follow-up Date', moment(consultation.follow_up_date).format('MMMM Do YYYY'))}
                </div>
                {consultation.referral_notes && (
                  <div className="mt-3 rounded border p-3">
                    <p className="text-sm text-gray-600 dark:text-gray-300">{consultation.referral_notes}</p>
                  </div>
                )}
              </div>
            )}

            {/* Referrals (screen) */}
            <div className="bg-card rounded-lg border p-6 shadow-sm">
              <h2 className="mb-4 flex items-center gap-2 text-xl font-semibold">
                <Hospital className="h-5 w-5 text-blue-600" />
                Referrals
              </h2>
              <div className="grid grid-cols-1 gap-4 sm:grid-cols-2">
                {renderRow('Radiology Referral', consultation.refer_for_radiology, <Microscope className="h-4 w-4 text-purple-600" />)}
                {renderRow('Lab Referral', consultation.refer_for_lab, <TestTube className="h-4 w-4 text-emerald-600" />)}
                {renderRow(
                  'Specialist Referral',
                  consultation.refer_to_specialist ? `Yes (${consultation.specialist_name})` : false,
                  <Stethoscope className="h-4 w-4 text-blue-600" />
                )}
                {renderRow('Casualty Referral', consultation.refer_to_casualty, <Hospital className="h-4 w-4 text-red-600" />)}
                {renderRow('Outpatient Referral', consultation.refer_out_patient, <ClipboardList className="h-4 w-4 text-sky-600" />)}

                {(anyReferralSelected || consultation.referred_hospital) &&
                  renderRow('Referred Hospital', consultation.referred_hospital || '-') }
              </div>
            </div>

            {/* Sick Leave / Restricted Duty (screen) */}
            {(consultation.sick_leave || consultation.restricted_duty || consultation.number_of_days || consultation.reason_for_leave || consultation.sick_leave_notes) && (
              <div className="bg-card rounded-lg border p-6 shadow-sm">
                <h2 className="mb-4 flex items-center gap-2 text-xl font-semibold">
                  <ClipboardList className="h-5 w-5 text-blue-600" />
                  Sick Leave / Restricted Duty
                </h2>
                <div className="grid grid-cols-1 gap-4 sm:grid-cols-2">
                  {renderRow('Sick Leave Granted', consultation.sick_leave)}
                  {renderRow('Restricted Duty', consultation.restricted_duty ?? false)}
                  {renderRow('Number of Days', consultation.number_of_days)}
                </div>
                {(consultation.reason_for_leave || consultation.sick_leave_notes) && (
                  <div className="mt-3 grid gap-3 sm:grid-cols-2">
                    {consultation.reason_for_leave && (
                      <div className="rounded border p-3">
                        <p className="text-xs font-semibold text-gray-500">Reason</p>
                        <p className="text-sm">{consultation.reason_for_leave}</p>
                      </div>
                    )}
                    {consultation.sick_leave_notes && (
                      <div className="rounded border p-3">
                        <p className="text-xs font-semibold text-gray-500">Notes</p>
                        <p className="text-sm">{consultation.sick_leave_notes}</p>
                      </div>
                    )}
                  </div>
                )}
              </div>
            )}
          </div>

          {/* Right column */}
          <div className="space-y-4">
            {/* Actions */}
            <div className="bg-card rounded-lg border p-6 shadow-sm">
              <h2 className="mb-4 flex items-center gap-2 text-xl font-semibold">
                <FileText className="h-5 w-5 text-blue-600" />
                Actions
              </h2>

              <div className="space-y-2">
                {/* Dispense Medication (nurse only, only when the consultation was by a DOCTOR) */}
                {JSON.stringify(wasConsultedByDoctor)}
                {isNurse && wasConsultedByDoctor && (
                  <Button asChild className="w-full">
                    <a
                      href={route('consultations.dispense', consultation.id)}
                      className="flex items-center justify-center gap-2"
                    >
                      <Pill className="h-4 w-4" />
                      Dispense Medication
                    </a>
                  </Button>
                )}

                <PrintAttendanceSlip
                  consultation={consultation}
                  buttonVariant="secondary"
                  className="w-full"
                />
                {consultation.sick_leave ? (
                  <PrintSickLeaveNote consultation={consultation} />
                ) : (
                  <p className="text-muted-foreground text-sm">No sick leave note available</p>
                )}

                <PrintReferralNote consultation={consultation} />

                <Button variant="secondary" className="w-full" onClick={reactToPrintFn}>
                  <Printer className="mr-2 h-4 w-4" />
                  Print Consultation
                </Button>

                {/* EDIT BUTTON: only show if server says can_edit = true */}
                {consultation.can_edit && (
                  <Button asChild className="w-full">
                    <a href={route('consultations.edit', consultation.id)} className="flex items-center justify-center gap-2">
                      <Pencil className="h-4 w-4" />
                      Edit Consultation
                    </a>
                  </Button>
                )}

                <Button asChild className="mb-2 w-full">
                  <a href={route('patients.consultations', consultation.patient_id)}>Consultation History</a>
                </Button>
                <Button variant="outline" className="w-full" asChild>
                  <a href={route('patients.show', consultation.patient_id)} className="flex items-center justify-center gap-2">
                    <User className="h-4 w-4" />
                    View Patient
                  </a>
                </Button>

                <Button variant="outline" className="w-full" asChild>
                  <a href={route('consultations.index')} className="flex items-center justify-center gap-2">
                    <FileText className="h-4 w-4" />
                    Back to List
                  </a>
                </Button>
              </div>
            </div>
          </div>
        </div>
      </div>
    </AppLayout>
  );
}

function formatCell(v: any) {
  if (v == null || v === '') return <span className="text-muted-foreground">—</span>
  if (typeof v === 'boolean') return v ? 'Yes' : 'No'
  return String(v)
}
