// resources/js/Pages/Consultations/ConsultationTransfer.tsx

import AppLayout from '@/layouts/app-layout';
import { Head, router, usePage } from '@inertiajs/react';
import { useMemo, useState } from 'react';
import Swal from 'sweetalert2';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Badge } from '@/components/ui/badge';
import { Search, User, Users, Eye } from 'lucide-react';

type PageProps = {
    consultation: {
        id: number | string;
        patient: { id: number | string; full_name: string };
    };
    patients: Array<{ id: number; full_name: string; is_dependent: boolean }>;
    filters: { q?: string | null };
};

export default function ConsultationTransfer() {
    const { consultation, patients, filters } = usePage<PageProps>().props;
    const [query, setQuery] = useState(filters.q ?? '');
    const [targetId, setTargetId] = useState<number | null>(null);
    const [submitting, setSubmitting] = useState(false);

    const onSearch = () => {
        router.get(
            route('consultations.transfer.form', consultation.id),
            { q: query || undefined },
            { preserveState: true, replace: true, preserveScroll: true }
        );
    };

    const submitTransfer = async () => {
        if (!targetId) {
            Swal.fire('Select a patient', 'Please choose a target patient', 'info');
            return;
        }
        if (Number(targetId) === Number(consultation.patient.id)) {
            Swal.fire('Invalid selection', 'Target cannot be the same as the current patient.', 'warning');
            return;
        }

        const targetName =
            patients.find((p) => p.id === targetId)?.full_name || `#${targetId}`;

        const result = await Swal.fire({
            title: 'Transfer Consultation?',
            html: `
        <div style="text-align:left">
          <p>This will move consultation <b>#${consultation.id}</b>:</p>
          <ul style="margin-left:1rem; list-style:disc;">
            <li>From: <b>${consultation.patient.full_name}</b></li>
            <li>To: <b>${targetName}</b></li>
          </ul>
          <p style="margin-top:.5rem;">Dispensations and any linked triage will be re-assigned.</p>
        </div>
      `,
            icon: 'warning',
            showCancelButton: true,
            confirmButtonText: 'Yes, transfer',
            cancelButtonText: 'Cancel',
            confirmButtonColor: '#2563eb',
            focusCancel: true,
        });

        if (!result.isConfirmed) return;

        setSubmitting(true);
        router.post(
            route('consultations.transfer', consultation.id),
            { to_patient_id: targetId },
            {
                onFinish: () => setSubmitting(false),
                onSuccess: () =>
                    Swal.fire('Transferred', 'Consultation moved successfully.', 'success'),
                onError: () =>
                    Swal.fire('Error', 'Failed to transfer consultation.', 'error'),
            }
        );
    };

    const list = useMemo(() => patients, [patients]);

    return (
        <AppLayout
            breadcrumbs={[
                { title: 'Consultations', href: route('consultations.index') },
                {
                    title: `Consultation #${consultation.id}`,
                    href: route('consultations.show', consultation.id),
                },
                { title: 'Transfer', href: '#' },
            ]}
        >
            <Head title={`Transfer Consultation #${consultation.id}`} />

            {/* Full-width container inside layout */}
            <div className="w-full px-4 py-8 sm:px-6">
                <div className="mb-6">
                    <h1 className="text-2xl font-bold">Transfer Consultation</h1>
                    <p className="text-sm text-muted-foreground">
                        You are transferring consultation{' '}
                        <b>#{consultation.id}</b> from{' '}
                        <b>{consultation.patient.full_name}</b> to another patient record.
                    </p>
                </div>

                {/* Search */}
                <div className="mb-4 flex gap-2">
                    <div className="relative w-full">
                        <Search className="text-muted-foreground absolute left-3 top-2.5 h-5 w-5" />
                        <Input
                            value={query ?? ''}
                            onChange={(e) => setQuery(e.target.value)}
                            onKeyDown={(e) => e.key === 'Enter' && onSearch()}
                            placeholder="Search patients by name, internal ID, or national ID number…"
                            className="pl-10"
                        />
                    </div>
                    <Button onClick={onSearch}>Search</Button>
                </div>

                {/* Results */}
                <div className="overflow-x-auto rounded-lg border">
                    <table className="min-w-full text-sm">
                        <thead className="bg-muted/50">
                            <tr>
                                <th className="px-3 py-2 text-left">Select</th>
                                <th className="px-3 py-2 text-left">Patient</th>
                                <th className="px-3 py-2 text-left">Type</th>
                                <th className="px-3 py-2 text-left">Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            {list.length === 0 ? (
                                <tr>
                                    <td
                                        colSpan={4}
                                        className="px-3 py-4 text-center text-muted-foreground"
                                    >
                                        No patients found. Try another search.
                                    </td>
                                </tr>
                            ) : (
                                list.map((p) => {
                                    const isCurrent = Number(p.id) === Number(consultation.patient.id);
                                    return (
                                        <tr key={p.id} className="border-t">
                                            <td className="px-3 py-2 align-middle">
                                                <input
                                                    type="radio"
                                                    name="to_patient_id"
                                                    value={p.id}
                                                    onChange={() => setTargetId(p.id)}
                                                    checked={targetId === p.id}
                                                    disabled={isCurrent}
                                                />
                                            </td>
                                            <td className="px-3 py-2 align-middle">
                                                <div className="flex items-center gap-2">
                                                    <span>
                                                        {p.full_name}{' '}
                                                        <span className="text-muted-foreground">(# {p.id})</span>
                                                    </span>
                                                    {isCurrent && (
                                                        <Badge variant="secondary">Current</Badge>
                                                    )}
                                                </div>
                                            </td>
                                            <td className="px-3 py-2 align-middle">
                                                {p.is_dependent ? (
                                                    <span className="inline-flex items-center gap-1">
                                                        <Users className="h-4 w-4" />
                                                        Dependent
                                                    </span>
                                                ) : (
                                                    <span className="inline-flex items-center gap-1">
                                                        <User className="h-4 w-4" />
                                                        Account Holder
                                                    </span>
                                                )}
                                            </td>
                                            <td className="px-3 py-2 align-middle">
                                                {/* Styled "View Profile" button link */}
                                                <Button
                                                    variant="outline"
                                                    size="sm"
                                                    className="gap-2"
                                                    onClick={() => router.visit(route('patients.show', p.id))}
                                                >
                                                    <Eye className="h-4 w-4" />
                                                    View Profile
                                                </Button>
                                            </td>
                                        </tr>
                                    );
                                })
                            )}
                        </tbody>
                    </table>
                </div>

                {/* Actions */}
                <div className="mt-6 flex items-center justify-end gap-2">
                    <Button
                        variant="outline"
                        onClick={() => router.visit(route('consultations.show', consultation.id))}
                    >
                        Cancel
                    </Button>
                    <Button onClick={submitTransfer} disabled={submitting}>
                        {submitting ? 'Transferring…' : 'Transfer Consultation'}
                    </Button>
                </div>
            </div>
        </AppLayout>
    );
}
