import { API } from '@/config';
import AppLayout from '@/layouts/app-layout';
import { Button } from '@/components/ui/button';
import { BreadcrumbItem } from '@/types';
import { PageProps } from '@inertiajs/core';
import { Head, Link, router, usePage } from '@inertiajs/react';
import { Eye, Pencil, Search, Trash } from 'lucide-react';
import moment from 'moment';
import { useState } from 'react';
import ReactPaginate from 'react-paginate';
import Swal from 'sweetalert2';
import React, { useRef } from 'react';
import { useReactToPrint } from 'react-to-print';
import { QRCodeSVG } from 'qrcode.react';

const logoUrl = 'https://providencehumancapital.com/headoffice/assets/images/PHC_Logo.png';

const SignatureUrl = 'http://127.0.0.1:8001';

interface PagePropsWithConsultations extends PageProps {
    consultations: {
        data: any;
        total: number;
        per_page: number;
        current_page: number;
        last_page: number;
    };
    patient: {
        id: number;
        first_name: string;
        surname: string;
    };
    clinics: Array<{ id: number; name: string }>;
    doctors: Array<{ id: number; name: string }>;
    filters: {
        search?: string;
        start_date?: string;
        end_date?: string;
        clinic_id?: number;
        doctor_id?: number;
    };
}

export default function PatientConsultations() {
    const { consultations, patient, clinics, doctors, filters } = usePage<PagePropsWithConsultations>().props;
    const [searchQuery, setSearchQuery] = useState(filters.search || '');
    const [startDate, setStartDate] = useState(filters.start_date || '');
    const [endDate, setEndDate] = useState(filters.end_date || '');
    const [clinicId, setClinicId] = useState(filters.clinic_id || '');
    const [doctorId, setDoctorId] = useState(filters.doctor_id || '');

    const breadcrumbs: BreadcrumbItem[] = [
        { title: 'Patients', href: `${API}/patients` },
        { title: `${patient.first_name} ${patient.surname}`, href: `${API}/patients/${patient.id}` },
        { title: 'Consultations', href: `${API}/patients/${patient.id}/consultations` },
    ];



    const contentRef = useRef<HTMLDivElement>(null);
    const reactToPrintFn = useReactToPrint({ contentRef });



    const handleFilter = () => {
        router.get(`${API}/patients/${patient.id}/consultations`, {
            search: searchQuery,
            start_date: startDate,
            end_date: endDate,
            clinic_id: clinicId,
            doctor_id: doctorId,
        }, { preserveState: true });
    };

    const handleResetFilters = () => {
        setSearchQuery('');
        setStartDate('');
        setEndDate('');
        setClinicId('');
        setDoctorId('');
        router.get(`${API}/patients/${patient.id}/consultations`, {}, { preserveState: true });
    };

    const handleDelete = (id: number) => {
        Swal.fire({
            title: 'Are you sure?',
            text: 'This consultation will be permanently deleted.',
            icon: 'warning',
            showCancelButton: true,
            confirmButtonColor: '#e3342f',
            cancelButtonColor: '#6c757d',
            confirmButtonText: 'Yes, delete it!',
            width: '350px',
        }).then((result) => {
            if (result.isConfirmed) {
                router.delete(route('consultations.destroy', id), {
                    onSuccess: () => {
                        Swal.fire({
                            title: 'Deleted!',
                            text: 'The consultation has been deleted.',
                            icon: 'success',
                            timer: 1500,
                            showConfirmButton: false,
                        });
                    },
                    onError: () => {
                        Swal.fire({
                            title: 'Error!',
                            text: 'Something went wrong.',
                            icon: 'error',
                        });
                    },
                });
            }
        });
    };

    const handlePageChange = (selected: { selected: number }) => {
        router.get(`${API}/patients/${patient.id}/consultations`, {
            page: selected.selected + 1,
            search: searchQuery,
            start_date: startDate,
            end_date: endDate,
            clinic_id: clinicId,
            doctor_id: doctorId,
        }, { preserveState: true });
    };


    const handlePrintSickLeave = (consultationId: number) => {
        reactToPrintFn();
    }

    return (
        <AppLayout breadcrumbs={breadcrumbs}>
            <Head title={`${patient.first_name}'s Consultations`} />
            <div className="bg-background text-foreground mx-8 my-6 flex flex-1 flex-col gap-4 rounded-xl p-6 shadow-sm">
                <div className="mb-4 flex flex-wrap items-center justify-between gap-4">
                    <h1 className="text-2xl font-bold">
                        {patient.first_name} {patient.surname}'s Consultations
                    </h1>
                </div>

                {/* Filter Section */}
                <div className="mb-6 rounded-lg border p-4">
                    <div className="grid grid-cols-1 gap-4 md:grid-cols-2 lg:grid-cols-4">
                        {/* Search */}
                        <div>
                            <label className="mb-1 block text-sm font-medium">Search</label>
                            <div className="relative">
                                <Search className="text-muted-foreground absolute top-2.5 left-3 h-5 w-5" />
                                <input
                                    type="text"
                                    value={searchQuery}
                                    onChange={(e) => setSearchQuery(e.target.value)}
                                    placeholder="Search..."
                                    className="border-border bg-background placeholder-muted-foreground w-full rounded-lg border py-2 pr-3 pl-10 text-sm focus:ring-2 focus:ring-blue-600 focus:outline-none"
                                />
                            </div>
                        </div>

                        {/* Date Range */}
                        <div>
                            <label className="mb-1 block text-sm font-medium">Date Range</label>
                            <div className="grid grid-cols-2 gap-2">
                                <input
                                    type="date"
                                    value={startDate}
                                    onChange={(e) => setStartDate(e.target.value)}
                                    className="border-border bg-background rounded-lg border p-2 text-sm"
                                />
                                <input
                                    type="date"
                                    value={endDate}
                                    onChange={(e) => setEndDate(e.target.value)}
                                    className="border-border bg-background rounded-lg border p-2 text-sm"
                                />
                            </div>
                        </div>

                        {/* Clinic Dropdown */}
                        <div>
                            <label className="mb-1 block text-sm font-medium">Clinic</label>
                            <select
                                value={clinicId}
                                onChange={(e) => setClinicId(e.target.value)}
                                className="border-border bg-background w-full rounded-lg border p-2 text-sm"
                            >
                                <option value="">All Clinics</option>
                                {clinics.map((clinic) => (
                                    <option key={clinic.id} value={clinic.id}>
                                        {clinic.name}
                                    </option>
                                ))}
                            </select>
                        </div>

                        {/* Doctor Dropdown */}
                        <div>
                            <label className="mb-1 block text-sm font-medium">Doctor</label>
                            <select
                                value={doctorId}
                                onChange={(e) => setDoctorId(e.target.value)}
                                className="border-border bg-background w-full rounded-lg border p-2 text-sm"
                            >
                                <option value="">All Doctors</option>
                                {doctors.map((doctor) => (
                                    <option key={doctor.id} value={doctor.id}>
                                        {doctor.name}
                                    </option>
                                ))}
                            </select>
                        </div>
                    </div>

                    <div className="mt-4 flex justify-end gap-2">
                        <button
                            onClick={handleResetFilters}
                            className="rounded-lg bg-gray-200 px-4 py-2 text-sm font-medium text-gray-700 hover:bg-gray-300"
                        >
                            Reset Filters
                        </button>
                        <button
                            onClick={handleFilter}
                            className="rounded-lg bg-blue-600 px-4 py-2 text-sm font-medium text-white hover:bg-blue-700"
                        >
                            Apply Filters
                        </button>
                    </div>
                </div>

                {/* Table */}
                <div className="overflow-x-auto">
                    <table className="min-w-full divide-y border text-sm">
                        <thead className="bg-muted">
                            <tr>
                                {['Date', 'Clinic', 'Doctor', 'Injury On Duty', 'Diagnosis', 'Treatment', 'Actions'].map((header) => (
                                    <th
                                        key={header}
                                        className="text-muted-foreground px-4 py-3 text-left text-xs font-semibold tracking-wider uppercase"
                                    >
                                        {header}
                                    </th>
                                ))}
                            </tr>
                        </thead>
                        <tbody className="divide-y">
                            {consultations.data.map((consultation: any) => (
                                <tr key={consultation.id} className="hover:bg-muted/50 transition">

                                    <div style={{ display: 'none' }}>
                                        <div ref={contentRef} className="p-6 font-sans text-sm text-gray-800">
                                            <style>
                                                {`
                                                    @page {
                                                        size: auto;
                                                        margin: 15mm 10mm;
                                                    }
                                                    @media print {
                                                        body { 
                                                            -webkit-print-color-adjust: exact;
                                                            print-color-adjust: exact;
                                                        }
                                                        .section-break {
                                                            page-break-after: always;
                                                        }
                                                        .avoid-break {
                                                            page-break-inside: avoid;
                                                        }
                                                    }
                                                `}
                                            </style>
                                            <div className="flex items-center justify-between border-b-2 border-gray-200 pb-4">
                                                <div className="flex items-center">
                                                    <img src={logoUrl} alt="PHC Logo" style={{ height: '4rem' }} />
                                                </div>
                                                <div className="text-center text-sm text-gray-800">
                                                    <p className="text-lg font-bold">Patient's Sick Leave Note</p>
                                                    <p>{moment(consultation?.consultation_date).format('MMMM Do YYYY, h:mm a')}</p>
                                                </div>
                                                <div className="flex items-center">
                                                    <QRCodeSVG value={`Attendance Date: ${consultation.patient?.id}`} size={60} />
                                                </div>
                                            </div>

                                            <div className="mt-8 flex justify-between">
                                                <div className="flex-1"></div>
                                                <div className="flex-1 text-right">
                                                    <h2 className="text-lg font-bold">Clinic Information</h2>
                                                    <p>{consultation?.clinic?.name}</p>
                                                    <p>Clinic Address: {consultation?.clinic?.address}</p>
                                                    <p>Contact Person: {consultation?.clinic?.contact_person}</p>
                                                    <p>Email Address: {consultation?.clinic?.email_address}</p>
                                                </div>
                                            </div>


                                            <div className="mt-4">
                                                <h2 className="text-lg font-bold">Patient Information</h2>
                                                <p>
                                                    Patient Name: {consultation?.patient?.first_name} {consultation?.patient?.surname}
                                                </p>
                                                <p>Patient ID: {consultation?.patient?.id}</p>
                                                <p>ID Number: {consultation?.patient.id_number}</p>
                                            </div>

                                            <div className="mt-4">
                                                <p>
                                                    This is to certify that <span style={{
                                                        fontWeight: 'bold', color: '#000080'
                                                    }}>{consultation?.patient?.first_name} {consultation?.patient?.surname},</span>
                                                    identified by National ID{" "} <span style={{
                                                        fontWeight: 'bold', color: '#000080'
                                                    }}>{consultation?.patient?.id_number}</span>, has been granted sick leave of {consultation?.number_of_days || 'N/A'} days
                                                    starting from <span style={{
                                                        fontWeight: 'bold', color: '#000080'
                                                    }}>{moment(consultation?.consultation_date).format('MMMM Do YYYY')}.</span>
                                                    The patient was seen by our medical team and, based on the examination, is advised to
                                                    refrain from work or strenuous activity for health recovery purposes.
                                                </p>

                                                <p className="mt-3">
                                                    <strong>Reason for Leave:</strong> {consultation?.reason_for_leave || "N/A"}.
                                                </p>

                                                <p>
                                                    <strong>Additional Notes:</strong> {consultation?.sick_leave_notes || "No additional notes provided."}
                                                </p>
                                            </div>


                                            <div className="mt-4" style={{
                                                marginTop: '6rem',
                                            }}>
                                                <h2 className="text-lg font-bold">Consultor's Information</h2>
                                                <p>Doctor's / RGN's Name:  {consultation?.doctor?.name} </p>
                                                <p>Doctor's Signature</p>
                                                <img
                                                    src={consultation?.doctor?.signature
                                                        ? `${SignatureUrl}/storage/${consultation.doctor.signature}`
                                                        : "/placeholder-signature.png"}
                                                    alt="Doctor Signature"
                                                    style={{ height: '4rem' }}
                                                />


                                            </div>

                                            <div
                                                className="mt-auto pt-8"
                                                style={{
                                                    marginTop: '7rem',
                                                }}
                                            >
                                                <div className="flex justify-between">
                                                    <div>
                                                        <p>Date: {moment().format('MMMM Do YYYY, h:mm A')}</p>
                                                    </div>
                                                    <div className="h-24 w-34 border-2 border-dashed border-gray-300"></div>
                                                </div>
                                            </div>

                                        </div>
                                    </div>

                                    {/* End Print Sick Leave Section */}
                                    <td className="px-4 py-3">{moment(consultation.consultation_date).format('MMMM Do YYYY')}</td>
                                    <td className="px-4 py-3">{consultation.clinic?.name}</td>
                                    <td className="px-4 py-3">{consultation.doctor?.name}</td>
                                    <td className="px-4 py-3">
                                        <span
                                            className={`inline-flex items-center rounded-full px-2.5 py-0.5 text-xs font-medium ${consultation.injury_on_duty ? 'bg-green-100 text-green-800' : 'bg-red-100 text-red-800'}`}
                                        >
                                            {consultation.injury_on_duty ? 'Yes' : 'No'}
                                        </span>
                                    </td>
                                    <td className="px-4 py-3">{consultation.diagnosis}</td>
                                    <td className="px-4 py-3">{consultation.treatment_plan}</td>
                                    <td className="flex items-center gap-2 px-4 py-3">
                                        <Link
                                            href={route('consultations.show', consultation.id)}
                                            className="inline-flex items-center justify-center rounded-full bg-blue-600 p-2 text-white hover:bg-blue-700"
                                            title="View"
                                        >
                                            <Eye className="h-4 w-4" />
                                        </Link>
                                        <Link
                                            href={route('consultations.edit', consultation.id)}
                                            className="inline-flex items-center justify-center rounded-full bg-yellow-500 p-2 text-white hover:bg-yellow-600"
                                            title="Edit"
                                        >
                                            <Pencil className="h-4 w-4" />
                                        </Link>
                                        <button
                                            onClick={() => handleDelete(consultation.id)}
                                            className="inline-flex items-center justify-center rounded-full bg-red-600 p-2 text-white hover:bg-red-700"
                                            title="Delete"
                                        >
                                            <Trash className="h-4 w-4" />
                                        </button>
                                        {consultation.sick_leave ? (
                                            <Button
                                                variant="outline"
                                                onClick={() => handlePrintSickLeave(consultation.id)}
                                                className="inline-flex items-center justify-center  p-2 text-white-700 hover:text-gray-900"
                                                title="Print Sick Leave"
                                                style={{
                                                    cursor: 'pointer',
                                                }}
                                            >
                                                Print Sick Leave
                                            </Button>
                                        ) : (
                                            <Button
                                                variant="outline"
                                                disabled
                                                className="inline-flex items-center justify-center p-2 text-white-400 cursor-not-allowed"
                                                title="No Sick Leave Available"
                                                style={{
                                                    cursor: 'not-allowed',
                                                }}
                                            >
                                                No Sick Leave
                                            </Button>
                                        )}
                                    </td>
                                </tr>
                            ))}
                        </tbody>
                    </table>
                </div>

                {/* Pagination */}
                <div className="mt-6 flex justify-center">
                    <ReactPaginate
                        pageCount={consultations.last_page}
                        forcePage={consultations.current_page - 1}
                        onPageChange={handlePageChange}
                        marginPagesDisplayed={1}
                        pageRangeDisplayed={3}
                        previousLabel="← Prev"
                        nextLabel="Next →"
                        breakLabel="..."
                        containerClassName="flex items-center gap-2 text-sm"
                        pageClassName="px-3 py-1 border border-border rounded hover:bg-muted"
                        activeClassName="bg-blue-600 text-white"
                        previousClassName="px-3 py-1 border border-border rounded hover:bg-muted"
                        nextClassName="px-3 py-1 border border-border rounded hover:bg-muted"
                        breakClassName="px-2"
                    />
                </div>
            </div>
        </AppLayout>
    );
}