import AppLayout from '@/layouts/app-layout';
import { Head, Link } from '@inertiajs/react';
import {
  Lock,
  ShieldAlert,
  Info,
  Users2,
  Building2,
  Stethoscope,
  ArrowLeft,
  Mail,
} from 'lucide-react';
import { Button } from '@/components/ui/button';
import { Card, CardHeader, CardTitle, CardDescription, CardContent, CardFooter } from '@/components/ui/card';
import { Alert, AlertTitle, AlertDescription } from '@/components/ui/alert';
import { Separator } from '@/components/ui/separator';
import { Badge } from '@/components/ui/badge';

interface Props {
  title?: string;
  message?: string;
  /** Optional: show a help email / link if you have one */
  supportEmail?: string; // e.g. "support@providencehealth.co.zw"
}

export default function PrivateFile({
  title = 'Access Restricted',
  message = 'You cannot access this file.',
  supportEmail = 'support@providencehealth.co.zw',
}: Props) {
  return (
    <AppLayout breadcrumbs={[{ title: 'Consultations', href: route('consultations.index') }]}>
      <Head title={title} />

         <div className="text-foreground mx-8 my-6 flex flex-1 flex-col gap-4 rounded-xl p-6 ">
        <Card className="border-destructive/20">
          <CardHeader className="flex items-start gap-3">
            <div className="flex h-12 w-12 shrink-0 items-center justify-center rounded-full bg-red-100 text-red-600 dark:bg-red-900/20 dark:text-red-300">
              <Lock className="h-6 w-6" />
            </div>
            <div className="space-y-1">
              <CardTitle className="text-xl">{title}</CardTitle>
              <CardDescription className="text-sm">{message}</CardDescription>
              <div className="mt-2 flex flex-wrap items-center gap-2">
                <Badge variant="destructive" className="gap-1">
                  <ShieldAlert className="h-3.5 w-3.5" />
                  Private File
                </Badge>
                <Badge variant="outline" className="gap-1">
                  <Building2 className="h-3.5 w-3.5" />
                  Company-protected
                </Badge>
              </div>
            </div>
          </CardHeader>

          <Separator />

          <CardContent className="space-y-5 pt-6">
            {/* Why am I seeing this? */}
            <Alert variant="destructive" className="border-destructive/30">
              <ShieldAlert className="h-5 w-5" />
              <AlertTitle>Why you’re seeing this</AlertTitle>
              <AlertDescription className="mt-1 text-sm leading-relaxed">
                This consultation is restricted by <b>company file access</b> and can only be viewed by:
                <ul className="ml-5 mt-2 list-disc space-y-1">
                  <li>The <b>creator</b> of the consultation</li>
                  <li>Users who have been <b>granted access</b> to the patient’s company file</li>
                  <li>A user with the <b>Doctor</b> role (bypass access)</li>
                </ul>
              </AlertDescription>
            </Alert>

            {/* Possible reasons */}
            <div className="rounded-lg border p-4">
              <div className="mb-2 flex items-center gap-2">
                <Info className="h-4 w-4 text-blue-600" />
                <p className="font-medium">Common reasons</p>
              </div>
              <ul className="ml-5 list-disc text-sm leading-relaxed">
                <li>You’re not the consultation creator.</li>
                <li>You don’t have access to the <b>company</b> this patient belongs to.</li>
                <li>Your role doesn’t include the required permission.</li>
                <li>You’re signed in to the wrong account.</li>
              </ul>
            </div>

            {/* What can I do? */}
            <div className="rounded-lg border p-4">
              <div className="mb-2 flex items-center gap-2">
                <Users2 className="h-4 w-4 text-emerald-600" />
                <p className="font-medium">What you can do</p>
              </div>
              <ul className="ml-5 list-disc text-sm leading-relaxed">
                <li>
                  Ask an administrator to grant you <b>Company File Access</b> for this patient’s company.
                </li>
                <li>
                  If you are a clinician and require urgent access, contact a user with the <b>Doctor</b> role or your admin.
                </li>
                <li>
                  Verify you’re signed in with the correct account and try again.
                </li>
              </ul>
            </div>
          </CardContent>

          <Separator />

          <CardFooter className="flex flex-wrap items-center justify-between gap-3">
            <div className="flex items-center gap-2 text-xs text-muted-foreground">
              <Stethoscope className="h-4 w-4" />
              <span>Privacy controls protect patient and company data.</span>
            </div>

            <div className="flex flex-wrap gap-2">
              <Button asChild variant="outline">
                <Link href={route('consultations.index')} className="inline-flex items-center gap-2">
                  <ArrowLeft className="h-4 w-4" />
                  Back to Consultations
                </Link>
              </Button>

              <Button asChild>
                <a
                  href={`mailto:${encodeURIComponent(supportEmail)}?subject=Access%20Request%20-%20Private%20File`}
                  className="inline-flex items-center gap-2"
                >
                  <Mail className="h-4 w-4" />
                  Request Access
                </a>
              </Button>
            </div>
          </CardFooter>
        </Card>
      </div>
    </AppLayout>
  );
}
