// resources/js/Components/PrintReferralNote.tsx
import { Button } from '@/components/ui/button';
import React, { useRef } from 'react';
import { useReactToPrint } from 'react-to-print';
import moment from 'moment';
import { QRCodeSVG } from 'qrcode.react';

const SignatureUrl = 'http://127.0.0.1:8001';
const SignatureUrlDep = 'http://providencehealth.online/public';

interface Consultation {
  id: string;
  patient_id: string;
  doctor_id: string;
  clinic_id: string;
  consultation_date: string;
  injury_on_duty: boolean;
  systolic: string;
  diastolic: string;
  temperature: string;
  heart_rate: string;
  respiratory_rate: string;
  oxygen_saturation: string;
  weight: string;
  height: string;
  bmi: string;
  blood_sugar: string;
  urine_test_result: string;
  hiv_status: string;
  pregnancy_test_result: string;
  presenting_complaint: string;
  clinical_notes: string;
  diagnosis: string;
  treatment_plan: string;
  prescription: string;
  follow_up_required: boolean;

  // Referral fields
  refer_for_radiology: boolean;
  refer_for_lab: boolean;
  refer_to_specialist: boolean;
  specialist_name: string;
  refer_to_casualty: boolean;
  refer_out_patient: boolean;

  // NEW: hospital the patient is referred to
  referred_hospital?: string;

  follow_up_date: string;
  referral_notes: string;
  created_at: string;

  patient: {
    id: number;
    full_name: string;
    first_name: string;
    surname: string;
    employee_number: string;
    home_address: string;
    date_of_birth: string;
    email: string;
    gender: string;
    company: string;
    id_number: string;
  };
  doctor: {
    name: string;
    email: string;
    signature: string | null;
    qualifications: string;
  };
  clinic: {
    name: string;
    address: string;
    contact_person: string;
    email_address: string;
    city: string;
  };
}

interface Props {
  consultation: Consultation;
  logoUrl?: string;

  // Button controls
  buttonLabel?: string;
  buttonVariant?: 'default' | 'destructive' | 'outline' | 'secondary' | 'ghost' | 'link';
  buttonSize?: 'sm' | 'default' | 'lg' | 'icon';
  buttonClassName?: string;
}

export default function PrintReferralNote({
  consultation,
  logoUrl = 'https://providence-human-capital.github.io/images/PHC_Logo_Health.png',
  buttonLabel = 'Print Referral Note',
  buttonVariant = 'outline',
  buttonSize = 'sm',
  buttonClassName = '',
}: Props) {
  const contentRef = useRef<HTMLDivElement>(null);
  const reactToPrintFn = useReactToPrint({ contentRef });

  const patientName =
    consultation?.patient?.full_name ||
    `${consultation?.patient?.first_name ?? ''} ${consultation?.patient?.surname ?? ''}`.trim();

  const referralTargets: string[] = [
    consultation.refer_for_radiology ? 'Radiology / Imaging' : '',
    consultation.refer_for_lab ? 'Laboratory Tests' : '',
    consultation.refer_to_specialist
      ? `Specialist${consultation.specialist_name ? `: ${consultation.specialist_name}` : ''}`
      : '',
    consultation.refer_to_casualty ? 'Casualty / Emergency' : '',
    consultation.refer_out_patient ? 'OPD / Out-Patient Department' : '',
  ].filter(Boolean);

  const referredHospital = (consultation.referred_hospital ?? '').trim();

  const qrValue = `Referral
Consultation: ${consultation?.id}
Patient: ${patientName} (#${consultation?.patient?.id ?? ''})
Date: ${moment(consultation.created_at).format('YYYY-MM-DD HH:mm')}
To: ${referralTargets.join(', ') || 'N/A'}
Hospital: ${referredHospital || 'N/A'}`;

  const signaturePath = consultation?.doctor?.signature || '';
  const primarySigSrc = signaturePath ? `${SignatureUrl}/storage/${signaturePath}` : '';
  const fallbackSigSrc = signaturePath ? `${SignatureUrlDep}/storage/${signaturePath}` : '';

  return (
    <>
      {/* Trigger Button */}
      <Button
        variant={buttonVariant}
        size={buttonSize}
        className={`inline-flex items-center ${buttonClassName}`}
        onClick={reactToPrintFn}
        style={{ cursor: 'pointer' }}
      >
        {buttonLabel}
      </Button>

      {/* Hidden printable content */}
      <div style={{ display: 'none' }}>
        <div ref={contentRef} className="print-page p-6 font-sans text-sm text-gray-800">
          <style>
            {`
              @page {
                size: A4;
                margin: 15mm 10mm; /* Top/Bottom 15mm, Left/Right 10mm */
              }
              @media print {
                body { -webkit-print-color-adjust: exact; print-color-adjust: exact; }
                .section-break { page-break-after: always; }
                .avoid-break { page-break-inside: avoid; }

                /* Grid to pin footer to bottom of printable area */
                .print-page {
                  display: grid;
                  grid-template-rows: auto 1fr auto; /* header / main / footer */
                  height: calc(297mm - 30mm); /* A4 height minus vertical margins */
                }
              }
              @media screen {
                .print-page {
                  display: grid;
                  grid-template-rows: auto 1fr auto;
                  min-height: 100vh;
                }
              }
            `}
          </style>

          {/* HEADER */}
          <header>
            <div className="flex items-center justify-between border-b-2 border-gray-200 pb-4">
              <div className="flex items-center">
                <img src={logoUrl} alt="PHC Logo" style={{ height: '4rem' }} />
              </div>
              <div className="text-center text-sm text-gray-800">
                <p className="text-lg font-bold">Patient Referral Note</p>
                <p>{moment(consultation.created_at).format('MMMM Do YYYY, h:mm a')}</p>
              </div>
              <div className="flex items-center">
                <QRCodeSVG value={qrValue} size={60} />
              </div>
            </div>
          </header>

          {/* MAIN */}
          <main className="pt-4">
            {/* Clinic Info */}
            <div
              className="mt-0 flex justify-between"
              style={{
                fontSize: '11px',
              }}
            >
              <div className="flex-1"></div>
              <div className="flex-1 text-right">
                <h2 className="text-lg font-bold">Clinic Information</h2>
                <p>{consultation.clinic?.name}</p>
                <p>Clinic Address: {consultation.clinic?.address}</p>
                <p>Contact Person: {consultation.clinic?.contact_person}</p>
                <p>Email Address: {consultation.clinic?.email_address}</p>
              </div>
            </div>

            {/* Patient Info */}
            <div
              className="mt-4"
              style={{
                fontSize: '11px',
              }}
            >
              <h2 className="text-lg font-bold">Patient Information</h2>
              <p>Patient Name: {patientName}</p>
              <p>Employee #: {consultation.patient.employee_number || ' - '}</p>
              <p>Company: {consultation.patient?.company || ' - '}</p>
              <p>ID Number: {consultation.patient?.id_number}</p>
            </div>

            {/* Referral Destination */}
            <div className="mt-4">
              <h2 className="text-lg font-bold">Referral Destination</h2>
              <p>
                <strong>Service(s):</strong>{' '}
                <span style={{ fontWeight: 'bold', color: '#000080' }}>
                  {referralTargets.length ? referralTargets.join(', ') : 'N/A'}
                </span>
              </p>
              <p className="mt-1">
                <strong>Referred Hospital/Facility:</strong>{' '}
                <span style={{ fontWeight: 'bold', color: '#000080' }}>{referredHospital || 'N/A'}</span>
              </p>
              {consultation.refer_to_specialist && (
                <p className="mt-1">
                  <strong>Specialist:</strong>{' '}
                  <span style={{ fontWeight: 'bold', color: '#000080' }}>
                    {consultation.specialist_name || 'N/A'}
                  </span>
                </p>
              )}
            </div>

            {/* Referral Body */}
            <div className="mt-4">
              <p>
                This is to refer{' '}
                <span style={{ fontWeight: 'bold', color: '#000080' }}>{patientName}</span> for further assessment and
                management as indicated above.
              </p>

              {consultation.diagnosis ? (
                <p className="mt-3">
                  <strong>Provisional Diagnosis:</strong> {consultation.diagnosis}
                </p>
              ) : null}

              {consultation.presenting_complaint ? (
                <p className="mt-1">
                  <strong>Presenting Complaint:</strong> {consultation.presenting_complaint}
                </p>
              ) : null}

              {consultation.clinical_notes ? (
                <p className="mt-1">
                  <strong>Clinical Summary:</strong> {consultation.clinical_notes}
                </p>
              ) : null}

              <p className="mt-3">
                <strong>Referral Notes / Instructions:</strong> {consultation.referral_notes ?? 'No specific notes provided.'}
              </p>

              {consultation.follow_up_required || consultation.follow_up_date ? (
                <p className="mt-3">
                  <strong>Follow-up:</strong>{' '}
                  {consultation.follow_up_required
                    ? `Required on ${
                        consultation.follow_up_date
                          ? moment(consultation.follow_up_date).format('MMMM Do YYYY')
                          : 'a scheduled date'
                      }.`
                    : 'Not required.'}
                </p>
              ) : null}
            </div>

            {/* NB (creation time) */}
            <div className="mt-8 text-xs text-gray-600 italic">
              NB: This referral note was created on {moment(consultation.created_at).format('MMMM Do YYYY, h:mm A')}.
            </div>

            {/* Referring Clinician */}
            <div className="mt-6">
              <h2 className="text-lg font-bold">Referring Clinician</h2>
              <p>Name: {consultation.doctor?.name}</p>
              {consultation.doctor?.qualifications ? <p>Qualifications: {consultation.doctor?.qualifications}</p> : null}
              <p>Signature:</p>

              {signaturePath ? (
                <img
                  src={primarySigSrc}
                  onError={(e) => {
                    const target = e.currentTarget as HTMLImageElement;
                    if (target.src !== fallbackSigSrc) target.src = fallbackSigSrc;
                  }}
                  alt="Doctor signature"
                  style={{ height: '1rem' }}
                />
              ) : (
                <p> - </p>
              )}
            </div>
          </main>

          {/* FOOTER — pinned to bottom */}
          <footer className="pt-8">
            <div className="flex justify-between">
              <div>{/* Spacer / extra info if needed */}</div>
              <div className="h-24 w-34 border-2 border-dashed border-gray-300"></div>
            </div>
          </footer>
        </div>
      </div>
    </>
  );
}
