import { Button } from '@/components/ui/button'; // Adjust path if needed
import React, { useRef } from 'react';
import { useReactToPrint } from 'react-to-print';
import moment from 'moment';
import { QRCodeSVG } from 'qrcode.react';

const SignatureUrl = 'http://127.0.0.1:8001';
const SignatureUrlDep = 'http://providencehealth.online/public';

interface Consultation {
  id: string;
  patient_id: string;
  doctor_id: string;
  clinic_id: string;
  consultation_date: string;
  injury_on_duty: boolean;
  systolic: string;
  diastolic: string;
  temperature: string;
  heart_rate: string;
  respiratory_rate: string;
  oxygen_saturation: string;
  weight: string;
  height: string;
  bmi: string;
  blood_sugar: string;
  urine_test_result: string;
  hiv_status: string;
  pregnancy_test_result: string;
  presenting_complaint: string;
  clinical_notes: string;
  diagnosis: string;
  treatment_plan: string;
  prescription: string;
  follow_up_required: boolean;
  created_at: string;

  // Leave-related
  sick_leave: boolean;
  restricted_duty: boolean; // <-- added
  reason_for_leave: string;
  sick_leave_notes: string;
  number_of_days: string;

  // Referrals
  refer_for_radiology: boolean;
  refer_for_lab: boolean;
  refer_to_specialist: boolean;
  specialist_name: string;
  refer_to_casualty: boolean;
  refer_out_patient: boolean;

  follow_up_date: string;
  referral_notes: string;

  patient: {
    id: number;
    full_name: string;
    first_name: string;
    surname: string;
    employee_number: string;
    home_address: string;
    date_of_birth: string;
    email: string;
    gender: string;
    id_number: string;
  };
  doctor: {
    name: string;
    email: string;
    signature: string;
    qualifications: string;
  };
  clinic: {
    name: string;
    address: string;
    contact_person: string;
    email_address: string;
    city: string;
  };
}

interface Props {
  consultation: Consultation;
  logoUrl?: string;
}

export default function PrintSickLeaveNote({
  consultation,
  logoUrl = 'https://providence-human-capital.github.io/images/PHC_Logo_Health.png',
}: Props) {
  const contentRef = useRef<HTMLDivElement>(null);
  const reactToPrintFn = useReactToPrint({ contentRef });

  // Sick leave period calculations (inclusive)
  const startDate = moment(consultation.created_at);
  const rawDays = Number(consultation?.number_of_days ?? 0);
  const totalDays = Number.isFinite(rawDays) ? Math.max(0, Math.floor(rawDays)) : 0;
  const endDate = startDate.clone().add(Math.max(totalDays - 1, 0), 'days');
  const hasDays = totalDays > 0;

  const patientName =
    (consultation?.patient?.full_name && consultation.patient.full_name.trim()) ||
    [consultation?.patient?.first_name, consultation?.patient?.surname].filter(Boolean).join(' ');

  const consultationDateTime = moment(consultation.created_at).format('MMMM Do YYYY, h:mm A');

  const Box = (label: string, checked = false) => (
    <span className="inline-flex items-center gap-1 mr-6">
      <span
        className="inline-block h-3.5 w-3.5 border border-gray-800 rounded-[2px] text-[10px] leading-3 text-center align-middle"
        style={{ fontWeight: 700 }}
      >
        {checked ? '✓' : ''}
      </span>
      <span>{label}</span>
    </span>
  );

  // Decide signature src (fallback domain if needed)
  const signaturePath = consultation?.doctor?.signature || '';
  const signatureSrc = signaturePath
    ? `${SignatureUrl}/storage/${signaturePath}`
    : `${SignatureUrlDep}/storage/${signaturePath}`;

  return (
    <>
      <Button variant="outline" className="w-full" onClick={reactToPrintFn} style={{ cursor: 'pointer' }}>
        Print Sick Leave Note
      </Button>

      {/* Hidden printable content */}
      <div style={{ display: 'none' }}>
        <div ref={contentRef} className="print-page p-6 font-sans text-sm text-gray-800">
          <style>
            {`
              @page {
                size: A4;
                margin: 15mm 10mm; /* top/bottom 15mm, left/right 10mm */
              }

              /* Make the page a 3-row grid so the footer sticks to the bottom of the printable area */
              @media print {
                html, body { height: auto; }
                .print-page {
                  display: grid;
                  grid-template-rows: auto 1fr auto; /* header / body / footer */
                  height: calc(297mm - 30mm); /* A4 height (297mm) minus top+bottom margins (15mm + 15mm) */
                }
                .section-break { page-break-after: always; }
                .avoid-break { page-break-inside: avoid; }
              }

              /* Optional: similar layout when previewing on screen */
              @media screen {
                .print-page {
                  display: grid;
                  grid-template-rows: auto 1fr auto;
                  min-height: 100vh;
                }
              }
            `}
          </style>

          {/* HEADER (row 1) */}
          <header>
            <div className="flex items-center justify-between border-b-2 border-gray-200 pb-4">
              <div className="flex items-center">
                <img src={logoUrl} alt="PHC Logo" style={{ height: '4rem' }} />
              </div>
              <div className="text-center text-sm text-gray-800">
                <p className="text-lg font-bold">Patient&apos;s Sick Leave Note</p>
                <p>{startDate.format('MMMM Do YYYY, h:mm a')}</p>
              </div>
              <div className="flex items-center">
                <QRCodeSVG
                  value={`Consultation ID: ${consultation?.id ?? ''} | Sick Leave: ${
                    hasDays ? `${startDate.format('YYYY-MM-DD')} to ${endDate.format('YYYY-MM-DD')}` : 'N/A'
                  }`}
                  size={60}
                />
              </div>
            </div>
          </header>

          {/* BODY (row 2) */}
          <main className="pt-4">
            <div className="mt-0 flex justify-between" style={{ fontSize: '11px' }}>
              <div className="flex-1"></div>
              <div className="flex-1 text-right">
                <h2 className="text-lg font-bold">Clinic Information</h2>
                <p>{consultation.clinic?.name}</p>
                <p>Clinic Address: {consultation.clinic?.address}</p>
                <p>Contact Person: {consultation.clinic?.contact_person}</p>
                <p>Email Address: {consultation.clinic?.email_address}</p>
              </div>
            </div>

            <div className="mt-4" style={{ fontSize: '11px' }}>
              <h2 className="text-lg font-bold">Patient Information</h2>
              <p>Patient Name: {patientName}</p>
              <p>ID Number: {consultation.patient?.id_number}</p>
            </div>

            {/* Attendance + Recommendation */}
            <div className="mt-4 avoid-break" style={{ fontSize: '14px' }}>
              <h2 className="text-lg font-bold">Attendance & Recommendation</h2>

              <p className="mt-2">
                <span style={{ fontWeight: 'bold', color: '#000080' }}>
                  {patientName || '______________________'}
                </span>{' '}
                was attended at our Facility on{' '}
                <span style={{ fontWeight: 'bold', color: '#000080' }}>{consultationDateTime}</span>. He or she has been
                granted:
              </p>

              {/* Only two options as requested: Sick Leave and Restricted Duty */}
              <div className="mt-2 flex flex-wrap items-center gap-y-2 text-[11px]">
                {Box('Sick Leave', Boolean(consultation.sick_leave))}
                {Box('Restricted Duty', Boolean(consultation.restricted_duty))}
              </div>

              <p className="mt-2">
                from{' '}
                <span style={{ fontWeight: 'bold', color: '#000080' }}>
                  {hasDays ? startDate.format('MMMM Do YYYY') : '________________'}
                </span>{' '}
                to{' '}
                <span style={{ fontWeight: 'bold', color: '#000080' }}>
                  {hasDays ? endDate.format('MMMM Do YYYY') : '________________'}
                </span>{' '}
                for the purposes of recovery.
              </p>
            </div>

            {/* Optional reason/notes */}
            {consultation.reason_for_leave && (
              <p className="mt-3" style={{ fontSize: '13px' }}>
                <strong>Reason for Leave:</strong> {consultation.reason_for_leave}.
              </p>
            )}
            {consultation.sick_leave_notes && (
              <p style={{ fontSize: '13px' }}>
                <strong>Additional Notes:</strong> {consultation.sick_leave_notes}
              </p>
            )}

            {/* Doctor info */}
            <div className="mt-6" style={{ fontSize: '11px' }}>
              <h2 className="text-lg font-bold">Doctor&apos;s Information</h2>
              <p>Doctor&apos;s / RGN&apos;s Name: {consultation.doctor?.name}</p>
              <p>Doctor&apos;s Signature</p>
              <img
                src={signatureSrc}
                alt="-"
                style={{ height: '1rem' }}
              />
            </div>
          </main>

          {/* FOOTER (row 3) — always at bottom of the page */}
          <footer className="pt-2">
            <div className="flex justify-between">
              <div>
                <p>Generated On: {moment().format('MMMM Do YYYY, h:mm A')}</p>
              </div>
              <div className="h-24 w-34 border-2 border-dashed border-gray-300"></div>
            </div>
          </footer>
        </div>
      </div>
    </>
  );
}
