import { ContactInfoSection, EmergencyContactSection, MedicalInfoSection, PersonalInfoSection, AddressInfoSection } from '@/components/PatientForm';
import { Button } from '@/components/ui/button';
import { Label } from '@/components/ui/label';
import AppLayout from '@/layouts/app-layout';
import { BreadcrumbItem } from '@/types';
import { Head, useForm, Link, usePage } from '@inertiajs/react';
import { useEffect, useMemo, useState } from 'react';
import Select from 'react-select';
import Swal from 'sweetalert2';

type ParentItem = {
  id: number | string;
  first_name: string;
  surname: string;
  company?: { id: number | string; name: string } | null;

  // new fields shipped from backend
  home_address?: string | null;
  work_area?: string | null;
  suburb?: string | null;
  phone?: string | null;
  emergency_contact_name?: string | null;
  emergency_contact_relation?: string | null;
  emergency_contact_phone?: string | null;
};

export default function DependentCreate() {
  const { parents, relationships, parent_id } = usePage<{
    parents: ParentItem[];
    relationships: string[];
    parent_id?: string;
  }>().props;

  const { data, setData, post, errors, processing } = useForm({
    first_name: '',
    middle_name: '',
    surname: '',
    parent_patient_id: parent_id || '',
    relationship: '',
    date_of_birth: '',
    email: '',
    gender: '',
    id_number: '',
    medical_aid_number: '',
    medical_aid_provider: '',
    phone: '',
    emergency_contact_name: '',
    emergency_contact_relation: '',
    emergency_contact_phone: '',
    allergies: '',
    is_smoker: false,
    occupation: '',

    home_address: '',
    work_area: '',
    suburb: '',

    marital_status: '',
  });

  const [isDarkMode, setIsDarkMode] = useState(false);
  const [selectedParent, setSelectedParent] = useState<any>(null);

  // Active parent (based on current form state, works with locked or user-chosen parent)
  const activeParent: ParentItem | undefined = useMemo(() => {
    if (!data.parent_patient_id) return undefined;
    return parents.find((p) => String(p.id) === String(data.parent_patient_id));
  }, [parents, data.parent_patient_id]);

  // Dark mode tracker
  useEffect(() => {
    const checkDarkMode = () => {
      setIsDarkMode(document.documentElement.classList.contains('dark'));
    };
    checkDarkMode();
    const observer = new MutationObserver(checkDarkMode);
    observer.observe(document.documentElement, { attributes: true, attributeFilter: ['class'] });
    return () => observer.disconnect();
  }, []);

  // Preselect parent in dropdown if coming from parent_id
  useEffect(() => {
    if (parent_id && parents.length > 0) {
      const parent = parents.find((p) => String(p.id) === String(parent_id));
      if (parent) {
        setSelectedParent({
          value: parent.id,
          label: `${parent.first_name} ${parent.surname}${parent.company?.name ? ' — ' + parent.company.name : ''}`,
        });
      }
    }
  }, [parent_id, parents]);

  const parentOptions = parents.map((parent) => ({
    value: parent.id,
    label: `${parent.first_name} ${parent.surname}${parent.company?.name ? ' — ' + parent.company.name : ''}`,
  }));

  const handleParentChange = (selectedOption: any) => {
    setSelectedParent(selectedOption);
    setData('parent_patient_id', selectedOption ? selectedOption.value : '');
  };

  const selectStyles = {
    control: (base: any) => ({
      ...base,
      backgroundColor: isDarkMode ? '#000' : '#fff',
      borderColor: isDarkMode ? '#374151' : '#d1d5db',
      color: isDarkMode ? '#fff' : '#000',
      '&:hover': { borderColor: isDarkMode ? '#4b5563' : '#9ca3af' },
    }),
    menu: (base: any) => ({ ...base, backgroundColor: isDarkMode ? '#000' : '#fff', color: isDarkMode ? '#fff' : '#000' }),
    option: (base: any, { isFocused }: any) => ({
      ...base,
      backgroundColor: isFocused ? (isDarkMode ? '#1e40af' : '#3b82f6') : 'transparent',
      color: isFocused ? '#fff' : isDarkMode ? '#fff' : '#000',
      '&:active': { backgroundColor: isDarkMode ? '#1e40af' : '#3b82f6' },
    }),
    singleValue: (base: any) => ({ ...base, color: isDarkMode ? '#fff' : '#000' }),
    input: (base: any) => ({ ...base, color: isDarkMode ? '#fff' : '#000' }),
    placeholder: (base: any) => ({ ...base, color: isDarkMode ? '#9ca3af' : '#6b7280' }),
  };

  // ---- Fill helpers ----
  const fillFromParentAddress = () => {
    if (!activeParent) return;
    setData('home_address', activeParent.home_address ?? '');
    setData('work_area', activeParent.work_area ?? '');
    setData('suburb', activeParent.suburb ?? '');
  };

  const fillParentAsEmergency = () => {
    if (!activeParent) return;
    const fullName = `${activeParent.first_name ?? ''} ${activeParent.surname ?? ''}`.trim();
    setData('emergency_contact_name', fullName || (activeParent.emergency_contact_name ?? ''));
    setData('emergency_contact_relation', 'Parent');
    setData('emergency_contact_phone', activeParent.phone ?? activeParent.emergency_contact_phone ?? '');
  };

  const fillParentPhone = () => {
    if (!activeParent) return;
    const parentPhone = activeParent.phone ?? activeParent.emergency_contact_phone ?? '';
    setData('phone', parentPhone);
  };

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();

    // ---- Client-side validation: Parent, Gender, Date of Birth ----
    const issues: string[] = [];

    // Parent required
    if (!String(data.parent_patient_id || '').trim()) {
      issues.push('Parent');
    }

    // Gender required
    if (!String(data.gender || '').trim()) {
      issues.push('Gender');
    }

    // Date of Birth required + must be a valid date not in the future
    const dobStr = String(data.date_of_birth || '').trim();
    if (!dobStr) {
      issues.push('Date of Birth');
    } else {
      const dob = new Date(dobStr);
      const isValid = !isNaN(dob.getTime());
      if (!isValid) {
        issues.push('Date of Birth (invalid date)');
      } else {
        const today = new Date();
        const dobYMD = new Date(dob.getFullYear(), dob.getMonth(), dob.getDate());
        const todayYMD = new Date(today.getFullYear(), today.getMonth(), today.getDate());
        if (dobYMD > todayYMD) {
          issues.push('Date of Birth (cannot be in the future)');
        }
      }
    }

    if (issues.length > 0) {
      Swal.fire({
        icon: 'error',
        title: 'Missing or invalid fields',
        html: `<div style="text-align:left">
                 <p>Please fix the following before saving:</p>
                 <ul style="margin-left:1rem; list-style:disc;">
                   ${issues.map((m) => `<li><b>${m}</b></li>`).join('')}
                 </ul>
               </div>`,
        confirmButtonText: 'OK',
      });
      return;
    }

    post(route('dependents.store'));
  };

  const breadcrumbs: BreadcrumbItem[] = [
    { title: 'Patients', href: route('patients.index') },
    { title: 'Create Dependent', href: route('dependents.create') },
  ];

  return (
    <AppLayout breadcrumbs={breadcrumbs}>
      <Head title="Create Dependent" />
      <div className="w-full px-4 py-6 sm:px-6">

        {/* Banner shows Parent Full Name — Company (not just ID) */}
        {parent_id && activeParent && (
          <div className="mb-6 rounded-lg border border-blue-300 bg-blue-50 p-3 text-blue-900 dark:border-blue-700 dark:bg-blue-900/20 dark:text-blue-200">
            This dependent will be linked to:&nbsp;
            <b>
              {activeParent.first_name} {activeParent.surname}
            </b>
            {activeParent.company?.name ? (
              <> — <span className="font-medium">{activeParent.company.name}</span></>
            ) : null}
            .
          </div>
        )}

        <form onSubmit={handleSubmit} className="space-y-8">
          <div className="grid gap-6 sm:grid-cols-12">
            <div className="col-span-12 space-y-6 sm:col-span-8">
              <div className="bg-card rounded-lg border p-6 shadow-sm dark:bg-neutral-900">
                <h2 className="mb-4 text-xl font-semibold">Personal Information</h2>
                <PersonalInfoSection data={data} errors={errors} onFieldChange={setData} />
              </div>

              <div className="bg-card rounded-lg border p-6 shadow-sm dark:bg-neutral-900">
                <div className="mb-4 flex items-center justify-between gap-4">
                  <h2 className="text-xl font-semibold">Parent & Relationship</h2>
                </div>
                <div className="grid grid-cols-1 gap-4 sm:grid-cols-2">
                  <div className="space-y-2">
                    <Label htmlFor="parent_patient_id">Parent*</Label>
                    <Select
                      options={parentOptions}
                      value={selectedParent}
                      onChange={handleParentChange}
                      placeholder="Select parent..."
                      styles={selectStyles}
                      className="react-select-container"
                      classNamePrefix="react-select"
                      isDisabled={!!parent_id}  // locked if coming from patient create flow
                      isSearchable
                    />
                    {errors.parent_patient_id && <p className="text-sm text-red-500">{errors.parent_patient_id}</p>}
                  </div>

                  <div className="space-y-2">
                    <Label htmlFor="relationship">Relationship*</Label>
                    <select
                      id="relationship"
                      value={data.relationship}
                      onChange={(e) => setData('relationship', e.target.value)}
                      className="w-full rounded-md border border-gray-300 bg-white px-3 py-2 text-sm text-gray-900 shadow-sm focus:border-blue-500 focus:ring-1 focus:ring-blue-500 focus:outline-none dark:border-gray-700 dark:bg-black dark:text-white"
                    >
                      <option value="">Select relationship</option>
                      {relationships.map((rel) => (
                        <option key={rel} value={rel}>
                          {rel}
                        </option>
                      ))}
                    </select>
                    {errors.relationship && <p className="text-sm text-red-500">{errors.relationship}</p>}
                  </div>
                </div>
              </div>

              {/* Address with "Use Parent Address" */}
              <div className="bg-card rounded-lg border p-6 shadow-sm dark:bg-neutral-900">
                <div className="mb-4 flex items-center justify-between gap-3">
                  <h2 className="text-xl font-semibold">Address Information</h2>
                  <Button
                    type="button"
                    variant="secondary"
                    size="sm"
                    onClick={fillFromParentAddress}
                    disabled={!activeParent}
                    title={activeParent ? 'Copy address fields from parent' : 'Select a parent first'}
                  >
                    Use Parent Address
                  </Button>
                </div>
                <AddressInfoSection data={data} errors={errors} onFieldChange={setData} />
              </div>

              {/* Contact with "Use Parent Phone" + "Use Parent As Emergency Contact" (in Emergency section below) */}
              <div className="bg-card rounded-lg border p-6 shadow-sm dark:bg-neutral-900">
                <div className="mb-4 flex items-center justify-between gap-3">
                  <h2 className="text-xl font-semibold">Contact Information</h2>
                  <Button
                    type="button"
                    variant="secondary"
                    size="sm"
                    onClick={fillParentPhone}
                    disabled={!activeParent || !(activeParent.phone || activeParent.emergency_contact_phone)}
                    title={
                      activeParent
                        ? (activeParent.phone || activeParent.emergency_contact_phone
                          ? 'Copy parent phone to dependent'
                          : 'Parent has no phone on file')
                        : 'Select a parent first'
                    }
                  >
                    Use Parent Phone
                  </Button>
                </div>
                <ContactInfoSection data={data} errors={errors} onFieldChange={setData} />
              </div>

              <div className="bg-card rounded-lg border p-6 shadow-sm dark:bg-neutral-900">
                <div className="mb-4 flex items-center justify-between gap-3">
                  <h2 className="text-xl font-semibold">Emergency Contact</h2>
                  <Button
                    type="button"
                    variant="secondary"
                    size="sm"
                    onClick={fillParentAsEmergency}
                    disabled={!activeParent}
                    title={activeParent ? 'Use parent as emergency contact' : 'Select a parent first'}
                  >
                    Use Parent As Emergency Contact
                  </Button>
                </div>
                <EmergencyContactSection data={data} errors={errors} onFieldChange={setData} />
              </div>

              <div className="bg-card rounded-lg border p-6 shadow-sm dark:bg-neutral-900">
                <h2 className="mb-4 text-xl font-semibold">Medical Information</h2>
                <MedicalInfoSection data={data} errors={errors} onFieldChange={setData} />
              </div>

              <div className="flex justify-end gap-4">
                <Button type="button" variant="outline" onClick={() => window.history.back()}>
                  Cancel
                </Button>
                <Button type="submit" disabled={processing}>
                  {processing ? 'Saving...' : 'Save'}
                </Button>
              </div>
            </div>

            <div className="col-span-12 space-y-6 sm:col-span-4">
              <div className="space-y-4">
                <div className="bg-card rounded-lg border p-6 shadow-sm" style={{ marginTop: '10px' }}>
                  <h2 className="mb-4 text-xl font-semibold">Actions</h2>
                  <Button asChild className="mb-3 w-full">
                    <Link href={route('patients.create')}>Create New Patient</Link>
                  </Button>
                </div>
              </div>
            </div>
          </div>
        </form>
      </div>
    </AppLayout>
  );
}
