import { ContactInfoSection, AddressInfoSection, EmergencyContactSection, MedicalInfoSection, PersonalInfoSection } from '@/components/PatientForm';
import { Button } from '@/components/ui/button';
import { Label } from '@/components/ui/label';
import AppLayout from '@/layouts/app-layout';
import { BreadcrumbItem } from '@/types';
import { Head, useForm, Link, usePage } from '@inertiajs/react';
import { useEffect, useState } from 'react';
import Select from 'react-select';

export default function DependentEdit() {
    const { dependent, parents, relationships, companies } = usePage<{
        dependent: any;
        parents: any[];
        relationships: string[];
        companies: any[];
    }>().props;

    const { data, setData, put, errors, processing } = useForm({
        ...dependent,
        parent_patient_id: dependent.parent?.id || '',
        relationship: dependent.relationship || '',
        company_id: dependent.company?.id || '',
    });

    const [isDarkMode, setIsDarkMode] = useState(false);
    const [selectedParent, setSelectedParent] = useState<any>(null);

    useEffect(() => {
        const checkDarkMode = () => {
            setIsDarkMode(document.documentElement.classList.contains('dark'));
        };
        checkDarkMode();

        const observer = new MutationObserver(checkDarkMode);
        observer.observe(document.documentElement, {
            attributes: true,
            attributeFilter: ['class'],
        });

        return () => observer.disconnect();
    }, []);

    useEffect(() => {
        if (dependent.parent) {
            setSelectedParent({ value: dependent.parent.id, label: `${dependent.parent.first_name} ${dependent.parent.surname}` });
        }
    }, [dependent.parent]);

    const parentOptions = parents.map((parent) => ({
        value: parent.id,
        label: `${parent.first_name} ${parent.surname}`,
    }));

    const handleParentChange = (selectedOption: any) => {
        setSelectedParent(selectedOption);
        setData('parent_patient_id', selectedOption ? selectedOption.value : '');
    };

    const selectStyles = {
        control: (base: any) => ({
            ...base,
            backgroundColor: isDarkMode ? '#000' : '#fff',
            borderColor: isDarkMode ? '#374151' : '#d1d5db',
            color: isDarkMode ? '#fff' : '#000',
            '&:hover': {
                borderColor: isDarkMode ? '#4b5563' : '#9ca3af',
            },
        }),
        menu: (base: any) => ({
            ...base,
            backgroundColor: isDarkMode ? '#000' : '#fff',
            color: isDarkMode ? '#fff' : '#000',
        }),
        option: (base: any, { isFocused }: any) => ({
            ...base,
            backgroundColor: isFocused ? (isDarkMode ? '#1e40af' : '#3b82f6') : 'transparent',
            color: isFocused ? '#fff' : isDarkMode ? '#fff' : '#000',
            '&:active': {
                backgroundColor: isDarkMode ? '#1e40af' : '#3b82f6',
            },
        }),
        singleValue: (base: any) => ({
            ...base,
            color: isDarkMode ? '#fff' : '#000',
        }),
        input: (base: any) => ({
            ...base,
            color: isDarkMode ? '#fff' : '#000',
        }),
        placeholder: (base: any) => ({
            ...base,
            color: isDarkMode ? '#9ca3af' : '#6b7280',
        }),
    };

    const handleSubmit = (e: React.FormEvent) => {
        e.preventDefault();
        put(route('dependents.update', dependent.id));
    };

    const breadcrumbs: BreadcrumbItem[] = [
        { title: 'Dependents', href: route('dependents.index') },
        { title: 'Edit Dependent', href: route('dependents.edit', dependent.id) },
    ];

    return (
        <AppLayout breadcrumbs={breadcrumbs}>
            <Head title={`Edit Dependent`} />
            <div className="w-full px-4 py-6 sm:px-6">
                <div className="mb-6 flex items-center justify-between">
                    <h1 className="text-2xl font-bold">Edit Dependent</h1>
                </div>

                <form onSubmit={handleSubmit} className="space-y-8">
                    <div className="grid grid-cols-12 gap-8">
                        <div className="col-span-8 space-y-6">
                            <div className="bg-card rounded-lg border p-6 shadow-sm dark:bg-neutral-900">
                                <h2 className="mb-4 text-xl font-semibold">Personal Information</h2>
                                <PersonalInfoSection data={data} errors={errors} onFieldChange={(field, value) => setData(field, value)} />
                            </div>

                            <div className="bg-card rounded-lg border p-6 shadow-sm dark:bg-neutral-900">
                                <h2 className="mb-4 text-xl font-semibold">Parent Information</h2>
                                <div className="grid grid-cols-1 gap-4 sm:grid-cols-2">
                                    <div className="space-y-2">
                                        <Label htmlFor="parent_patient_id">Parent*</Label>
                                        <Select
                                            options={parentOptions}
                                            value={selectedParent}
                                            onChange={handleParentChange}
                                            placeholder="Select parent..."
                                            styles={selectStyles}
                                            className="react-select-container"
                                            classNamePrefix="react-select"
                                            isSearchable
                                        />
                                        {errors.parent_patient_id && <p className="text-sm text-red-500">{errors.parent_patient_id}</p>}
                                    </div>

                                    <div className="space-y-2">
                                        <Label htmlFor="relationship">Relationship*</Label>
                                        <select
                                            id="relationship"
                                            value={data.relationship || ''}
                                            onChange={(e) => setData('relationship', e.target.value)}
                                            className="w-full rounded-md border border-gray-300 bg-white px-3 py-2 text-sm text-gray-900 shadow-sm focus:border-blue-500 focus:ring-1 focus:ring-blue-500 focus:outline-none dark:border-gray-700 dark:bg-black dark:text-white dark:focus:border-blue-500 dark:focus:ring-blue-500"
                                        >
                                            <option value="">Select relationship</option>
                                            {relationships.map((rel) => (
                                                <option key={rel} value={rel}>
                                                    {rel}
                                                </option>
                                            ))}
                                        </select>
                                        {errors.relationship && <p className="text-sm text-red-500">{errors.relationship}</p>}
                                    </div>
                                </div>
                            </div>

                            <div className="bg-card rounded-lg border p-6 shadow-sm dark:bg-neutral-900">
                                <h2 className="mb-4 text-xl font-semibold">Address Information</h2>
                                <AddressInfoSection data={data} errors={errors} onFieldChange={(field, value) => setData(field, value)} />
                            </div>
                            <div className="bg-card rounded-lg border p-6 shadow-sm dark:bg-neutral-900">
                                <h2 className="mb-4 text-xl font-semibold">Contact Information</h2>
                                <ContactInfoSection data={data} errors={errors} onFieldChange={(field, value) => setData(field, value)} />
                            </div>

                            <div className="bg-card rounded-lg border p-6 shadow-sm dark:bg-neutral-900">
                                <h2 className="mb-4 text-xl font-semibold">Medical Information</h2>
                                <MedicalInfoSection data={data} errors={errors} onFieldChange={(field, value) => setData(field, value)} />
                            </div>

                            <div className="bg-card rounded-lg border p-6 shadow-sm dark:bg-neutral-900">
                                <h2 className="mb-4 text-xl font-semibold">Emergency Contact</h2>
                                <EmergencyContactSection data={data} errors={errors} onFieldChange={(field, value) => setData(field, value)} />
                            </div>

                            <div className="flex justify-end gap-4">
                                <Button type="button" variant="outline" onClick={() => window.history.back()}>
                                    Cancel
                                </Button>
                                <Button type="submit" disabled={processing}>
                                    {processing ? 'Saving...' : 'Save Changes'}
                                </Button>
                            </div>
                        </div>
                        <div className="col-span-4 space-y-6">
                            <div className="space-y-4">
                                <div className="bg-card rounded-lg border p-6 shadow-sm" style={{ marginTop: '10px' }}>
                                    <h2 className="mb-4 text-xl font-semibold">Actions</h2>
                                    <Button asChild className="mb-3 w-full">
                                        <Link href={route('patients.create')}>Create New Patient</Link>
                                    </Button>

                                    <Button asChild variant="outline" className="mb-3 w-full">
                                        <Link href={route('dependents.create')}>Create New Dependent</Link>
                                    </Button>

                                </div>
                            </div>
                        </div>

                    </div>
                </form>
            </div>
        </AppLayout>
    );
}
