// resources/js/Pages/Dependents/Index.tsx

import AppLayout from '@/layouts/app-layout';
import { API } from '@/config';
import { Head, Link, router, usePage } from '@inertiajs/react';

import { useMemo, useState } from 'react';
import { useDebouncedCallback } from 'use-debounce';
import ReactPaginate from 'react-paginate';
import moment from 'moment';
import Swal from 'sweetalert2';

import { Eye, Pencil, Plus, Search as SearchIcon, Trash2, Stethoscope } from 'lucide-react';

import { Badge } from '@/components/ui/badge';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import {
  Select,
  SelectTrigger,
  SelectValue,
  SelectContent,
  SelectItem,
} from '@/components/ui/select';

export default function DependentIndex() {
  const pageProps = usePage().props as any;
  const { dependents, filters } = pageProps;

  // Detect superadmin
  const user = pageProps?.auth?.user;
  const isSuperAdmin =
    user?.role === 'superadmin' ||
    (Array.isArray(user?.roles) && user.roles.includes('superadmin'));

  // Filters state
  const [search, setSearch] = useState<string>(filters.search || '');
  const [gender, setGender] = useState<string>(filters.gender || '');
  const [relationship, setRelationship] = useState<string>(filters.relationship || '');
  const [ageMin, setAgeMin] = useState<string>(filters.ageMin || '');
  const [ageMax, setAgeMax] = useState<string>(filters.ageMax || '');

  const relationshipOptions = useMemo(
    () => [
      'Father', 'Mother', 'Brother', 'Sister', 'Son', 'Daughter',
      'Spouse', 'Uncle', 'Aunt', 'Cousin', 'Extended Family', 'Other',
    ],
    []
  );

  // Make a request with current state (or overrides)
  const fetch = (page?: number, overrides?: Record<string, any>) => {
    router.get(
      `${API}/dependents`,
      {
        page: page ?? dependents?.current_page ?? 1,
        search,
        gender,
        relationship,
        ageMin,
        ageMax,
        ...(overrides || {}),
      },
      { preserveState: true, replace: true, preserveScroll: true }
    );
  };

  const applyFilters = (page?: number) => fetch(page);

  const resetFilters = () => {
    setSearch('');
    setGender('');
    setRelationship('');
    setAgeMin('');
    setAgeMax('');
    router.get(`${API}/dependents`, {}, { preserveState: true, replace: true, preserveScroll: true });
  };

  const handlePageChange = (selected: { selected: number }) => {
    applyFilters(selected.selected + 1);
  };

  // Debounced search: 400ms after user stops typing
  const debouncedSearch = useDebouncedCallback((value: string) => {
    fetch(1, { search: value });
  }, 400);

  const onChangeSearch = (value: string) => {
    setSearch(value);
    debouncedSearch(value);
  };

  const handleDelete = async (dependent: any) => {
    const fullName = [dependent.first_name, dependent.middle_name, dependent.surname]
      .filter(Boolean)
      .join(' ');

    const { isConfirmed } = await Swal.fire({
      title: `Delete ${fullName}?`,
      icon: 'warning',
      html: `
        <div style="text-align:left">
          <p>This will permanently remove:</p>
          <ul style="margin-left:1rem; list-style:disc;">
            <li><b>Dependent profile</b></li>
            <li><b>All consultations</b> for this dependent</li>
            <li><b>All triages</b> for this dependent</li>
          </ul>
          <p style="margin-top:.5rem;"><b>This action cannot be undone.</b></p>
        </div>
      `,
      showCancelButton: true,
      confirmButtonText: 'Yes, delete',
      cancelButtonText: 'Cancel',
      confirmButtonColor: '#dc2626',
      focusCancel: true,
    });

    if (!isConfirmed) return;

    Swal.fire({
      title: 'Deleting...',
      text: 'Please wait',
      allowOutsideClick: false,
      allowEscapeKey: false,
      showConfirmButton: false,
      didOpen: () => Swal.showLoading(),
    });

    let url = '';
    try {
      url = route('patients.destroy', dependent.id);
    } catch {
      try {
        url = route('dependents.destroy', dependent.id);
      } catch {
        url = `${API}/patients/${dependent.id}`;
      }
    }

    router.visit(url, {
      method: 'delete',
      preserveScroll: true,
      onFinish: () => Swal.close(),
      onSuccess: () => {
        Swal.fire('Deleted', 'Dependent and related records were deleted successfully.', 'success');
        applyFilters(); // reload current page with filters
      },
      onError: () => {
        Swal.fire('Error', 'Failed to delete the dependent.', 'error');
      },
    });
  };

  return (
    <AppLayout breadcrumbs={[{ title: 'Dependents', href: `${API}/dependents` }]}>
      <Head title="Dependents" />

      <div className="bg-background mx-8 my-6 rounded-xl p-6 shadow-sm">
        <div className="mb-4 flex items-center justify-between">
          <h1 className="text-2xl font-bold">Dependents</h1>

          <Button asChild>
            <Link href={route('dependents.create')} className="inline-flex items-center gap-2">
              <Plus className="h-4 w-4" />
              Add Dependent
            </Link>
          </Button>
        </div>

        {/* Filters */}
        <div className="mb-4 flex flex-wrap items-end gap-3">
          <div className="relative w-64">
            <SearchIcon className="text-muted-foreground absolute left-3 top-2.5 h-5 w-5" />
            <Input
              value={search}
              onChange={(e) => onChangeSearch(e.target.value)}
              onKeyDown={(e) => e.key === 'Enter' && applyFilters(1)}
              placeholder="Search dependents or parents..."
              className="pl-10"
            />
          </div>

          {/* Gender Select (uses sentinel to avoid empty value) */}
          <Select
            value={gender || undefined}
            onValueChange={(v) => setGender(v === '__all__' ? '' : v)}
          >
            <SelectTrigger className="w-40">
              <SelectValue placeholder="All Genders" />
            </SelectTrigger>
            <SelectContent>
              <SelectItem value="__all__">All Genders</SelectItem>
              <SelectItem value="Male">Male</SelectItem>
              <SelectItem value="Female">Female</SelectItem>
            </SelectContent>
          </Select>

          {/* Relationship Select (uses sentinel to avoid empty value) */}
          <Select
            value={relationship || undefined}
            onValueChange={(v) => setRelationship(v === '__all__' ? '' : v)}
          >
            <SelectTrigger className="w-56">
              <SelectValue placeholder="All Relationships" />
            </SelectTrigger>
            <SelectContent>
              <SelectItem value="__all__">All Relationships</SelectItem>
              {relationshipOptions.map((rel) => (
                <SelectItem key={rel} value={rel}>
                  {rel}
                </SelectItem>
              ))}
            </SelectContent>
          </Select>

          {/* Ages */}
          <Input
            type="number"
            placeholder="Min Age"
            className="w-28"
            value={ageMin}
            onChange={(e) => setAgeMin(e.target.value)}
          />
          <Input
            type="number"
            placeholder="Max Age"
            className="w-28"
            value={ageMax}
            onChange={(e) => setAgeMax(e.target.value)}
          />

          <Button onClick={() => applyFilters(1)}>Search</Button>
          <Button variant="secondary" onClick={resetFilters}>
            Reset
          </Button>
        </div>

        {/* Table */}
        <div className="overflow-x-auto rounded-xl border">
          <table className="min-w-full divide-y overflow-hidden rounded-xl bg-white text-sm dark:bg-neutral-900">
            <thead className="bg-muted/50">
              <tr>
                <th className="px-4 py-2 text-left">Name</th>
                <th className="px-4 py-2 text-left">Gender</th>
                <th className="px-4 py-2 text-left">Relationship</th>
                <th className="px-4 py-2 text-left">Parent</th>
                <th className="px-4 py-2 text-left">Date of Birth</th>
                <th className="px-4 py-2 text-left">Actions</th>
              </tr>
            </thead>
            <tbody className="divide-y">
              {dependents.data.map((dependent: any) => (
                <tr key={dependent.id}>
                  <td className="px-4 py-2">
                    {[dependent.first_name, dependent.middle_name, dependent.surname]
                      .filter(Boolean)
                      .join(' ')}
                  </td>
                  <td className="px-4 py-2">
                    {dependent.gender ? (
                      <Badge variant="outline" className="capitalize">
                        {dependent.gender}
                      </Badge>
                    ) : (
                      <span className="text-muted-foreground">-</span>
                    )}
                  </td>
                  <td className="px-4 py-2">
                    {dependent.relationship ? (
                      <Badge variant="outline" className="capitalize">
                        {dependent.relationship}
                      </Badge>
                    ) : (
                      <span className="text-muted-foreground">-</span>
                    )}
                  </td>
                  <td className="px-4 py-2">
                    {dependent.parent
                      ? [dependent.parent.first_name, dependent.parent.middle_name, dependent.parent.surname]
                        .filter(Boolean)
                        .join(' ')
                      : '-'}
                  </td>
                  <td className="px-4 py-2">
                    {dependent.date_of_birth
                      ? moment(dependent.date_of_birth).format('MMM D, YYYY')
                      : '-'}
                  </td>
                  <td className="px-4 py-2">
                    <div className="flex flex-wrap items-center gap-2">
                      <Button asChild size="icon" title="View Dependent">
                        <Link href={route('dependents.show', dependent.id)}>
                          <Eye className="h-4 w-4" />
                        </Link>
                      </Button>

                      <Button asChild size="icon" variant="secondary" title="Edit Dependent">
                        <Link href={route('dependents.edit', dependent.id)}>
                          <Pencil className="h-4 w-4" />
                        </Link>
                      </Button>

                      {dependent.latest_consultation_id ? (
                        <Button
                          asChild
                          variant="outline"
                          size="sm"
                          title="View Last Consultation"
                          className="inline-flex items-center gap-2"
                        >
                          <Link href={route('consultations.show', dependent.latest_consultation_id)}>
                            <Stethoscope className="h-4 w-4" />
                            <span>View Last Consultation</span>
                          </Link>
                        </Button>
                      ) : null}
                      {isSuperAdmin && (
                        <Button
                          size="icon"
                          variant="destructive"
                          onClick={() => handleDelete(dependent)}
                          title="Delete Dependent"
                        >
                          <Trash2 className="h-4 w-4" />
                        </Button>
                      )}
                    </div>
                  </td>
                </tr>
              ))}
            </tbody>
          </table>
        </div>

        {/* Pagination */}
        <div className="mt-4 flex justify-center">
          <ReactPaginate
            pageCount={dependents.last_page}
            forcePage={dependents.current_page - 1}
            onPageChange={handlePageChange}
            marginPagesDisplayed={1}
            pageRangeDisplayed={3}
            previousLabel="← Prev"
            nextLabel="Next →"
            breakLabel="..."
            containerClassName="flex items-center gap-2 text-sm"
            pageClassName="px-3 py-1 border rounded hover:bg-muted"
            activeClassName="bg-blue-600 text-white"
            previousClassName="px-3 py-1 border rounded hover:bg-muted"
            nextClassName="px-3 py-1 border rounded hover:bg-muted"
            breakClassName="px-2"
          />
        </div>
      </div>
    </AppLayout>
  );
}
