import { Button } from '@/components/ui/button';
import AppLayout from '@/layouts/app-layout';
import { BreadcrumbItem } from '@/types';
import { Head, router, usePage } from '@inertiajs/react';
import { ImageIcon } from 'lucide-react';
import moment from 'moment';
import { QRCodeSVG } from 'qrcode.react';
import React, { useRef } from 'react';
import { useReactToPrint } from 'react-to-print';
import Swal from 'sweetalert2';
import { type NavItem, type SharedData } from '@/types';


const logoUrl = 'https://providencehumancapital.com/headoffice/assets/images/PHC_Logo.png';


interface Company {
    id: number;
    name: string;
}
interface Consultation {
    id: number;
    consultation_date: string;
    presenting_complaint?: string;
    diagnosis?: string;
    doctor?: {
        name: string;
    };
    clinic?: {
        name: string;
        address: string;
        contact_person: string;
        email_address: string;
    };
}

interface Parent {
    id: number;
    first_name: string;
    middle_name?: string;
    surname: string;
    employee_number?: string;
    company?: Company;
    phone?: string;
    email?: string;
    id_number?: string;
    medical_aid_number?: string;
    medical_aid_provider?: string;
}

interface Dependent {
    id: number;
    first_name: string;
    middle_name?: string;
    surname: string;
    date_of_birth?: string;
    id_number?: string;
    relationship?: string;
    email?: string;
    gender?: string;
    phone?: string;
    emergency_contact_name?: string;
    emergency_contact_relation?: string;
    emergency_contact_phone?: string;

    home_address?: string;
    work_area?: string;
    suburb?: string;

    allergies?: string;
    is_smoker?: boolean;
    occupation?: string;
    marital_status?: string;
    medical_aid_number?: string;
    medical_aid_provider?: string;
    parent?: Parent;
    company?: Company;
    image_url?: string;
    latest_consultation?: Consultation;
}

type UserRole =
    | 'superadmin'
    | 'receptionist'
    | 'nurse'
    | 'doctor'
    | 'lab_technician'
    | 'radiologist'
    | 'pharmacist'
    | 'nurse_aid'
    | 'admin'
    | 'specialist'
    | 'casualty'
    | 'user'
    | 'patient';


export default function DependentShow() {
    const { dependent } = usePage<{ dependent: Dependent }>().props;

    const page = usePage<SharedData>();
    const { auth } = page.props;

    const userRole: UserRole | undefined = auth.user?.role as UserRole | undefined;


    const contentRef = useRef<HTMLDivElement>(null);
    const reactToPrintFn = useReactToPrint({ contentRef });

    const {
        first_name,
        middle_name,
        surname,
        id_number,
        date_of_birth,
        gender,
        email,
        phone,
        occupation,
        marital_status,
        is_smoker,
        medical_aid_number,
        medical_aid_provider,
        allergies,
        emergency_contact_name,
        emergency_contact_relation,
        emergency_contact_phone,

        home_address,
        work_area,
        suburb,

        relationship,
        parent,
        image_url,
    } = dependent;

    const fullName = [first_name, middle_name, surname].filter(Boolean).join(' ');
    const parentFullName = parent ? [parent.first_name, parent.middle_name, parent.surname].filter(Boolean).join(' ') : '';

    const breadcrumbs: BreadcrumbItem[] = [
        { title: 'Dependents', href: route('dependents.index') },
        { title: fullName, href: route('dependents.show', dependent.id) },
    ];

    const Info = ({ label, value }: { label: string; value?: string | null }) => (
        <div>
            <div className="text-muted-foreground text-sm">{label}</div>
            <div className="text-foreground font-medium">{value || '-'}</div>
        </div>
    );

    const InfoGrid = ({ items }: { items: { label: string; value?: string | null }[] }) => (
        <div className="grid grid-cols-1 gap-4 sm:grid-cols-2 md:grid-cols-3">
            {items.map((item) => (
                <Info key={item.label} {...item} />
            ))}
        </div>
    );

    const Section = ({ title, children }: { title: string; children: React.ReactNode }) => (
        <section className="bg-card rounded-lg border p-6 shadow-sm dark:bg-neutral-900">
            <h2 className="text-foreground mb-4 text-xl font-semibold">{title}</h2>
            {children}
        </section>
    );

    const handleWritePrescription = () => {
        if (!dependent?.latest_consultation) {
            Swal.fire({
                title: 'No Consultation Found',
                text: 'Please create a consultation before writing a prescription.',
                icon: 'warning',
                confirmButtonText: 'OK',
            });
        } else {
            router.visit(route('consultations.prescriptions.create', dependent.latest_consultation.id));
        }
    }

    return (
        <AppLayout breadcrumbs={breadcrumbs}>
            <Head title={`Dependent: ${fullName}`} />

            <div style={{ display: 'none' }}>
                <div ref={contentRef} className="p-6 font-sans text-sm text-gray-800">
                    <style>
                        {`
                                        @page {
                                            size: auto;
                                            margin: 15mm 10mm;
                                        }
                                        @media print {
                                            body { 
                                                -webkit-print-color-adjust: exact;
                                                print-color-adjust: exact;
                                            }
                                            .section-break {
                                                page-break-after: always;
                                            }
                                            .avoid-break {
                                                page-break-inside: avoid;
                                            }
                                        }
                                    `}
                    </style>
                    <div className="flex items-center justify-between border-b-2 border-gray-200 pb-4">
                        <div className="flex items-center">
                            <img src={logoUrl} alt="PHC Logo" style={{ height: '4rem' }} />
                        </div>
                        <div className="text-center text-sm text-gray-800">
                            <p className="text-lg font-bold">Clinic Attendance</p>
                            <p>{moment(dependent.latest_consultation?.consultation_date).format('MMMM Do YYYY, h:mm a')}</p>
                        </div>
                        <div className="flex items-center">
                            <QRCodeSVG value={`Attendance Date: ${dependent.id}`} size={60} />
                        </div>
                    </div>
                    <div className="mt-8 flex justify-between">
                        <div className="flex-1"></div>
                        <div className="flex-1 text-right">
                            <h2 className="text-lg font-bold">Clinic Information</h2>
                            <p>{dependent.latest_consultation?.clinic?.name}</p>
                            <p>Clinic Address: {dependent.latest_consultation?.clinic?.address}</p>
                            <p>Contact Person: {dependent.latest_consultation?.clinic?.contact_person}</p>
                            <p>Email Address: {dependent.latest_consultation?.clinic?.email_address}</p>
                        </div>
                    </div>

                    <div className="mt-4">
                        <h2 className="text-lg font-bold">Patient Information</h2>
                        <p>
                            Patient Name: {dependent.first_name} {dependent.surname}
                        </p>
                        <p>Patient ID: {dependent.id}</p>
                        <p>ID Number: {dependent.id_number}</p>
                    </div>

                    <div className="mt-4">
                        <p>
                            This is to certify that {dependent.first_name} {dependent.surname}, identified by ID number {dependent.id_number}, attended the
                            clinic on {moment(dependent.latest_consultation?.consultation_date).format('MMMM Do YYYY')}. The patient was seen by our
                            medical team and received the necessary care and attention. This slip serves as official documentation of the patient's
                            attendance and consultation.
                        </p>
                    </div>

                    <div className="mt-4" style={{
                        marginTop: '6rem',
                    }}>
                        <h2 className="text-lg font-bold">Doctor's Information</h2>
                        <p>Doctor's Name: ______________________</p>
                        <p>Doctor's Signature</p>
                        <div className="mt-2 h-16 w-48 border-2 border-gray-300"></div>
                    </div>

                    <div
                        className="mt-auto pt-8"
                        style={{
                            marginTop: '7rem',
                        }}
                    >
                        <div className="flex justify-between">
                            <div>
                                <p>Date: {moment().format('MMMM Do YYYY, h:mm A')}</p>
                            </div>
                            <div className="h-24 w-34 border-2 border-gray-300"></div>
                        </div>
                    </div>
                </div>
            </div>









            <div className="w-full space-y-8 px-4 py-10 sm:px-6">
                {/* Header: Image + Name + Buttons */}
                <div className="flex flex-col gap-4 sm:flex-row sm:items-center sm:justify-between">
                    <div className="flex items-center gap-4">
                        {image_url ? (
                            <img src={image_url} alt="Dependent" className="h-24 w-24 rounded-full border object-cover" />
                        ) : (
                            <div className="bg-muted text-muted-foreground flex h-24 w-24 items-center justify-center rounded-full border">
                                <ImageIcon className="h-8 w-8" />
                            </div>
                        )}
                        <div>
                            <h1 className="text-foreground text-2xl font-bold">{fullName}</h1>
                            <div className="mt-1 inline-block rounded-full bg-red-100 px-3 py-1 text-sm font-medium text-red-800">Dependent</div>
                        </div>
                    </div>

                    {/* Action Buttons */}
                    <div className="flex flex-wrap gap-2">
                        <div className="flex flex-wrap gap-2">
                            <Button
                                onClick={() => (window.location.href = route('patients.create'))}
                                variant="outline"
                                style={{
                                    cursor: 'pointer',
                                }}
                            >
                                Add Patient
                            </Button>
                            {userRole && ["superadmin", "nurse", "doctor"].includes(userRole) && (
                                <>
                                    <Button variant="secondary" onClick={() => router.visit(route('consultations.create', { patient_id: dependent.id }))}>
                                        Start Consultation
                                    </Button>
                                    <Button onClick={handleWritePrescription}>Write Prescription</Button>
                                </>
                            )}
                            <Button
                                onClick={reactToPrintFn}
                                variant="outline"
                                style={{
                                    cursor: 'pointer',
                                }}
                            >
                                Print Attendance Slip
                            </Button>
                            <Button
                                onClick={() => alert('Export clicked')}
                                variant="destructive"
                                style={{
                                    cursor: 'pointer',
                                }}
                            >
                                Export Record
                            </Button>
                        </div>
                    </div>
                </div>

                <div className="grid grid-cols-12 gap-8">
                    <div className="col-span-8 space-y-6">
                        <Section title="Personal Information">
                            <InfoGrid
                                items={[
                                    { label: 'Full Name', value: fullName },
                                    { label: 'ID Number', value: id_number },
                                    { label: 'Date of Birth', value: date_of_birth ? moment(date_of_birth).format('DD MMM YYYY') : '-' },

                                    { label: 'Gender', value: gender },
                                    { label: 'Email', value: email },
                                    { label: 'Phone', value: phone },
                                    { label: 'Marital Status', value: marital_status },
                                ]}
                            />
                        </Section>

                        <Section title="Home Address & Work">
                            <InfoGrid
                                items={[
                                    { label: 'Home Address', value: home_address },
                                    { label: 'Suburb', value: work_area },
                                    { label: 'Work Place', value: suburb },
                                ]}
                            />
                        </Section>

                        <Section title="Medical & Lifestyle">
                            <InfoGrid
                                items={[
                                    { label: 'Medical Aid Number', value: medical_aid_number },
                                    { label: 'Medical Aid Provider', value: medical_aid_provider },
                                    { label: 'Allergies', value: allergies },
                                    { label: 'Smoker', value: is_smoker ? 'Yes' : 'No' },
                                ]}
                            />
                        </Section>

                        <Section title="Emergency Contact">
                            <InfoGrid
                                items={[
                                    { label: 'Name', value: emergency_contact_name },
                                    { label: 'Relation', value: emergency_contact_relation },
                                    { label: 'Phone', value: emergency_contact_phone },
                                ]}
                            />
                        </Section>
                    </div>

                    <div className="col-span-4 space-y-6">
                        <div
                            className="space-y-4"
                            style={{
                                marginBottom: '10px',
                            }}
                        >
                            <div className="bg-card rounded-lg border p-6 shadow-sm">
                                <h2 className="mb-4 text-xl font-semibold">Actions</h2>
                                {userRole && ["superadmin", "nurse", "doctor"].includes(userRole) && (
                                    <Button asChild className="w-full">
                                        <a href={route('patients.consultations', dependent.id)}>Consultation History</a>
                                    </Button>
                                )}

                                <Button asChild variant="outline" className="w-full">
                                    <a href={route('dependents.edit', dependent.id)}>Edit Patient</a>
                                </Button>
                            </div>
                        </div>
                        {userRole && ["superadmin", "nurse", "doctor"].includes(userRole) && (
                            <Section title="Latest Consultation">
                                {dependent.latest_consultation ? (
                                    <div className="space-y-4">
                                        <InfoGrid
                                            items={[
                                                {
                                                    label: 'Date',
                                                    value: moment(dependent.latest_consultation.consultation_date).format('DD MMM YYYY, h:mm a'),
                                                },
                                                {
                                                    label: 'Doctor',
                                                    value: dependent.latest_consultation.doctor?.name || '-',
                                                },
                                                {
                                                    label: 'Clinic',
                                                    value: dependent.latest_consultation.clinic?.name || '-',
                                                },
                                            ]}
                                        />

                                        {dependent.latest_consultation.presenting_complaint && (
                                            <div>
                                                <div className="text-muted-foreground text-sm">Presenting Complaint</div>
                                                <div className="text-foreground rounded border p-3 font-medium">
                                                    {dependent.latest_consultation.presenting_complaint}
                                                </div>
                                            </div>
                                        )}

                                        {dependent.latest_consultation.diagnosis && (
                                            <div>
                                                <div className="text-muted-foreground text-sm">Diagnosis</div>
                                                <div className="text-foreground rounded border p-3 font-medium">
                                                    {dependent.latest_consultation.diagnosis}
                                                </div>
                                            </div>
                                        )}

                                        <Button
                                            variant="outline"
                                            className="w-full"
                                            onClick={() => router.visit(route('consultations.show', dependent.latest_consultation?.id))}
                                        >
                                            View Full Consultation
                                        </Button>
                                    </div>
                                ) : (
                                    <div className="text-muted-foreground">No consultations found</div>
                                )}
                            </Section>
                        )}

                        <Section title="Parent Information">
                            <div className="space-y-4">
                                <div>
                                    <div className="text-muted-foreground text-sm">Parent Name</div>
                                    <div className="text-foreground font-medium">
                                        {parentFullName || '-'}
                                        {parent && (
                                            <Button
                                                variant="link"
                                                className="text-primary ml-2 h-auto p-0"
                                                onClick={() => (window.location.href = route('patients.show', parent.id))}
                                            >
                                                View
                                            </Button>
                                        )}
                                    </div>
                                </div>

                                <InfoGrid
                                    items={[
                                        { label: 'Relationship', value: relationship },
                                        { label: 'Employee Number', value: parent?.employee_number },
                                        { label: 'ID Number', value: parent?.id_number },
                                    ]}
                                />
                            </div>
                        </Section>

                        {parent?.company && (
                            <Section title="Parent's Company">
                                <InfoGrid
                                    items={[
                                        { label: 'Company', value: parent.company.name },
                                        { label: 'Phone', value: parent.phone },
                                        { label: 'Email', value: parent.email },
                                    ]}
                                />
                            </Section>
                        )}

                        <Section title="Medical Aid (Parent)">
                            <InfoGrid
                                items={[
                                    { label: 'Medical Aid Number', value: parent?.medical_aid_number },
                                    { label: 'Medical Aid Provider', value: parent?.medical_aid_provider },
                                ]}
                            />
                        </Section>
                    </div>
                </div>
            </div>
        </AppLayout>
    );
}
