import InputError from '@/components/input-error';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import AppLayout from '@/layouts/app-layout';
import { BreadcrumbItem } from '@/types';
import { Head, useForm } from '@inertiajs/react';
import { FormEvent } from 'react';
import * as Yup from 'yup';

interface DiagnosisPresetData {
    name: string;
    icd_code: string;
    description?: string;
    [key: string]: any;
}

export default function EditDiagnosisPreset({ diagnosis }: { diagnosis: DiagnosisPresetData }) {
    const { data, setData, put, processing, errors, reset } = useForm<DiagnosisPresetData>({
        name: diagnosis.name,
        icd_code: diagnosis.icd_code,
        description: diagnosis.description || '',
    });

    const schema = Yup.object().shape({
        name: Yup.string().required('Name is required'),
        icd_code: Yup.string().required('ICD code is required'),
    });

    const handleSubmit = async (e: FormEvent) => {
        e.preventDefault();
        try {
            await schema.validate(data, { abortEarly: false });
            put(route('diagnosis-presets.update', diagnosis.id), {
                preserveScroll: true,
                onSuccess: () => reset(),
            });
        } catch (validationError: any) {
            const fieldErrors: Record<string, string> = {};
            validationError.inner.forEach((err: any) => {
                if (err.path) fieldErrors[err.path] = err.message;
            });
        }
    };

    const breadcrumbs: BreadcrumbItem[] = [
        { title: 'Diagnosis Presets', href: route('diagnosis-presets.index') },
        { title: 'Edit Diagnosis Preset', href: route('diagnosis-presets.edit', diagnosis.id) },
    ];

    return (
        <AppLayout breadcrumbs={breadcrumbs}>
            <Head title="Edit Diagnosis Preset" />

            <div className="w-full space-y-8 px-4 py-10 sm:px-6">
                <h1 className="text-foreground text-2xl font-bold">Edit Diagnosis Preset</h1>

                <form onSubmit={handleSubmit} className="space-y-8">
                    <div className="grid grid-cols-1 gap-6 sm:grid-cols-2">
                        {['name', 'icd_code', 'description'].map((field) => (
                            <div key={field} className="grid gap-2">
                                <Label htmlFor={field}>
                                    {field.replace(/_/g, ' ').replace(/\b\w/g, (l) => l.toUpperCase())}
                                    {field !== 'description' && <span className="text-red-500">*</span>}
                                </Label>
                                <Input
                                    id={field}
                                    name={field}
                                    value={data[field]}
                                    onChange={(e) => setData(field, e.target.value)}
                                />
                                <InputError message={errors[field]} />
                            </div>
                        ))}
                    </div>

                    <div className="flex justify-end gap-3">
                        <Button type="button" variant="outline" onClick={() => window.history.back()}>
                            Cancel
                        </Button>
                        <Button type="submit" disabled={processing}>
                            {processing ? 'Updating...' : 'Update Diagnosis'}
                        </Button>
                    </div>
                </form>
            </div>
        </AppLayout>
    );
}