import { API } from '@/config';
import AppLayout from '@/layouts/app-layout';
import { BreadcrumbItem } from '@/types';
import { PageProps } from '@inertiajs/core';
import { Head, Link, router, usePage } from '@inertiajs/react';
import { Download, Pencil, Plus, Search, Trash } from 'lucide-react';
import { useState } from 'react';
import ReactPaginate from 'react-paginate';
import Swal from 'sweetalert2';
import * as XLSX from 'xlsx';

const breadcrumbs: BreadcrumbItem[] = [{ title: 'Diagnosis Presets', href: `${API}/diagnosis-presets` }];

interface DiagnosisPreset {
    id: number;
    name: string;
    icd_code: string;
    description?: string;
}

interface PagePropsWithDiagnoses extends PageProps {
    diagnoses: {
        data: DiagnosisPreset[];
        total: number;
        per_page: number;
        current_page: number;
        last_page: number;
    };
    filters: {
        search?: string;
    };
}

type AuthUser = {
    id: number;
    name: string;
    role?: string;
    roles?: string[];
    can?: Record<string, boolean>;
};

export default function DiagnosisPresetList() {
    const { diagnoses, filters } = usePage<PagePropsWithDiagnoses>().props;
    const [searchQuery, setSearchQuery] = useState(filters.search || '');

    const { auth } = usePage<{ auth: { user: AuthUser } }>().props;

    const canModify =
        auth?.user?.role === 'superadmin' ||
        auth?.user?.roles?.includes?.('superadmin') ||
        auth?.user?.can?.delete_consultations === true;
    const handleSearch = () => {
        router.get(`${API}/diagnosis-presets`, { search: searchQuery }, { preserveState: true });
    };

    const handlePageChange = (selected: { selected: number }) => {
        router.get(`${API}/diagnosis-presets`, { page: selected.selected + 1, search: searchQuery }, { preserveState: true });
    };

    const exportToExcel = () => {
        // Prepare data for export
        const data = diagnoses.data.map((diagnosis) => ({
            Name: diagnosis.name,
            'ICD Code': diagnosis.icd_code,
            Description: diagnosis.description || '-',
        }));

        // Create worksheet
        const worksheet = XLSX.utils.json_to_sheet(data);

        // Calculate max length for each column
        type DiagnosisExportKey = 'Name' | 'ICD Code' | 'Description';
        const keys: DiagnosisExportKey[] = ['Name', 'ICD Code', 'Description'];
        const cols = keys.map((key) => {
            // Get max length for this column: consider header + data lengths
            const maxLength = Math.max(
                key.length,
                ...data.map((row) => (row[key] ? row[key].toString().length : 0))
            );
            // Approximate width in characters, add some padding
            return { wch: maxLength + 2 };
        });

        // Assign column widths to worksheet
        worksheet['!cols'] = cols;

        // Create workbook
        const workbook = XLSX.utils.book_new();
        XLSX.utils.book_append_sheet(workbook, worksheet, 'Diagnosis Presets');

        // Generate file and download
        XLSX.writeFile(workbook, 'diagnosis_presets.xlsx', {
            compression: true, // Enable ZIP compression for smaller files
        });
    };


    const handleDelete = (id: number) => {
        const isDarkMode = document.documentElement.classList.contains('dark');

        Swal.fire({
            title: 'Are you sure?',
            text: 'This diagnosis preset will be permanently deleted.',
            icon: 'warning',
            showCancelButton: true,
            confirmButtonColor: '#e3342f',
            cancelButtonColor: '#6c757d',
            confirmButtonText: 'Yes, delete it!',
            width: '300px',
            padding: '1rem',
            background: isDarkMode ? '#1a1a1a' : '#ffffff',
            color: isDarkMode ? '#ffffff' : '#000000',
            customClass: {
                title: 'text-lg',
                htmlContainer: 'text-sm',
                confirmButton: 'text-sm py-1 px-3',
                cancelButton: 'text-sm py-1 px-3',
            },
        }).then((result) => {
            if (result.isConfirmed) {
                router.delete(route('diagnosis-presets.destroy', id), {
                    onSuccess: () => {
                        Swal.fire({
                            title: 'Deleted!',
                            text: 'The diagnosis preset has been deleted.',
                            icon: 'success',
                            timer: 1500,
                            showConfirmButton: false,
                            width: '300px',
                            padding: '1rem',
                            background: isDarkMode ? '#1a1a1a' : '#ffffff',
                            color: isDarkMode ? '#ffffff' : '#000000',
                            customClass: {
                                title: 'text-lg',
                                htmlContainer: 'text-sm',
                            },
                        });
                    },
                    onError: () => {
                        Swal.fire({
                            title: 'Error!',
                            text: 'Something went wrong.',
                            icon: 'error',
                            width: '300px',
                            padding: '1rem',
                            background: isDarkMode ? '#1a1a1a' : '#ffffff',
                            color: isDarkMode ? '#ffffff' : '#000000',
                            customClass: {
                                title: 'text-lg',
                                htmlContainer: 'text-sm',
                            },
                        });
                    },
                });
            }
        });
    };

    return (
        <AppLayout breadcrumbs={breadcrumbs}>
            <Head title="Diagnosis Presets" />
            <div className="bg-background text-foreground mx-8 my-6 flex flex-1 flex-col gap-4 rounded-xl p-6 shadow-sm">
                {/* Rest of your component remains the same */}
                <div className="mb-4 flex flex-wrap items-center justify-between gap-4">
                    <h1 className="text-2xl font-bold">Diagnosis Presets</h1>
                    <div className="flex items-center gap-3">
                        <button
                            onClick={exportToExcel}
                            className="inline-flex items-center gap-2 rounded-lg bg-green-600 px-4 py-2 text-sm font-medium text-white hover:bg-green-700"
                        >
                            <Download className="h-4 w-4" />
                            Export Excel
                        </button>
                        <Link
                            href={route('diagnosis-presets.create')}
                            className="inline-flex items-center gap-2 rounded-lg bg-blue-600 px-4 py-2 text-sm font-medium text-white hover:bg-blue-700"
                        >
                            <Plus className="h-4 w-4" />
                            Add Diagnosis
                        </Link>
                    </div>
                </div>

                <div className="mb-4 flex gap-2">
                    <div className="relative w-72">
                        <Search className="text-muted-foreground absolute top-2.5 left-3 h-5 w-5" />
                        <input
                            type="text"
                            value={searchQuery}
                            onChange={(e) => setSearchQuery(e.target.value)}
                            onKeyDown={(e) => e.key === 'Enter' && handleSearch()}
                            placeholder="Search diagnoses..."
                            className="border-border bg-background placeholder-muted-foreground w-full rounded-lg border py-2 pr-3 pl-10 text-sm focus:ring-2 focus:ring-blue-600 focus:outline-none"
                        />
                    </div>
                    <button onClick={handleSearch} className="rounded-lg bg-blue-600 px-4 py-2 text-sm font-medium text-white hover:bg-blue-700">
                        Search
                    </button>
                </div>

                <div className="overflow-x-auto rounded-xl border bg-neutral-100">
                    <table className="min-w-full divide-y overflow-hidden bg-neutral-100 rounded-xl bg-white text-sm dark:bg-neutral-900">
                        <thead className="bg-muted">
                            <tr>
                                {['Name', 'ICD Code', 'Description', 'Actions'].map((header) => (
                                    <th
                                        key={header}
                                        className="text-muted-foreground px-4 py-3 text-left text-xs font-semibold tracking-wider uppercase"
                                    >
                                        {header}
                                    </th>
                                ))}
                            </tr>
                        </thead>
                        <tbody className="divide-y">
                            {diagnoses.data.map((diagnosis) => (
                                <tr key={diagnosis.id} className="hover:bg-muted/50 transition">
                                    <td className="px-4 py-3">{diagnosis.name}</td>
                                    <td className="px-4 py-3">{diagnosis.icd_code}</td>
                                    <td className="px-4 py-3">{diagnosis.description || '-'}</td>
                                    <td className="flex items-center gap-2 px-4 py-3">
                                        <Link
                                            href={route('diagnosis-presets.edit', diagnosis.id)}
                                            className="inline-flex items-center justify-center rounded-full bg-yellow-500 p-2 text-white hover:bg-yellow-600"
                                            title="Edit"
                                        >
                                            <Pencil className="h-4 w-4" />
                                        </Link>
                                        {
                                            canModify && (
                                                <>
                                                    <button
                                                        onClick={() => handleDelete(diagnosis.id)}
                                                        className="inline-flex items-center justify-center rounded-full bg-red-600 p-2 text-white hover:bg-red-700"
                                                        title="Delete"
                                                    >
                                                        <Trash className="h-4 w-4" />
                                                    </button>
                                                </>
                                            )
                                        }

                                    </td>
                                </tr>
                            ))}
                        </tbody>
                    </table>
                </div>

                <div className="mt-6 flex justify-center">
                    <ReactPaginate
                        pageCount={diagnoses.last_page}
                        forcePage={diagnoses.current_page - 1}
                        onPageChange={handlePageChange}
                        marginPagesDisplayed={1}
                        pageRangeDisplayed={3}
                        previousLabel="← Prev"
                        nextLabel="Next →"
                        breakLabel="..."
                        containerClassName="flex items-center gap-2 text-sm"
                        pageClassName="px-3 py-1 border border-border rounded hover:bg-muted"
                        activeClassName="bg-blue-600 text-white"
                        previousClassName="px-3 py-1 border border-border rounded hover:bg-muted"
                        nextClassName="px-3 py-1 border border-border rounded hover:bg-muted"
                        breakClassName="px-2"
                    />
                </div>
            </div>
        </AppLayout>
    );
}
