import { API } from '@/config';
import AppLayout from '@/layouts/app-layout';
import { BreadcrumbItem } from '@/types';
import { PageProps } from '@inertiajs/core';
import { Head, Link, router, usePage } from '@inertiajs/react';
import { Eye, Pencil, Plus, Search, Trash, Download } from 'lucide-react';
import moment from 'moment';
import { useState } from 'react';
import ReactPaginate from 'react-paginate';
import Swal from 'sweetalert2';
import axios from 'axios';

const breadcrumbs: BreadcrumbItem[] = [{ title: 'Dispensations', href: `${API}/dispensations` }];

interface Patient {
  id: number;
  first_name: string;
  middle_name: string;
  surname: string;
}

interface User {
  id: number;
  name: string;
}

interface Medication {
  id: number;
  name: string;
}

interface MedicationBatch {
  id: number;
  medication: Medication;
}

interface ClinicMedicationStock {
  id: number;
  medication_batch: MedicationBatch;
}
interface NurseMedicationStock {
  id: number;
  medication_batch: MedicationBatch;
}

interface Dispensation {
  id: number;
  patient_id: number;
  consultation_id: number;
  clinic_medication_stock_id: number;
  dispensed_by: User;
  quantity: number;
  route: string;
  dispensed_at: string;
  notes: string;
  patient: Patient;
  clinic_medication_stock: ClinicMedicationStock;
  nurse_medication_stock: NurseMedicationStock;
}

interface PagePropsWithDispensations extends PageProps {
  dispensations: {
    data: Dispensation[];
    total: number;
    per_page: number;
    current_page: number;
    last_page: number;
  };
  patients: Patient[];
  users: User[];
  filters: {
    search?: string;
    start_date?: string;
    end_date?: string;
    patient_id?: string;
    dispensed_by?: string;
  };
}


type AuthUser = {
  id: number;
  name: string;
  role?: string;
  roles?: string[];
  can?: Record<string, boolean>;
};


export default function AllDispensations() {
  const { dispensations, patients, users, filters } = usePage<PagePropsWithDispensations>().props;
  const [searchQuery, setSearchQuery] = useState(filters.search || '');
  const [startDate, setStartDate] = useState(filters.start_date || '');
  const [endDate, setEndDate] = useState(filters.end_date || '');
  const [patientId, setPatientId] = useState(filters.patient_id || '');
  const [dispensedById, setDispensedById] = useState(filters.dispensed_by || '');
  const [isExporting, setIsExporting] = useState(false);


  const { auth } = usePage<{ auth: { user: AuthUser } }>().props;

  // visible if role is exactly "superadmin" (with a couple of safe fallbacks)
  const canModify =
    auth?.user?.role === 'superadmin' ||
    auth?.user?.roles?.includes?.('superadmin') ||
    auth?.user?.can?.delete_consultations === true;

  const handleFilter = () => {
    router.get(
      `${API}/dispensations`,
      {
        search: searchQuery,
        start_date: startDate,
        end_date: endDate,
        patient_id: patientId,
        dispensed_by: dispensedById,
      },
      { preserveState: true },
    );
  };

  // NEW: server-generated Excel (includes Diagnosis + SickLeave tick, per-user sheets if no user selected)
  const handleExportToExcel = async () => {
    setIsExporting(true);
    try {
      // Build URL with current filters
      const params = new URLSearchParams();
      if (searchQuery) params.append('search', searchQuery);
      if (startDate) params.append('start_date', startDate);
      if (endDate) params.append('end_date', endDate);
      if (patientId) params.append('patient_id', patientId);
      if (dispensedById) params.append('dispensed_by', dispensedById);

      Swal.fire({
        title: 'Preparing export…',
        html: 'Generating the dispensation report',
        allowOutsideClick: false,
        didOpen: () => Swal.showLoading(),
      });

      const url = route('dispensations.report') + (params.toString() ? `?${params.toString()}` : '');
      const res = await axios.get(url, { responseType: 'blob' });

      let filename = 'Dispensations.xlsx';
      const dispo = res.headers['content-disposition'];
      if (dispo) {
        const match = /filename[^;=\n]*=((['"]).*?\2|[^;\n]*)/.exec(dispo);
        if (match?.[1]) filename = decodeURIComponent(match[1].replace(/['"]/g, ''));
      }

      const blob = new Blob([res.data], { type: 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet' });
      const blobUrl = window.URL.createObjectURL(blob);
      const a = document.createElement('a');
      a.href = blobUrl;
      a.download = filename;
      document.body.appendChild(a);
      a.click();
      a.remove();
      window.URL.revokeObjectURL(blobUrl);

      Swal.close();
      Swal.fire({ icon: 'success', title: 'Download started', timer: 1200, showConfirmButton: false });
    } catch (error: any) {
      Swal.close();
      const msg = error?.response?.data?.message || error?.message || 'Export failed';
      Swal.fire({ title: 'Export Failed', text: msg, icon: 'error' });
    } finally {
      setIsExporting(false);
    }
  };

  const handleResetFilters = () => {
    setSearchQuery('');
    setStartDate('');
    setEndDate('');
    setPatientId('');
    setDispensedById('');
    router.get(`${API}/dispensations`, {}, { preserveState: true });
  };

  const handleDelete = (id: number) => {
    Swal.fire({
      title: 'Are you sure?',
      text: 'This dispensation will be permanently deleted.',
      icon: 'warning',
      showCancelButton: true,
      confirmButtonColor: '#e3342f',
      cancelButtonColor: '#6c757d',
      confirmButtonText: 'Yes, delete it!',
      width: '350px',
    }).then((result) => {
      if (result.isConfirmed) {
        router.delete(route('dispensations.destroy', id), {
          onSuccess: () => {
            Swal.fire({
              title: 'Deleted!',
              text: 'The dispensation has been deleted.',
              icon: 'success',
              timer: 1500,
              showConfirmButton: false,
            });
          },
          onError: () => {
            Swal.fire({
              title: 'Error!',
              text: 'Something went wrong.',
              icon: 'error',
            });
          },
        });
      }
    });
  };

  const handlePageChange = (selected: { selected: number }) => {
    router.get(
      `${API}/dispensations`,
      {
        page: selected.selected + 1,
        search: searchQuery,
        start_date: startDate,
        end_date: endDate,
        patient_id: patientId,
        dispensed_by: dispensedById,
      },
      { preserveState: true },
    );
  };

  return (
    <AppLayout breadcrumbs={breadcrumbs}>
      <Head title="All Dispensations" />

      <div className="bg-background text-foreground mx-8 my-6 flex flex-col gap-4 rounded-xl p-6 shadow-sm">
        <div className="mb-4 flex flex-wrap items-center justify-between gap-4">
          <h1 className="text-2xl font-bold">Drug Usage Report</h1>
          <div className="flex items-center gap-4">
            <button
              onClick={handleExportToExcel}
              disabled={isExporting}
              className="inline-flex items-center gap-2 rounded-lg bg-green-600 px-5 py-2 text-sm font-medium text-white hover:bg-green-700 disabled:opacity-70"
              style={{ cursor: 'pointer' }}
            >
              <Download className="h-4 w-4" />
              {isExporting ? 'Exporting…' : 'Export to Excel'}
            </button>
            <Link
              href="/dispensations/create"
              className="inline-flex items-center gap-2 rounded-lg bg-blue-600 px-5 py-2 text-sm font-medium text-white hover:bg-blue-700"
              style={{ cursor: 'pointer' }}
            >
              <Plus className="h-4 w-4" />
              Add Dispensation
            </Link>
          </div>
        </div>

        {/* Filter Section */}
        <div className="mb-6 rounded-lg border p-4">
          <div className="grid grid-cols-1 gap-4 md:grid-cols-2 lg:grid-cols-4">
            {/* Search */}
            <div>
              <label className="mb-1 block text-sm font-medium">Search</label>
              <div className="relative">
                <Search className="text-muted-foreground absolute top-2.5 left-3 h-5 w-5" />
                <input
                  type="text"
                  value={searchQuery}
                  onChange={(e) => setSearchQuery(e.target.value)}
                  placeholder="Search..."
                  className="border-border bg-background placeholder-muted-foreground w-full rounded-lg border py-2 pr-3 pl-10 text-sm focus:ring-2 focus:ring-blue-600 focus:outline-none"
                />
              </div>
            </div>

            {/* Date Range */}
            <div>
              <label className="mb-1 block text-sm font-medium">Date Range</label>
              <div className="grid grid-cols-2 gap-2">
                <input
                  type="date"
                  value={startDate}
                  onChange={(e) => setStartDate(e.target.value)}
                  className="border-border bg-background rounded-lg border p-2 text-sm"
                />
                <input
                  type="date"
                  value={endDate}
                  onChange={(e) => setEndDate(e.target.value)}
                  className="border-border bg-background rounded-lg border p-2 text-sm"
                />
              </div>
            </div>

            {/* Patient */}
            <div>
              <label className="mb-1 block text-sm font-medium">Patient</label>
              <select
                value={patientId}
                onChange={(e) => setPatientId(e.target.value)}
                className="border-border bg-background w-full rounded-lg border p-2 text-sm"
              >
                <option value="">All Patients</option>
                {patients.map((patient) => (
                  <option key={patient.id} value={patient.id}>
                    {`${patient.first_name} ${patient.surname}`}
                  </option>
                ))}
              </select>
            </div>

            {/* Dispensed By */}
            <div>
              <label className="mb-1 block text-sm font-medium">Dispensed By</label>
              <select
                value={dispensedById}
                onChange={(e) => setDispensedById(e.target.value)}
                className="border-border bg-background w-full rounded-lg border p-2 text-sm"
              >
                <option value="">All Users</option>
                {users.map((user) => (
                  <option key={user.id} value={user.id}>
                    {user.name}
                  </option>
                ))}
              </select>
            </div>
          </div>

          <div className="mt-4 flex justify-end gap-2">
            <button
              onClick={handleResetFilters}
              className="rounded-lg bg-gray-200 px-4 py-2 text-sm font-medium text-gray-700 hover:bg-gray-300"
            >
              Reset Filters
            </button>
            <button
              onClick={handleFilter}
              className="rounded-lg bg-blue-600 px-4 py-2 text-sm font-medium text-white hover:bg-blue-700"
            >
              Apply Filters
            </button>
          </div>
        </div>

        {/* Table */}
        <div className="overflow-x-auto rounded-xl border">
          <table className="min-w-full divide-y overflow-hidden rounded-xl bg-white text-sm dark:bg-neutral-900">
            <thead className="bg-muted">
              <tr>
                {['Date', 'Patient Name', 'Medication', 'Quantity', 'Dosage', 'Dispensed By', 'Actions'].map((header) => (
                  <th
                    key={header}
                    className="text-muted-foreground px-4 py-3 text-left text-xs font-semibold tracking-wider uppercase"
                  >
                    {header}
                  </th>
                ))}
              </tr>
            </thead>
            <tbody className="divide-y">
              {dispensations.data.map((dispensation) => (
                <tr key={dispensation.id} className="hover:bg-muted/50 transition">
                  <td className="px-4 py-3">{moment(dispensation.dispensed_at).format('MMMM Do YYYY')}</td>
                  <td className="px-4 py-3">{`${dispensation.patient?.first_name} ${dispensation.patient?.surname}`}</td>
                  <td className="px-4 py-3">
                    {dispensation.clinic_medication_stock?.medication_batch?.medication?.name ||
                      dispensation.nurse_medication_stock?.medication_batch?.medication?.name ||
                      '-'}
                  </td>
                  <td className="px-4 py-3">{dispensation.quantity}</td>
                  <td className="px-4 py-3">{dispensation.route}</td>
                  <td className="px-4 py-3">{dispensation.dispensed_by?.name || 'N/A'}</td>
                  <td className="flex items-center gap-2 px-4 py-3">
                    <Link
                      href={`/dispensations/${dispensation.id}`}
                      className="inline-flex items-center justify-center rounded-full bg-blue-600 p-2 text-white hover:bg-blue-700"
                      title="View"
                    >
                      <Eye className="h-4 w-4" />
                    </Link>

                    {
                      canModify && (
                        <>
                          <Link
                            href={`/dispensations/${dispensation.id}/edit`}
                            className="inline-flex items-center justify-center rounded-full bg-yellow-500 p-2 text-white hover:bg-yellow-600"
                            title="Edit"
                          >
                            <Pencil className="h-4 w-4" />
                          </Link>
                          <button
                            onClick={() => handleDelete(dispensation.id)}
                            className="inline-flex items-center justify-center rounded-full bg-red-600 p-2 text-white hover:bg-red-700"
                            title="Delete"
                          >
                            <Trash className="h-4 w-4" />
                          </button>
                        </>
                      )
                    }

                  </td>
                </tr>
              ))}
            </tbody>
          </table>
        </div>

        {/* Pagination */}
        <div className="mt-6 flex justify-center">
          <ReactPaginate
            pageCount={dispensations.last_page}
            forcePage={dispensations.current_page - 1}
            onPageChange={handlePageChange}
            marginPagesDisplayed={1}
            pageRangeDisplayed={3}
            previousLabel="← Prev"
            nextLabel="Next →"
            breakLabel="..."
            containerClassName="flex items-center gap-2 text-sm"
            pageClassName="px-3 py-1 border border-border rounded hover:bg-muted"
            activeClassName="bg-blue-600 text-white"
            previousClassName="px-3 py-1 border border-border rounded hover:bg-muted"
            nextClassName="px-3 py-1 border border-border rounded hover:bg-muted"
            breakClassName="px-2"
          />
        </div>
      </div>
    </AppLayout>
  );
}
