// resources/js/Pages/Dispensations/CorrectionsIndex.tsx
import { Head, router, usePage } from '@inertiajs/react';
import AppLayout from '@/layouts/app-layout';
import { useMemo, useState } from 'react';
import moment from 'moment';

// shadcn/ui imports (ensure these components exist in your /components/ui folder)
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { Input } from '@/components/ui/input';
import { Textarea } from '@/components/ui/textarea';
import { Label } from '@/components/ui/label';
import {
    Dialog, DialogContent, DialogHeader, DialogTitle, DialogFooter
} from '@/components/ui/dialog';
import {
    Tabs, TabsList, TabsTrigger, TabsContent
} from '@/components/ui/tabs';
import {
    Table, TableBody, TableCell, TableHead, TableHeader, TableRow
} from '@/components/ui/table';

type Paginated<T> = {
    data: T[];
    current_page: number;
    last_page: number;
    per_page: number;
    total: number;
};

type StockInfo = {
    id: number;
    batch_number?: string | null;
    quantity: number;
    medication_name?: string | null;
};

type Patient = {
    id: number;
    first_name: string;
    surname: string;
    employee_number?: string | null;
    company?: { id: number; name: string } | null;
    parent_patient_id?: number | null;
};

type User = { id: number; name: string };

type DispensationLite = {
    id: number;
    consultation_id: number;
    patient_id: number;
    clinic_medication_stock_id?: number | null;
    nurse_medication_stock_id?: number | null;
    quantity: number;
    notes?: string | null;
    dispensed_at?: string | null;
};

type Correction = {
    id: number;
    dispensation_id: number;
    consultation_id: number;
    patient_id: number;
    from_stock_type: 'clinic' | 'nurse';
    from_stock_id: number;
    to_stock_type: 'clinic' | 'nurse';
    to_stock_id: number;
    quantity: number;
    reason?: string | null;
    status: 'pending' | 'approved' | 'rejected' | 'cancelled';
    requested_by: number;
    requested_at: string;
    approved_by?: number | null;
    approved_at?: string | null;
    rejected_by?: number | null;
    rejected_at?: string | null;

    // eager
    dispensation: DispensationLite;
    patient: Patient;
    requester: User;
    approver?: User | null;
};

type PageProps = {
    corrections: Paginated<Correction>;
    filters: {
        status: string;
        search?: string;
        start_date?: string;
        end_date?: string;
        per_page: number;
    };
    clinicStocksMap: Record<number, StockInfo>;
    nurseStocksMap: Record<number, StockInfo>;
    canApprove: boolean;
};

export default function CorrectionsIndex() {
    const { corrections, filters, clinicStocksMap, nurseStocksMap, canApprove } = usePage<PageProps>().props;

    const [statusTab, setStatusTab] = useState(filters.status || 'pending');
    const [search, setSearch] = useState(filters.search || '');
    const [start, setStart] = useState(filters.start_date || '');
    const [end, setEnd] = useState(filters.end_date || '');
    const [perPage, setPerPage] = useState<number>(filters.per_page || 15);

    const [approveOpen, setApproveOpen] = useState(false);
    const [rejectOpen, setRejectOpen] = useState(false);
    const [activeId, setActiveId] = useState<number | null>(null);
    const [rejectNotes, setRejectNotes] = useState('');

    const currentPage = corrections.current_page;
    const lastPage = corrections.last_page;

    const applyFilters = (extra: Record<string, any> = {}) => {
        router.get(
            route('corrections.index'),
            {
                status: statusTab,
                search: search || undefined,
                start_date: start || undefined,
                end_date: end || undefined,
                per_page: perPage,
                page: extra.page || currentPage,
            },
            { preserveState: true, replace: true }
        );
    };

    const onTabChange = (value: string) => {
        setStatusTab(value);
        router.get(
            route('corrections.index'),
            { ...filters, status: value, page: 1 },
            { preserveState: true, replace: true }
        );
    };

    const stockInfo = (type: 'clinic' | 'nurse', id: number | null | undefined) => {
        if (!id) return null;
        return type === 'clinic' ? clinicStocksMap[id] : nurseStocksMap[id];
    };

    const onApprove = (id: number) => { setActiveId(id); setApproveOpen(true); };
    const onReject = (id: number) => { setActiveId(id); setRejectOpen(true); };

    const performApprove = () => {
        if (!activeId) return;
        router.post(route('corrections.approve', activeId), {}, {
            preserveScroll: true,
            onSuccess: () => { setApproveOpen(false); setActiveId(null); },
        });
    };

    const performReject = () => {
        if (!activeId) return;
        router.post(route('corrections.reject', activeId), { rejection_notes: rejectNotes }, {
            preserveScroll: true,
            onSuccess: () => { setRejectOpen(false); setRejectNotes(''); setActiveId(null); },
        });
    };

    const statusBadge = (status: Correction['status']) => {
        const color =
            status === 'pending' ? 'bg-yellow-100 text-yellow-800 dark:bg-yellow-900/25 dark:text-yellow-300'
                : status === 'approved' ? 'bg-green-100 text-green-800 dark:bg-green-900/25 dark:text-green-300'
                    : status === 'rejected' ? 'bg-red-100 text-red-800 dark:bg-red-900/25 dark:text-red-300'
                        : 'bg-gray-100 text-gray-800 dark:bg-gray-800 dark:text-gray-200';
        return <Badge className={color}>{status}</Badge>;
    };

    const fromMedicationName = (c: Correction) => {
        const d = c.dispensation;
        const id = d.clinic_medication_stock_id || d.nurse_medication_stock_id || null;
        const type = d.clinic_medication_stock_id ? 'clinic' : 'nurse';
        const info = stockInfo(type as any, id);
        return info?.medication_name || '—';
    };

    const toMedicationName = (c: Correction) => {
        const info = stockInfo(c.to_stock_type, c.to_stock_id);
        return info?.medication_name || '—';
    };

    return (
        <AppLayout breadcrumbs={[{ title: 'Medication', href: '#' }, { title: 'Corrections', href: '#' }]}>
            <Head title="Medication Corrections" />

            <div className="mx-8 my-6 rounded-xl border bg-background p-6 shadow-sm">
                <div className="mb-4 flex flex-col gap-3 md:flex-row md:items-end md:justify-between">
                    <div>
                        <h1 className="text-2xl font-semibold">Medication Corrections</h1>
                        <p className="text-muted-foreground text-sm">Review, approve, or reject correction requests.</p>
                    </div>

                    <div className="flex flex-col items-stretch gap-2 md:flex-row">
                        <div className="flex items-center gap-2">
                            <Label htmlFor="search" className="text-xs">Search</Label>
                            <Input id="search" value={search} onChange={(e) => setSearch(e.target.value)} placeholder="Patient, employee #, reason..." className="w-64" />
                        </div>
                        <div className="flex items-center gap-2">
                            <Label htmlFor="start" className="text-xs">Start</Label>
                            <Input id="start" type="date" value={start} onChange={(e) => setStart(e.target.value)} />
                        </div>
                        <div className="flex items-center gap-2">
                            <Label htmlFor="end" className="text-xs">End</Label>
                            <Input id="end" type="date" value={end} onChange={(e) => setEnd(e.target.value)} />
                        </div>
                        <div className="flex items-center gap-2">
                            <Label htmlFor="perpage" className="text-xs">Per Page</Label>
                            <Input id="perpage" type="number" min={5} className="w-24" value={perPage} onChange={(e) => setPerPage(parseInt(e.target.value || '15', 10))} />
                        </div>
                        <Button onClick={() => applyFilters({ page: 1 })}>Apply</Button>
                    </div>
                </div>

                <Tabs value={statusTab} onValueChange={onTabChange} className="w-full">
                    <TabsList>
                        <TabsTrigger value="pending">Pending</TabsTrigger>
                        <TabsTrigger value="approved">Approved</TabsTrigger>
                        <TabsTrigger value="rejected">Rejected</TabsTrigger>
                        <TabsTrigger value="all">All</TabsTrigger>
                    </TabsList>

                    <TabsContent value={statusTab} className="mt-4">
                        <div className="overflow-x-auto rounded-md border">
                            <Table>
                                <TableHeader>
                                    <TableRow>
                                        <TableHead>ID</TableHead>
                                        <TableHead>Requested</TableHead>
                                        <TableHead>Patient</TableHead>
                                        <TableHead>From</TableHead>
                                        <TableHead>To</TableHead>
                                        <TableHead>Qty</TableHead>
                                        <TableHead>Reason</TableHead>
                                        <TableHead>Status</TableHead>
                                        <TableHead>Actions</TableHead>
                                    </TableRow>
                                </TableHeader>
                                <TableBody>
                                    {corrections.data.map((c) => {
                                        const patientName = `${c.patient.first_name ?? ''} ${c.patient.surname ?? ''}`.trim();
                                        const company = c.patient.company?.name ? ` • ${c.patient.company?.name}` : '';
                                        const fromName = fromMedicationName(c);
                                        const toName = toMedicationName(c);
                                        return (
                                            <TableRow key={c.id}>
                                                <TableCell className="whitespace-nowrap">#{c.id}</TableCell>
                                                <TableCell className="whitespace-nowrap">
                                                    {moment(c.requested_at).format('DD MMM YYYY, HH:mm')}
                                                    <div className="text-muted-foreground text-xs">by {c.requester?.name || '—'}</div>
                                                </TableCell>
                                                <TableCell className="min-w-[220px]">
                                                    <div className="font-medium">{patientName}</div>
                                                    <div className="text-muted-foreground text-xs">
                                                        {c.patient.employee_number || '—'}{company}
                                                    </div>
                                                </TableCell>
                                                <TableCell className="min-w-[200px]">
                                                    <div className="font-medium">{fromName}</div>
                                                    <div className="text-muted-foreground text-xs">
                                                        {c.from_stock_type} stock #{c.from_stock_id}
                                                    </div>
                                                </TableCell>
                                                <TableCell className="min-w-[200px]">
                                                    <div className="font-medium">{toName}</div>
                                                    <div className="text-muted-foreground text-xs">
                                                        {c.to_stock_type} stock #{c.to_stock_id}
                                                    </div>
                                                </TableCell>
                                                <TableCell>{c.quantity}</TableCell>
                                                <TableCell className="max-w-[260px]">
                                                    <div title={c.reason || ''} className="line-clamp-2">{c.reason || '—'}</div>
                                                </TableCell>
                                                <TableCell>{statusBadge(c.status)}</TableCell>
                                                <TableCell className="space-x-2">
                                                    {c.status === 'pending' ? (
                                                        <>
                                                            <Button size="sm" onClick={() => onApprove(c.id)} style={{
                                                                cursor: 'pointer'
                                                            }}>Approve</Button>
                                                            <Button size="sm" variant="destructive" onClick={() => onReject(c.id)}
                                                                style={{
                                                                    cursor: 'pointer'
                                                                }}
                                                            >Reject</Button>
                                                        </>
                                                    ) : (
                                                        <span className="text-muted-foreground text-xs">—</span>
                                                    )}
                                                </TableCell>
                                            </TableRow>
                                        );
                                    })}
                                </TableBody>
                            </Table>
                        </div>

                        {/* Pagination */}
                        <div className="mt-4 flex items-center justify-between text-sm">
                            <div>
                                Page {currentPage} of {lastPage} • {corrections.total} total
                            </div>
                            <div className="space-x-2">
                                <Button
                                    variant="outline"
                                    disabled={currentPage <= 1}
                                    onClick={() => applyFilters({ page: currentPage - 1 })}
                                >
                                    Previous
                                </Button>
                                <Button
                                    variant="outline"
                                    disabled={currentPage >= lastPage}
                                    onClick={() => applyFilters({ page: currentPage + 1 })}
                                >
                                    Next
                                </Button>
                            </div>
                        </div>
                    </TabsContent>
                </Tabs>
            </div>

            {/* Approve dialog */}
            <Dialog open={approveOpen} onOpenChange={setApproveOpen}>
                <DialogContent>
                    <DialogHeader>
                        <DialogTitle>Approve Correction</DialogTitle>
                    </DialogHeader>
                    <p className="text-sm text-muted-foreground">
                        This will <strong>return</strong> the wrong medication to stock and <strong>dispense</strong> the correct medication for the same patient/consultation.
                    </p>
                    <DialogFooter>
                        <Button variant="outline" onClick={() => setApproveOpen(false)}>Cancel</Button>
                        <Button onClick={performApprove}>Approve</Button>
                    </DialogFooter>
                </DialogContent>
            </Dialog>

            {/* Reject dialog */}
            <Dialog open={rejectOpen} onOpenChange={setRejectOpen}>
                <DialogContent>
                    <DialogHeader>
                        <DialogTitle>Reject Correction</DialogTitle>
                    </DialogHeader>
                    <div className="space-y-2">
                        <Label htmlFor="rejection_notes">Reason (optional)</Label>
                        <Textarea
                            id="rejection_notes"
                            value={rejectNotes}
                            onChange={(e) => setRejectNotes(e.target.value)}
                            placeholder="Add an explanation for rejecting this correction"
                            rows={4}
                        />
                    </div>
                    <DialogFooter>
                        <Button variant="outline" onClick={() => setRejectOpen(false)}>Cancel</Button>
                        <Button variant="destructive" onClick={performReject}>Reject</Button>
                    </DialogFooter>
                </DialogContent>
            </Dialog>
        </AppLayout>
    );
}
