import AppLayout from '@/layouts/app-layout';
import { Button } from '@/components/ui/button';
import { Head, router, usePage } from '@inertiajs/react';
import moment from 'moment';
import Swal from 'sweetalert2';
import { Pill, ExternalLink, ArrowLeft } from 'lucide-react';

type Stock = {
  id: number;
  quantity: number;
  medication_batch: {
    batch_number?: string | null;
    expiry_date?: string | null;
    medication: { name: string; dosage?: string | null; unit?: string | null; form?: string | null };
  };
};
type Disp = {
  id: number;
  quantity: number;
  frequency?: string | null;
  route?: string | null;
  notes?: string | null;
  dispensed_at: string | null;
  clinic_medication_stock?: Stock | null;
  nurse_medication_stock?: Stock | null;
};

interface Props {
  consultation: {
    id: number | string;
    patient: { first_name: string; surname: string };
    clinic?: { name?: string | null };
    doctor?: { name?: string | null };
    consultation_date?: string | null;
  };
  dispensations: Disp[];
  clinicMedicationStocks: Stock[];
  nurseMedicationStocks: Stock[];
}

export default function EditDispensed() {
  const { consultation, dispensations } = usePage<Props>().props;

  const handleDeleteAndReturn = async (d: Disp) => {
    const fromClinic = !!d.clinic_medication_stock;
    const stockObj = fromClinic ? d.clinic_medication_stock! : d.nurse_medication_stock!;
    const medName = stockObj.medication_batch.medication.name;
    const batchNo = stockObj.medication_batch.batch_number ?? '-';
    const currentQty = Number(stockObj.quantity ?? 0);
    const returnQty = Number(d.quantity ?? 0);

    const { isConfirmed } = await Swal.fire({
      title: 'Delete & Return to Stock?',
      html: `
        <div style="text-align:left">
          <p><strong>Medication:</strong> ${medName}</p>
          <p><strong>Source:</strong> ${fromClinic ? 'Clinic' : 'Nurse'} stock #${stockObj.id}</p>
          <p><strong>Batch:</strong> ${batchNo}</p>
          <hr />
          <p><strong>Current Stock:</strong> ${currentQty}</p>
          <p><strong>Return Qty:</strong> ${returnQty}</p>
          <p><strong>New Stock:</strong> ${currentQty + returnQty}</p>
        </div>
      `,
      icon: 'warning',
      showCancelButton: true,
      confirmButtonText: 'Yes, delete & return',
      cancelButtonText: 'Cancel',
      confirmButtonColor: '#dc2626',
      reverseButtons: true,
      focusCancel: true,
    });
    if (!isConfirmed) return;

    Swal.fire({ title: 'Processing…', allowOutsideClick: false, didOpen: () => Swal.showLoading() });
    router.delete(route('dispensations.delete_return', d.id), {
      preserveScroll: true,
      onSuccess: () => {
        Swal.close();
        Swal.fire('Done', 'Dispensation deleted and returned.', 'success');
        router.reload({ only: ['dispensations'] });
      },
      onError: (e) => {
        Swal.close();
        const msg = e && typeof e === 'object' ? Object.values(e).join('\n') : 'Failed to delete.';
        Swal.fire('Error', String(msg || 'Failed.'), 'error');
      },
    });
  };

  return (
    <AppLayout
      breadcrumbs={[
        { title: 'Consultations', href: route('consultations.index') },
        { title: `Consultation #${consultation.id}`, href: route('consultations.show', consultation.id) },
        { title: 'Edit Dispensed', href: '#' },
      ]}
    >
      <Head title="Edit Dispensed Medication" />
      <div className="mx-8 my-6 rounded-xl border bg-card p-6 shadow-sm">
        <div className="mb-4 flex items-center justify-between">
          <h1 className="text-2xl font-bold">Edit Dispensed Medication</h1>
          <div className="flex gap-2">
            <Button variant="outline" onClick={() => router.visit(route('consultations.show', consultation.id))}>
              <ArrowLeft className="mr-2 h-4 w-4" />
              Back to Consultation
            </Button>
            <Button variant="outline" onClick={() => router.visit(route('consultations.dispense', consultation.id))}>
              <ExternalLink className="mr-2 h-4 w-4" />
              Go to Dispense Screen
            </Button>
          </div>
        </div>

        <div className="mb-6 grid grid-cols-1 gap-4 md:grid-cols-3">
          <div>
            <div className="text-sm text-muted-foreground">Patient</div>
            <div className="font-medium capitalize">
              {consultation.patient.first_name} {consultation.patient.surname}
            </div>
          </div>
          <div>
            <div className="text-sm text-muted-foreground">Doctor/Clinic</div>
            <div className="font-medium">
              {consultation.doctor?.name || '-'} • {consultation.clinic?.name || '-'}
            </div>
          </div>
          <div>
            <div className="text-sm text-muted-foreground">Date</div>
            <div className="font-medium">{consultation.consultation_date || '-'}</div>
          </div>
        </div>

        <div className="rounded-lg border p-4">
          <h2 className="mb-4 flex items-center gap-2 text-lg font-semibold">
            <Pill className="h-5 w-5 text-muted-foreground" />
            Dispensed Items
          </h2>

          {dispensations.length === 0 ? (
            <p className="text-sm text-muted-foreground">No medications dispensed.</p>
          ) : (
            <div className="overflow-x-auto">
              <table className="min-w-full divide-y text-sm">
                <thead className="bg-muted">
                  <tr>
                    <th className="px-3 py-2 text-left">Medication</th>
                    <th className="px-3 py-2 text-left">Qty</th>
                    <th className="px-3 py-2 text-left">Frequency</th>
                    <th className="px-3 py-2 text-left">Route</th>
                    <th className="px-3 py-2 text-left">Date</th>
                    <th className="px-3 py-2 text-left">Actions</th>
                  </tr>
                </thead>
                <tbody className="divide-y">
                  {dispensations.map((d) => {
                    const stock = d.clinic_medication_stock ?? d.nurse_medication_stock!;
                    const med = stock.medication_batch.medication;
                    const medLine = `${med.name}`;
                    const meta = [
                      med.dosage ? `${med.dosage}${med.unit ? ` ${med.unit}` : ''}` : null,
                      med.form || null,
                      stock.medication_batch.batch_number ? `Batch ${stock.medication_batch.batch_number}` : null,
                    ]
                      .filter(Boolean)
                      .join(' • ');

                    return (
                      <tr key={d.id} className="hover:bg-muted/50">
                        <td className="px-3 py-2">
                          <div className="font-medium">{medLine}</div>
                          <div className="text-xs text-muted-foreground">{meta || '-'}</div>
                        </td>
                        <td className="px-3 py-2">{d.quantity}</td>
                        <td className="px-3 py-2">{d.frequency || '-'}</td>
                        <td className="px-3 py-2">{d.route || '-'}</td>
                        <td className="px-3 py-2">{d.dispensed_at ? moment(d.dispensed_at).format('DD MMM YYYY') : '-'}</td>
                        <td className="px-3 py-2">
                          <div className="flex gap-2">
                            <Button
                              variant="outline"
                              onClick={() => router.visit(route('dispensations.corrections.create', d.id))}
                            >
                              Request Correction
                            </Button>
                            <Button variant="destructive" onClick={() => handleDeleteAndReturn(d)}>
                              Delete & Return
                            </Button>
                          </div>
                        </td>
                      </tr>
                    );
                  })}
                </tbody>
              </table>
            </div>
          )}
        </div>
      </div>
    </AppLayout>
  );
}
