// resources/js/Pages/Dispensations/RequestCorrection.tsx
import AppLayout from '@/layouts/app-layout';
import { Head, router, usePage } from '@inertiajs/react';
import { useMemo, useState } from 'react';
import Select, { StylesConfig } from 'react-select';
import Swal from 'sweetalert2';
import { ArrowLeft } from 'lucide-react';

// shadcn/ui
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { Textarea } from '@/components/ui/textarea';
import { Card, CardContent } from '@/components/ui/card';

type Stock = {
  id: number;
  quantity: number;
  medication_batch: {
    batch_number?: string | null;
    medication: { name: string; dosage?: string | null; unit?: string | null; form?: string | null };
  };
};

type Disp = {
  id: number | string;
  quantity: number;
  clinic_medication_stock?: Stock | null;
  nurse_medication_stock?: Stock | null;
  consultation_id: number | string;
};

interface Props {
  dispensation: Disp;
  consultation: { id: number | string; patient: { first_name: string; surname: string } };
  clinicMedicationStocks: Stock[];
  nurseMedicationStocks: Stock[];
}

/** UUID for idempotency */
const uuid = () =>
  (crypto as any)?.randomUUID
    ? (crypto as any).randomUUID()
    : `${Date.now()}-${Math.random().toString(16).slice(2)}`;

/** Read CSS variables to theme react-select for light/dark */
function useThemeTokens() {
  const read = (name: string, fallback: string) =>
    typeof window !== 'undefined'
      ? getComputedStyle(document.documentElement).getPropertyValue(name).trim() || fallback
      : fallback;

  const bg = read('--background', '#ffffff');
  const fg = read('--foreground', '#0a0a0a');
  const border = read('--border', '220 13% 91%');
  const muted = read('--muted', '220 14% 96%');
  const ring = read('--ring', '221 83% 53%');

  const hsl = (triple: string) => (triple.includes('%') ? `hsl(${triple})` : triple);
  return { bg, fg, border: hsl(border), muted: hsl(muted), ring: hsl(ring) };
}

function makeSelectStyles(tokens: ReturnType<typeof useThemeTokens>): StylesConfig<any, false> {
  return {
    container: (base) => ({ ...base, width: '100%' }),
    control: (base, state) => ({
      ...base,
      minHeight: 40,
      borderRadius: 8,
      backgroundColor: tokens.bg,
      borderColor: state.isFocused ? tokens.ring : tokens.border,
      boxShadow: state.isFocused ? `0 0 0 2px ${tokens.ring}22` : 'none',
      ':hover': { borderColor: state.isFocused ? tokens.ring : tokens.border },
    }),
    valueContainer: (base) => ({ ...base, paddingBlock: 6, paddingInline: 10 }),
    input: (base) => ({ ...base, color: tokens.fg }),
    singleValue: (base) => ({ ...base, color: tokens.fg }),
    placeholder: (base) => ({ ...base, color: `${tokens.fg}99` }),
    menu: (base) => ({
      ...base,
      zIndex: 60,
      backgroundColor: tokens.bg,
      border: `1px solid ${tokens.border}`,
      boxShadow: `0 8px 24px rgba(0,0,0,.12)`,
      overflow: 'hidden',
      borderRadius: 8,
    }),
    menuList: (base) => ({ ...base, paddingBlock: 4 }),
    option: (base, state) => ({
      ...base,
      backgroundColor: state.isSelected
        ? `${tokens.ring}22`
        : state.isFocused
          ? tokens.muted
          : 'transparent',
      color: tokens.fg,
      cursor: 'pointer',
    }),
  };
}

export default function RequestCorrection() {
  const { dispensation, consultation, clinicMedicationStocks, nurseMedicationStocks } = usePage<Props>().props;

  const tokens = useThemeTokens();
  const selectStyles = makeSelectStyles(tokens);

  const fromType: 'clinic' | 'nurse' = dispensation.clinic_medication_stock ? 'clinic' : 'nurse';
  const [toType, setToType] = useState<'clinic' | 'nurse'>(fromType === 'clinic' ? 'nurse' : 'clinic');

  const [toStockId, setToStockId] = useState<number | null>(null);
  const [quantity, setQuantity] = useState<number>(Math.max(1, Number(dispensation.quantity || 1)));
  const [reason, setReason] = useState<string>('');
  const [submitting, setSubmitting] = useState(false);

  const medName =
    dispensation.clinic_medication_stock?.medication_batch.medication.name ||
    dispensation.nurse_medication_stock?.medication_batch.medication.name ||
    'Medication';

  const dosage =
    dispensation.clinic_medication_stock?.medication_batch.medication.dosage ??
    dispensation.nurse_medication_stock?.medication_batch.medication.dosage ??
    null;

  const unit =
    dispensation.clinic_medication_stock?.medication_batch.medication.unit ??
    dispensation.nurse_medication_stock?.medication_batch.medication.unit ??
    null;

  const form =
    dispensation.clinic_medication_stock?.medication_batch.medication.form ??
    dispensation.nurse_medication_stock?.medication_batch.medication.form ??
    null;

  const batchNo =
    dispensation.clinic_medication_stock?.medication_batch?.batch_number ??
    dispensation.nurse_medication_stock?.medication_batch?.batch_number ??
    '-';

  const fromStockId =
    fromType === 'clinic'
      ? dispensation.clinic_medication_stock?.id
      : dispensation.nurse_medication_stock?.id;

  // --- Only change below: include dosage (and unit, if present) in the dropdown labels ---
  const clinicOptions = useMemo(
    () =>
      clinicMedicationStocks.map((s) => {
        const d = s.medication_batch.medication.dosage;
        const u = s.medication_batch.medication.unit;
        const dose = d ? `${d}${u ? ` ${u}` : ''}` : '';
        return {
          value: s.id,
          label: `${s.medication_batch.medication.name}${dose ? ` — ${dose}` : ''} — batch ${s.medication_batch.batch_number || '-'} (${s.quantity} avail)`,
        };
      }),
    [clinicMedicationStocks]
  );

  const nurseOptions = useMemo(
    () =>
      nurseMedicationStocks.map((s) => {
        const d = s.medication_batch.medication.dosage;
        const u = s.medication_batch.medication.unit;
        const dose = d ? `${d}${u ? ` ${u}` : ''}` : '';
        return {
          value: s.id,
          label: `${s.medication_batch.medication.name}${dose ? ` — ${dose}` : ''} — batch ${s.medication_batch.batch_number || '-'} (${s.quantity} avail)`,
        };
      }),
    [nurseMedicationStocks]
  );
  // --- end minimal change ---

  const onSubmit = () => {
    if (submitting) return;

    // Client-side checks
    if (!toStockId) {
      Swal.fire('Missing selection', 'Please choose a target stock item.', 'warning');
      return;
    }
    if (toType === fromType && Number(toStockId) === Number(fromStockId)) {
      Swal.fire('Invalid target', 'Choose a different stock as the correction target.', 'error');
      return;
    }
    if (!quantity || quantity < 1) {
      Swal.fire('Invalid quantity', 'Quantity must be at least 1.', 'warning');
      return;
    }
    if (quantity > Number(dispensation.quantity)) {
      Swal.fire('Too large', `Quantity must be ≤ originally dispensed (${dispensation.quantity}).`, 'warning');
      return;
    }

    setSubmitting(true);

    // Non-blocking loading modal (no await!)
    Swal.fire({
      title: 'Submitting…',
      allowOutsideClick: false,
      allowEscapeKey: false,
      didOpen: () => Swal.showLoading(),
    });

    router.post(
      route('dispensations.corrections.store', dispensation.id),
      {
        to_stock_type: toType,
        to_stock_id: toStockId,
        quantity,
        reason: reason || '',
        request_uuid: uuid(),
      },
      {
        preserveScroll: true,
        onSuccess: () => {
          Swal.close();
          Swal.fire('Sent', 'Correction request submitted.', 'success');
          router.visit(route('consultations.dispensed.edit', consultation.id));
        },
        onError: (errors) => {
          Swal.close();
          const msg =
            (errors && typeof errors === 'object' ? Object.values(errors).join('\n') : null) ||
            'Failed to submit correction.';
          Swal.fire('Error', msg, 'error');
        },
        onFinish: () => setSubmitting(false),
      }
    );
  };

  return (
    <AppLayout
      breadcrumbs={[
        { title: 'Consultations', href: route('consultations.index') },
        { title: `Consultation #${consultation.id}`, href: route('consultations.show', consultation.id) },
        { title: 'Edit Dispensed', href: route('consultations.dispensed.edit', consultation.id) },
        { title: 'Request Correction', href: '#' },
      ]}
    >
      <Head title="Request Dispensation Correction" />

      <div className="mx-4 my-6 rounded-xl border bg-card p-4 shadow-sm sm:mx-8 sm:p-6">
        <div className="mb-4 flex flex-col gap-3 sm:flex-row sm:items-center sm:justify-between">
          <h1 className="text-2xl font-bold">Request Dispensation Correction</h1>
          <Button variant="outline" onClick={() => router.visit(route('consultations.dispensed.edit', consultation.id))}>
            <ArrowLeft className="mr-2 h-4 w-4" />
            Back
          </Button>
        </div>

        {/* Original dispensation summary (now shows batch explicitly) */}
        <Card className="mb-6">
          <CardContent className="p-4 sm:p-6">
            <div className="mb-2 text-sm font-medium">Original Dispensation</div>
            <div className="grid grid-cols-1 gap-2 text-sm md:grid-cols-2 lg:grid-cols-3">
              <div>
                <div className="text-muted-foreground">Medication</div>
                <div className="font-medium">
                  {medName}
                  {(dosage || unit) && (
                    <span className="text-muted-foreground"> — {dosage}{unit ? ` ${unit}` : ''}</span>
                  )}
                  {form ? <span className="text-muted-foreground"> • {form}</span> : null}
                </div>
              </div>

              <div>
                <div className="text-muted-foreground">From Stock</div>
                <div className="font-medium">
                  {fromType === 'clinic' ? 'Clinic' : 'Nurse'} #{fromStockId ?? '-'}
                </div>
              </div>

              <div>
                <div className="text-muted-foreground">Batch</div>
                <div className="font-medium">#{batchNo}</div>
              </div>

              <div>
                <div className="text-muted-foreground">Dispensed Qty</div>
                <div className="font-medium">{dispensation.quantity}</div>
              </div>
            </div>
          </CardContent>
        </Card>

        {/* Tabs swap target stock type */}
        <Tabs
          value={toType}
          onValueChange={(v) => {
            setToType(v as 'clinic' | 'nurse');
            setToStockId(null);
          }}
          className="w-full"
        >
          <TabsList className="grid w-full grid-cols-2">
            <TabsTrigger value="clinic">Correct To: Clinic Stock</TabsTrigger>
            <TabsTrigger value="nurse">Correct To: Nurse Stock</TabsTrigger>
          </TabsList>

          <TabsContent value="clinic" className="mt-4">
            <div className="grid grid-cols-1 gap-6 md:grid-cols-2">
              <div className="space-y-2">
                <Label>Target Stock Item</Label>
                <Select
                  instanceId="clinic_stock"
                  styles={selectStyles}
                  classNamePrefix="rs"
                  placeholder="Search clinic stock…"
                  options={clinicOptions}
                  value={toStockId ? clinicOptions.find((o) => o.value === toStockId) || null : null}
                  onChange={(opt: any) => setToStockId(opt?.value ?? null)}
                  menuPortalTarget={typeof document !== 'undefined' ? document.body : undefined}
                  menuPosition="fixed"
                />
              </div>

              <div className="space-y-2">
                <Label htmlFor="qty">Quantity to Correct</Label>
                <Input
                  id="qty"
                  type="number"
                  min={1}
                  max={Number(dispensation.quantity)}
                  value={String(quantity)}
                  onChange={(e) => setQuantity(Math.max(1, Number(e.target.value || 1)))}
                />
                <p className="text-xs text-muted-foreground">
                  Must be ≤ originally dispensed ({dispensation.quantity})
                </p>
              </div>

              <div className="md:col-span-2 space-y-2">
                <Label htmlFor="reason">Reason (optional)</Label>
                <Textarea
                  id="reason"
                  rows={4}
                  placeholder="Brief reason for the correction"
                  value={reason}
                  onChange={(e) => setReason(e.target.value)}
                />
              </div>
            </div>
          </TabsContent>

          <TabsContent value="nurse" className="mt-4">
            <div className="grid grid-cols-1 gap-6 md:grid-cols-2">
              <div className="space-y-2">
                <Label>Target Stock Item</Label>
                <Select
                  instanceId="nurse_stock"
                  styles={selectStyles}
                  classNamePrefix="rs"
                  placeholder="Search nurse stock…"
                  options={nurseOptions}
                  value={toStockId ? nurseOptions.find((o) => o.value === toStockId) || null : null}
                  onChange={(opt: any) => setToStockId(opt?.value ?? null)}
                  menuPortalTarget={typeof document !== 'undefined' ? document.body : undefined}
                  menuPosition="fixed"
                />
              </div>

              <div className="space-y-2">
                <Label htmlFor="qty2">Quantity to Correct</Label>
                <Input
                  id="qty2"
                  type="number"
                  min={1}
                  max={Number(dispensation.quantity)}
                  value={String(quantity)}
                  onChange={(e) => setQuantity(Math.max(1, Number(e.target.value || 1)))}
                />
                <p className="text-xs text-muted-foreground">
                  Must be ≤ originally dispensed ({dispensation.quantity})
                </p>
              </div>

              <div className="md:col-span-2 space-y-2">
                <Label htmlFor="reason2">Reason (optional)</Label>
                <Textarea
                  id="reason2"
                  rows={4}
                  placeholder="Brief reason for the correction"
                  value={reason}
                  onChange={(e) => setReason(e.target.value)}
                />
              </div>
            </div>
          </TabsContent>
        </Tabs>

        <div className="mt-6 flex justify-end gap-3">
          <Button
            variant="outline"
            onClick={() => router.visit(route('consultations.dispensed.edit', consultation.id))}
            disabled={submitting}
          >
            Cancel
          </Button>
          <Button onClick={onSubmit} disabled={submitting}>
            {submitting ? 'Submitting…' : 'Submit Request'}
          </Button>
        </div>
      </div>
    </AppLayout>
  );
}
