import AppLayout from '@/layouts/app-layout';
import { BreadcrumbItem } from '@/types';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Badge } from '@/components/ui/badge';  // Assuming you have a Badge component
import { Textarea } from '@/components/ui/textarea';
import { Head, useForm, usePage, router } from '@inertiajs/react';
import Select from 'react-select';
import { useState, useEffect } from 'react';
import moment from 'moment';
import PrintSickLeaveNote from '../Consultations/components/PrintSickLeaveNote';

export default function DoctorNoteCreate() {
    const { consultation } = usePage<{ consultation: any }>().props;

    const { data, setData, post, errors, processing } = useForm({
        consultation_id: consultation.id,
        patient_id: consultation?.patient.id,
        title: '',
        body: '',
        note_type: 'progress',
        private: false as boolean,
        meta: '',
    });

    const [isDarkMode, setIsDarkMode] = useState(false);

    useEffect(() => {
        const checkDarkMode = () => {
            setIsDarkMode(document.documentElement.classList.contains('dark'));
        };
        checkDarkMode();

        const observer = new MutationObserver(checkDarkMode);
        observer.observe(document.documentElement, {
            attributes: true,
            attributeFilter: ['class'],
        });

        return () => observer.disconnect();
    }, []);

    const noteTypeOptions = [
        { value: 'Progress', label: 'Progress Note' },
        { value: 'Discharge', label: 'Discharge Summary' },
        { value: 'Prescription', label: 'Prescription' },
        { value: 'Other', label: 'Other' },
    ];

    const selectStyles = {
        control: (base: any) => ({
            ...base,
            backgroundColor: isDarkMode ? '#000' : '#fff',
            borderColor: isDarkMode ? '#374151' : '#d1d5db',
            color: isDarkMode ? '#fff' : '#000',
            '&:hover': {
                borderColor: isDarkMode ? '#4b5563' : '#9ca3af',
            },
        }),
        menu: (base: any) => ({
            ...base,
            backgroundColor: isDarkMode ? '#000' : '#fff',
            color: isDarkMode ? '#fff' : '#000',
        }),
        option: (base: any, { isFocused }: any) => ({
            ...base,
            backgroundColor: isFocused ? (isDarkMode ? '#1e40af' : '#3b82f6') : 'transparent',
            color: isFocused ? '#fff' : isDarkMode ? '#fff' : '#000',
            '&:active': {
                backgroundColor: isDarkMode ? '#1e40af' : '#3b82f6',
            },
        }),
        singleValue: (base: any) => ({
            ...base,
            color: isDarkMode ? '#fff' : '#000',
        }),
        input: (base: any) => ({
            ...base,
            color: isDarkMode ? '#fff' : '#000',
        }),
        placeholder: (base: any) => ({
            ...base,
            color: isDarkMode ? '#9ca3af' : '#6b7280',
        }),
    };

    const Info = ({ label, value }: { label: string; value?: string | null }) => (
        <div>
            <div className="text-muted-foreground text-sm">{label}</div>
            <div className="text-foreground font-medium">{value || '-'}</div>
        </div>
    );

    const InfoGrid = ({ items }: { items: { label: string; value?: string | null }[] }) => (
        <div className="grid grid-cols-1 gap-4 sm:grid-cols-2 md:grid-cols-3">
            {items.map((item) => (
                <Info key={item.label} {...item} />
            ))}
        </div>
    );

    const Section = ({ title, children }: { title: string; children: React.ReactNode }) => (
        <section className="bg-card rounded-lg border p-6 shadow-sm dark:bg-neutral-900">
            <h2 className="text-foreground mb-4 text-xl font-semibold">{title}</h2>
            {children}
        </section>
    );


    const handleSubmit = (e: React.FormEvent) => {
        e.preventDefault();
        post(route('doctor-notes.store'));
    };

    const breadcrumbs: BreadcrumbItem[] = [
        { title: 'Consultations', href: route('consultations.index') },
        { title: 'Doctor Notes', href: '#' },
        { title: 'Create Note', href: route('consultations.doctor-notes.create', consultation.id) },
    ];

    return (
        <AppLayout breadcrumbs={breadcrumbs}>
            <Head title="Create Doctor Note" />
            <div className="w-full px-4 py-6 sm:px-6">
                <div className="grid grid-cols-12 gap-6">
                    {/* Left side: Consultation Information (4 columns) */}
                    <div className="col-span-12 sm:col-span-4">
                        <Section title="Consultation Information">
                            {consultation ? (
                                <>
                                    <InfoGrid
                                        items={[
                                            {
                                                label: 'Date',
                                                value: moment(consultation.consultation_date).format('DD MMM YYYY, h:mm a'),
                                            },
                                            {
                                                label: 'Doctor',
                                                value: consultation.doctor?.name || '-',
                                            },
                                            {
                                                label: 'Clinic',
                                                value: consultation.clinic?.name || '-',
                                            },
                                            {
                                                label: 'Height',
                                                value: consultation.height ? `${consultation.height} cm` : '-',
                                            },
                                            {
                                                label: 'Weight',
                                                value: consultation.weight ? `${consultation.weight} kg` : '-',
                                            },
                                            {
                                                label: 'BMI',
                                                value: consultation.bmi ? consultation.bmi : '-',
                                            },
                                            {
                                                label: 'Injury on Duty',
                                                value: consultation.injury_on_duty ? (
                                                    <Badge variant="destructive">Yes</Badge>
                                                ) : (
                                                    <Badge variant="secondary">No</Badge>
                                                ),
                                            },
                                            {
                                                label: 'Systolic BP',
                                                value: consultation.systolic ? consultation.systolic : '-',
                                            },
                                            {
                                                label: 'Diastolic BP',
                                                value: consultation.diastolic ? consultation.diastolic : '-',
                                            },
                                        ]}
                                    />

                                    {consultation.presenting_complaint && (
                                        <div className="mt-4">
                                            <div className="text-muted-foreground text-sm">Presenting Complaint</div>
                                            <div className="text-foreground rounded border p-3 font-medium">
                                                {consultation.presenting_complaint}
                                            </div>
                                        </div>
                                    )}

                                    {consultation.diagnosis && (
                                        <div className="my-4">
                                            <div className="text-muted-foreground text-sm">Diagnosis</div>
                                            <div className="text-foreground rounded border p-3 font-medium">
                                                {consultation.diagnosis}
                                            </div>
                                        </div>
                                    )}

                                    {consultation.clinical_notes && (
                                        <div className="my-4">
                                            <div className="text-muted-foreground text-sm">Clinical Notes</div>
                                            <div className="text-foreground rounded border p-3 font-medium">
                                                {consultation.clinical_notes}
                                            </div>
                                        </div>
                                    )}

                                    {/* Sick leave details */}
                                    {consultation.sick_leave ? (
                                        <div className="my-4">
                                            <div className="text-muted-foreground text-sm">Sick Leave Notes</div>
                                            <div className="text-foreground rounded border p-3 font-medium">
                                                {consultation.sick_leave_notes || 'No notes provided'}
                                            </div>

                                            <div className="mt-2 flex space-x-4">
                                                <div>
                                                    <div className="text-muted-foreground text-sm">Number of Days</div>
                                                    <div className="text-foreground font-medium">{consultation.number_of_days}</div>
                                                </div>
                                                <div>
                                                    <div className="text-muted-foreground text-sm">Reason for Leave</div>
                                                    <div className="text-foreground font-medium">{consultation.reason_for_leave}</div>
                                                </div>
                                            </div>
                                        </div>
                                    ) : (
                                        <p className="text-muted-foreground mt-4">No sick leave note available</p>
                                    )}
                                    <div style={{
                                        marginTop: '10px',

                                    }}></div>
                                    {
                                        consultation.sick_leave ? (
                                            <PrintSickLeaveNote
                                                consultation={consultation}
                                            />
                                        ) : (
                                            <p className="text-muted-foreground">No sick leave note available</p>
                                        )
                                    }

                                    <Button
                                        variant="outline"
                                        className="w-full mt-6"
                                        onClick={() => router.visit(route('consultations.show', consultation.id))}
                                        style={{ cursor: 'pointer' }}
                                    >
                                        View Full Consultation
                                    </Button>
                                </>
                            ) : (
                                <div className="text-muted-foreground">No consultation found</div>
                            )}
                        </Section>
                    </div>

                    {/* Right side: Doctor Note Form (8 columns) */}
                    <div className="col-span-12 sm:col-span-8">
                        <Section title="Patient Information">
                            <InfoGrid
                                items={[
                                    { label: 'Full Name', value: consultation.patient?.first_name + ' ' + consultation.patient?.surname },
                                    { label: 'Date of Birth', value: consultation.patient?.date_of_birth ? moment(consultation?.patient?.date_of_birth).format('DD MMM YYYY') : '-' },
                                    { label: 'Gender', value: consultation?.patient?.gender },
                                ]}
                            />
                        </Section>
                        <form onSubmit={handleSubmit} className="space-y-8 mt-3">
                            <div className="bg-card rounded-lg border p-6 shadow-sm dark:bg-neutral-900">
                                <h2 className="mb-4 text-xl font-semibold">Doctor's Examination Notes</h2>

                                <div className="space-y-2">
                                    <Label htmlFor="title">Title</Label>
                                    <Input
                                        id="title"
                                        value={data.title}
                                        onChange={(e) => setData('title', e.target.value)}
                                    />
                                    {errors.title && <p className="text-sm text-red-500">{errors.title}</p>}
                                </div>

                                <div className="space-y-2 mt-4">
                                    <Label htmlFor="body">Body</Label>
                                    <Textarea
                                        id="body"
                                        rows={6}
                                        style={{ height: '200px' }}
                                        value={data.body}
                                        onChange={(e) => setData('body', e.target.value)}
                                    />
                                    {errors.body && <p className="text-sm text-red-500">{errors.body}</p>}
                                </div>

                                <div className="space-y-2 mt-4">
                                    <Label htmlFor="note_type">Note Type</Label>
                                    <Select
                                        options={noteTypeOptions}
                                        value={noteTypeOptions.find(option => option.value === data.note_type)}
                                        onChange={(option) => setData('note_type', option ? option.value : '')}
                                        styles={selectStyles}
                                    />
                                    {errors.note_type && <p className="text-sm text-red-500">{errors.note_type}</p>}
                                </div>

                                <div className="space-y-2 mt-4">
                                    <label className="flex items-center space-x-2">
                                        <input
                                            type="checkbox"
                                            checked={data.private}
                                            onChange={(e) => setData('private', e.target.checked)}
                                        />
                                        <span>Private Note (only visible to doctor/clinic)</span>
                                    </label>
                                </div>
                            </div>

                            <div className="flex flex-col justify-end gap-4 sm:flex-row">
                                <Button type="button" variant="outline" onClick={() => window.history.back()}>
                                    Cancel
                                </Button>
                                <Button type="submit" disabled={processing}>
                                    {processing ? 'Saving...' : 'Save'}
                                </Button>
                            </div>
                        </form>
                    </div>
                </div>
            </div>
        </AppLayout>
    );
}
