import AppLayout from '@/layouts/app-layout'
import { Head, router, usePage } from '@inertiajs/react'
import type { PageProps as InertiaPageProps } from '@inertiajs/core'
import { useEffect, useRef, useState } from 'react'
import moment from 'moment'
import { Search, StickyNote, ShieldAlert, User, Eye } from 'lucide-react'

import { Button } from '@/components/ui/button'
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card'
import { Input } from '@/components/ui/input'
import { Badge } from '@/components/ui/badge'
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select'
import { Table, TableHeader, TableRow, TableHead, TableBody, TableCell } from '@/components/ui/table'

interface Doctor { id: number; name: string; email?: string | null }
interface Patient { id: number; first_name: string; surname: string; employee_number?: string | null }
interface ConsultationMini { id: number; consultation_date?: string | null }
interface Note {
    id: number; title: string | null; body: string; note_type: 'Progress' | 'Discharge' | 'Prescription' | 'Other';
    private: boolean; created_at: string; doctor?: Doctor | null; patient?: Patient | null; consultation?: ConsultationMini | null;
}
interface Pagination<T> { data: T[]; current_page: number; last_page: number; per_page: number; total: number }
interface Props extends InertiaPageProps {
    notes: Pagination<Note>;
    filters: { search?: string; note_type?: string | null; private?: string | null; only_mine?: boolean };
    noteTypes: string[]; canCreate: boolean;
}

const NOTE_TYPE_ALL = '__all_types__'
const PRIVACY_ALL = '__all_privacy__'

function useDebounced<T>(v: T, ms = 450) {
    const [d, setD] = useState(v)
    useEffect(() => { const id = setTimeout(() => setD(v), ms); return () => clearTimeout(id) }, [v, ms])
    return d
}

export default function DoctorNotesIndex() {
    const { notes, filters, noteTypes, canCreate } = usePage<Props>().props

    const [search, setSearch] = useState(filters.search ?? '')
    const [type, setType] = useState(filters.note_type ? String(filters.note_type) : NOTE_TYPE_ALL)
    const [privacy, setPrivacy] = useState(filters.private ? String(filters.private) : PRIVACY_ALL)
    const [onlyMine, setOnlyMine] = useState<boolean>(!!filters.only_mine)

    const debouncedSearch = useDebounced(search, 450)
    const firstRun = useRef(true)

    useEffect(() => {
        if (firstRun.current) { firstRun.current = false; return }
        push(1, debouncedSearch, type, privacy, onlyMine, true)
    }, [debouncedSearch])

    function build(page: number, s = search, t = type, p = privacy, mine = onlyMine) {
        const q: Record<string, any> = { page, only_mine: mine ? 1 : 0 }
        if (s.trim()) q.search = s.trim()
        if (t !== NOTE_TYPE_ALL) q.note_type = t
        if (p !== PRIVACY_ALL) q.private = p // 'yes'|'no'
        return q
    }
    function push(page: number, s?: string, t?: string, p?: string, mine?: boolean, replace = false) {
        router.get(route('doctor-notes.index'), build(page, s, t, p, mine), { preserveState: true, replace, preserveScroll: true })
    }

    return (
        <AppLayout breadcrumbs={[{ title: 'Doctor Notes', href: route('doctor-notes.index') }]}>
            <Head title="Doctor Notes" />

            <div className="mx-8 my-6 flex flex-col gap-4 rounded-xl p-6 shadow-sm bg-background text-foreground">
                <div className="mb-2 flex items-center justify-between">
                    <h1 className="text-2xl font-bold flex items-center gap-2"><StickyNote className="h-5 w-5" />Doctor Notes</h1>
                    {canCreate && <Button asChild><a href={route('consultations.index')}>Create (from consultation)</a></Button>}
                </div>

                <Card>
                    <CardHeader><CardTitle>Filters</CardTitle></CardHeader>
                    <CardContent className="grid grid-cols-1 gap-3 md:grid-cols-4">
                        <div className="relative md:col-span-2">
                            <Search className="absolute left-3 top-2.5 h-4 w-4 text-muted-foreground" />
                            <Input value={search} onChange={(e) => setSearch(e.target.value)} onKeyDown={(e) => e.key === 'Enter' && push(1)} placeholder="Search title, body, doctor, patient…" className="pl-9" />
                        </div>

                        <div>
                            <Select value={type} onValueChange={setType}>
                                <SelectTrigger><SelectValue placeholder="All types" /></SelectTrigger>
                                <SelectContent>
                                    <SelectItem value={NOTE_TYPE_ALL}>All types</SelectItem>
                                    {noteTypes.map((t) => <SelectItem key={t} value={t}>{t}</SelectItem>)}
                                </SelectContent>
                            </Select>
                        </div>

                        <div>
                            <Select value={privacy} onValueChange={setPrivacy}>
                                <SelectTrigger><SelectValue placeholder="Private / Public" /></SelectTrigger>
                                <SelectContent>
                                    <SelectItem value={PRIVACY_ALL}>All</SelectItem>
                                    <SelectItem value="yes">Private</SelectItem>
                                    <SelectItem value="no">Public</SelectItem>
                                </SelectContent>
                            </Select>
                        </div>

                        <div className="md:col-span-4 flex items-center justify-between">
                            <label className="flex items-center gap-2 text-sm">
                                <input type="checkbox" checked={onlyMine} onChange={(e) => setOnlyMine(e.target.checked)} />
                                Show only my notes
                            </label>
                            <div className="flex gap-2">
                                <Button variant="outline" onClick={() => { setSearch(''); setType(NOTE_TYPE_ALL); setPrivacy(PRIVACY_ALL); setOnlyMine(false); router.get(route('doctor-notes.index'), {}, { preserveState: true, replace: true, preserveScroll: true }) }}>
                                    Reset
                                </Button>
                                <Button onClick={() => push(1)}>Apply</Button>
                            </div>
                        </div>
                    </CardContent>
                </Card>

                <div className="overflow-x-auto rounded-xl border">
                    <Table className="min-w-full">
                        <TableHeader>
                            <TableRow>
                                <TableHead>Title</TableHead>
                                <TableHead>Type</TableHead>
                                <TableHead>Visibility</TableHead>
                                <TableHead>Doctor</TableHead>
                                <TableHead>Patient</TableHead>
                                <TableHead>Consultation</TableHead>
                                <TableHead>Date</TableHead>
                                <TableHead className="text-right">Actions</TableHead>
                            </TableRow>
                        </TableHeader>
                        <TableBody>
                            {notes.data.map((n) => (
                                <TableRow key={n.id} className="hover:bg-muted/50 transition">
                                    <TableCell className="max-w-[320px]">
                                        <div className="font-medium truncate">{n.title ?? '(Untitled)'}</div>
                                        <div className="text-xs text-muted-foreground line-clamp-1">{n.body}</div>
                                    </TableCell>
                                    <TableCell><Badge variant="secondary">{n.note_type}</Badge></TableCell>
                                    <TableCell>
                                        {n.private
                                            ? <span className="inline-flex items-center gap-1 text-amber-700"><ShieldAlert className="h-4 w-4" /> Private</span>
                                            : <span className="inline-flex items-center gap-1 text-emerald-700"><User className="h-4 w-4" /> Public</span>}
                                    </TableCell>
                                    <TableCell>{n.doctor?.name ?? '-'}</TableCell>
                                    <TableCell>
                                        {n.patient ? `${n.patient.first_name} ${n.patient.surname}` : '-'}
                                        {n.patient?.employee_number ? <span className="ml-1 text-xs text-muted-foreground">• {n.patient.employee_number}</span> : null}
                                    </TableCell>
                                    <TableCell>
                                        {n.consultation?.consultation_date ? moment(n.consultation.consultation_date).format('YYYY-MM-DD') : '-'}
                                    </TableCell>
                                    <TableCell>{moment(n.created_at).format('YYYY-MM-DD HH:mm')}</TableCell>
                                    <TableCell className="text-right">
                                        <Button asChild variant="outline" size="sm">
                                            <a href={route('doctor-notes.show', n.id)} className="inline-flex items-center gap-2">
                                                <Eye className="h-4 w-4" /> View
                                            </a>
                                        </Button>
                                    </TableCell>
                                </TableRow>
                            ))}

                            {notes.data.length === 0 && (
                                <TableRow>
                                    <TableCell colSpan={8} className="text-center py-6 text-muted-foreground">No notes found.</TableCell>
                                </TableRow>
                            )}
                        </TableBody>
                    </Table>
                </div>

                {notes.last_page > 1 && (
                    <div className="flex items-center justify-between text-sm">
                        <div>
                            Showing {(notes.current_page - 1) * notes.per_page + 1}–
                            {Math.min(notes.current_page * notes.per_page, notes.total)} of {notes.total}
                        </div>
                        <div className="flex gap-2">
                            <Button variant="outline" size="sm" disabled={notes.current_page <= 1} onClick={() => push(notes.current_page - 1)}>Prev</Button>
                            <Button variant="outline" size="sm" disabled={notes.current_page >= notes.last_page} onClick={() => push(notes.current_page + 1)}>Next</Button>
                        </div>
                    </div>
                )}
            </div>
        </AppLayout>
    )
}
