import AppLayout from '@/layouts/app-layout';
import { Head, usePage, Link } from '@inertiajs/react';
import { PageProps as InertiaPageProps } from '@inertiajs/core';
import moment from 'moment';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { Button } from '@/components/ui/button';
import {
    StickyNote,
    ShieldAlert,
    ArrowLeft,
    User,
    Stethoscope,
    CalendarDays,
    Building,
    ExternalLink
} from 'lucide-react';

type NoteType = 'Progress' | 'Discharge' | 'Prescription' | 'Other';

interface Doctor { id?: number; name?: string | null; email?: string | null; }
interface Patient {
    id?: number;
    first_name?: string;
    surname?: string;
    employee_number?: string | null;
    gender?: string | null;
    id_number?: string | null;
}
interface ClinicMini {
    id?: number | null;
    name?: string | null;
    address?: string | null;
    city?: string | null;
    email_address?: string | null;
}

interface ConsultationDetails {
    id?: number;
    consultation_date?: string | null;
    created_at?: string | null;
    injury_on_duty?: boolean;

    // Vitals
    systolic?: string | number | null;
    diastolic?: string | number | null;
    temperature?: string | number | null;
    heart_rate?: string | number | null;
    respiratory_rate?: string | number | null;
    oxygen_saturation?: string | number | null;
    weight?: string | number | null;
    height?: string | number | null;
    bmi?: string | number | null;

    // Glucose
    blood_sugar?: string | number | null;
    blood_sugar_unit?: string | null;
    blood_sugar_context?: string | null;
    blood_sugar_measured_at?: string | null;

    // Other tests
    urine_test_result?: string | null;
    hiv_status?: string | null;
    pregnancy_test_result?: string | null;

    // Notes
    presenting_complaint?: string | null;
    clinical_notes?: string | null;
    diagnosis?: string | null;
    treatment_plan?: string | null;
    prescription?: string | null;

    // Leave / follow-up
    sick_leave?: boolean;
    reason_for_leave?: string | null;
    sick_leave_notes?: string | null;
    number_of_days?: number | string | null;
    follow_up_date?: string | null;
    follow_up_required?: boolean;

    // Referrals
    referral_notes?: string | null;
    refer_for_radiology?: boolean;
    refer_for_lab?: boolean;
    refer_to_specialist?: boolean;
    specialist_name?: string | null;
    refer_to_casualty?: boolean;
    refer_out_patient?: boolean;

    // Status
    is_active?: boolean;
    status?: boolean;

    clinic?: ClinicMini | null;
}

interface NotePayload {
    id: number;
    title: string | null;
    body: string | null;
    note_type: NoteType;
    private: boolean;
    meta?: Record<string, any> | null;
    created_at?: string | null;
    updated_at?: string | null;
    doctor?: Doctor | null;
    patient?: Patient | null;
    consultation?: ConsultationDetails | null;
}

interface Props extends InertiaPageProps {
    note: NotePayload;
}

function fmt(val: unknown, fallback = '-'): string {
    if (val === null || val === undefined || val === '') return fallback;
    return String(val);
}

function yesNo(val?: boolean): string {
    return val ? 'Yes' : 'No';
}

export default function DoctorNotesShow() {
    const { note } = usePage<Props>().props;

    const vitals: Array<{ label: string; value: unknown }> = [
        { label: 'Systolic', value: note.consultation?.systolic },
        { label: 'Diastolic', value: note.consultation?.diastolic },
        { label: 'Temperature', value: note.consultation?.temperature },
        { label: 'Heart Rate', value: note.consultation?.heart_rate },
        { label: 'Respiratory Rate', value: note.consultation?.respiratory_rate },
        { label: 'O₂ Saturation', value: note.consultation?.oxygen_saturation },
        { label: 'Weight', value: note.consultation?.weight },
        { label: 'Height', value: note.consultation?.height },
        { label: 'BMI', value: note.consultation?.bmi },
    ];

    const hasVitals = vitals.some(v => v.value !== null && v.value !== undefined && v.value !== '');

    return (
        <AppLayout
            breadcrumbs={[
                { title: 'Doctor Notes', href: route('doctor-notes.index') },
                { title: note.title ?? '(Untitled)', href: '#' },
            ]}
        >
            <Head title={`Note • ${note.title ?? 'Untitled'}`} />

            <div className="mx-8 my-6 flex flex-col gap-4 rounded-xl p-6 shadow-sm bg-background text-foreground">
                {/* Header */}
                <div className="flex items-center justify-between">
                    <div className="flex items-center gap-2">
                        <StickyNote className="h-6 w-6" />
                        <h1 className="text-2xl font-bold">{note.title ?? 'Untitled note'}</h1>
                    </div>

                    <div className="flex items-center gap-2">
                        <Button variant="outline" onClick={() => window.history.back()}>
                            <ArrowLeft className="mr-2 h-4 w-4" />
                            Back
                        </Button>

                        {note.consultation?.id && (
                            <Button asChild>
                                <Link href={route('consultations.show', note.consultation.id!)}>
                                    <ExternalLink className="mr-2 h-4 w-4" />
                                    View Consultation
                                </Link>
                            </Button>
                        )}
                    </div>
                </div>

                {/* Summary header */}
                <Card>
                    <CardHeader className="flex flex-col gap-2 md:flex-row md:items-center md:justify-between">
                        <div className="flex flex-wrap items-center gap-3">
                            <Badge variant="secondary">{note.note_type}</Badge>
                            {note.private ? (
                                <span className="inline-flex items-center gap-1 text-amber-700 text-sm">
                                    <ShieldAlert className="h-4 w-4" /> Private
                                </span>
                            ) : (
                                <span className="inline-flex items-center gap-1 text-emerald-700 text-sm">
                                    Public
                                </span>
                            )}
                            <span className="text-sm text-muted-foreground">
                                Created {note.created_at ? moment(note.created_at).format('YYYY-MM-DD HH:mm') : '-'}
                            </span>
                        </div>
                    </CardHeader>

                    <CardContent className="grid grid-cols-1 gap-4 md:grid-cols-3">
                        {/* Doctor */}
                        <div className="rounded border p-3">
                            <div className="mb-1 flex items-center gap-2 text-sm text-muted-foreground">
                                <Stethoscope className="h-4 w-4" /> Doctor
                            </div>
                            <div className="text-sm">{note.doctor?.name ?? '-'}</div>
                            <div className="text-xs text-muted-foreground">{note.doctor?.email ?? ''}</div>
                        </div>

                        {/* Patient */}
                        <div className="rounded border p-3">
                            <div className="mb-1 flex items-center gap-2 text-sm text-muted-foreground">
                                <User className="h-4 w-4" /> Patient
                            </div>
                            {note.patient ? (
                                <>
                                    <div className="text-sm">
                                        {note.patient.first_name} {note.patient.surname}
                                    </div>
                                    <div className="text-xs text-muted-foreground">
                                        Emp#: {note.patient.employee_number ?? '-'} • ID: {note.patient.id_number ?? '-'} • {note.patient.gender ?? '-'}
                                    </div>
                                </>
                            ) : (
                                <div className="text-sm">-</div>
                            )}
                        </div>

                        {/* Consultation summary */}
                        <div className="rounded border p-3">
                            <div className="mb-1 flex items-center gap-2 text-sm text-muted-foreground">
                                <CalendarDays className="h-4 w-4" /> Consultation
                            </div>
                            {note.consultation ? (
                                <>
                                    <div className="text-sm">
                                        Date:{' '}
                                        {note.consultation.consultation_date
                                            ? moment(note.consultation.consultation_date).format('YYYY-MM-DD')
                                            : '-'}
                                    </div>
                                    <div className="text-xs text-muted-foreground">
                                        Injury on duty: {yesNo(note.consultation.injury_on_duty)}
                                    </div>
                                    <div className="text-xs text-muted-foreground">
                                        Created: {note.consultation.created_at ? moment(note.consultation.created_at).format('YYYY-MM-DD HH:mm') : '-'}
                                    </div>
                                    {note.consultation.clinic && (
                                        <div className="mt-2 text-xs text-muted-foreground flex items-center gap-2">
                                            <Building className="h-3 w-3" />
                                            <span>
                                                {note.consultation.clinic.name ?? '-'}
                                                {note.consultation.clinic.city ? ` • ${note.consultation.clinic.city}` : ''}
                                            </span>
                                        </div>
                                    )}
                                </>
                            ) : (
                                <div className="text-sm">-</div>
                            )}
                        </div>
                    </CardContent>
                </Card>

                {/* Note body */}
                <Card>
                    <CardHeader>
                        <CardTitle>Note</CardTitle>
                    </CardHeader>
                    <CardContent>
                        <div className="whitespace-pre-wrap text-sm">{note.body ?? ''}</div>
                    </CardContent>
                </Card>

                {/* Consultation details (Vitals) */}
                {note.consultation && hasVitals && (
                    <Card>
                        <CardHeader>
                            <CardTitle>Consultation • Vitals</CardTitle>
                        </CardHeader>
                        <CardContent>
                            <div className="grid grid-cols-2 md:grid-cols-4 gap-3 text-sm">
                                {vitals.map(({ label, value }) => (
                                    <div key={label} className="rounded border p-2">
                                        <div className="text-xs font-semibold text-muted-foreground">{label}</div>
                                        <div>{fmt(value)}</div>
                                    </div>
                                ))}
                            </div>
                        </CardContent>
                    </Card>
                )}

                {/* Consultation details (Glucose & Tests) */}
                {note.consultation && (
                    <>
                        {(note.consultation.blood_sugar ||
                            note.consultation.blood_sugar_unit ||
                            note.consultation.blood_sugar_context ||
                            note.consultation.blood_sugar_measured_at) && (
                                <Card>
                                    <CardHeader>
                                        <CardTitle>Glucose</CardTitle>
                                    </CardHeader>
                                    <CardContent>
                                        <div className="grid grid-cols-2 md:grid-cols-4 gap-3 text-sm">
                                            <div className="rounded border p-2">
                                                <div className="text-xs font-semibold text-muted-foreground">Blood Sugar</div>
                                                <div>
                                                    {fmt(note.consultation.blood_sugar)} {fmt(note.consultation.blood_sugar_unit, '')}
                                                </div>
                                            </div>
                                            <div className="rounded border p-2">
                                                <div className="text-xs font-semibold text-muted-foreground">Context</div>
                                                <div>{fmt(note.consultation.blood_sugar_context)}</div>
                                            </div>
                                            <div className="rounded border p-2">
                                                <div className="text-xs font-semibold text-muted-foreground">Measured At</div>
                                                <div>
                                                    {note.consultation.blood_sugar_measured_at
                                                        ? moment(note.consultation.blood_sugar_measured_at).format('YYYY-MM-DD HH:mm')
                                                        : '-'}
                                                </div>
                                            </div>
                                        </div>
                                    </CardContent>
                                </Card>
                            )}

                        {(note.consultation.urine_test_result ||
                            note.consultation.hiv_status ||
                            note.consultation.pregnancy_test_result) && (
                                <Card>
                                    <CardHeader>
                                        <CardTitle>Other Tests</CardTitle>
                                    </CardHeader>
                                    <CardContent>
                                        <div className="grid grid-cols-2 md:grid-cols-3 gap-3 text-sm">
                                            <div className="rounded border p-2">
                                                <div className="text-xs font-semibold text-muted-foreground">Urine Test</div>
                                                <div>{fmt(note.consultation.urine_test_result)}</div>
                                            </div>
                                            <div className="rounded border p-2">
                                                <div className="text-xs font-semibold text-muted-foreground">HIV Status</div>
                                                <div>{fmt(note.consultation.hiv_status)}</div>
                                            </div>
                                            <div className="rounded border p-2">
                                                <div className="text-xs font-semibold text-muted-foreground">Pregnancy Test</div>
                                                <div>{fmt(note.consultation.pregnancy_test_result)}</div>
                                            </div>
                                        </div>
                                    </CardContent>
                                </Card>
                            )}
                    </>
                )}

                {/* Consultation details (Notes summary) */}
                {note.consultation && (
                    <>
                        {(note.consultation.presenting_complaint ||
                            note.consultation.diagnosis ||
                            note.consultation.treatment_plan) && (
                                <Card>
                                    <CardHeader>
                                        <CardTitle>Consultation Summary</CardTitle>
                                    </CardHeader>
                                    <CardContent>
                                        <div className="space-y-3 text-sm">
                                            {note.consultation.presenting_complaint && (
                                                <div className="rounded border p-2">
                                                    <div className="text-xs font-semibold text-muted-foreground">Presenting Complaint</div>
                                                    <div className="whitespace-pre-wrap">{note.consultation.presenting_complaint}</div>
                                                </div>
                                            )}
                                            {note.consultation.diagnosis && (
                                                <div className="rounded border p-2">
                                                    <div className="text-xs font-semibold text-muted-foreground">Diagnosis</div>
                                                    <div className="whitespace-pre-wrap">{note.consultation.diagnosis}</div>
                                                </div>
                                            )}
                                            {note.consultation.treatment_plan && (
                                                <div className="rounded border p-2">
                                                    <div className="text-xs font-semibold text-muted-foreground">Treatment Plan</div>
                                                    <div className="whitespace-pre-wrap">{note.consultation.treatment_plan}</div>
                                                </div>
                                            )}
                                        </div>
                                    </CardContent>
                                </Card>
                            )}
                    </>
                )}

                {/* Consultation details (Leave & Follow-up) */}
                {note.consultation && (
                    <>
                        {(typeof note.consultation.sick_leave !== 'undefined' ||
                            note.consultation.follow_up_date ||
                            typeof note.consultation.follow_up_required !== 'undefined' ||
                            note.consultation.number_of_days ||
                            note.consultation.reason_for_leave ||
                            note.consultation.sick_leave_notes) && (
                                <Card>
                                    <CardHeader>
                                        <CardTitle>Leave & Follow-up</CardTitle>
                                    </CardHeader>
                                    <CardContent>
                                        <div className="grid grid-cols-2 md:grid-cols-3 gap-3 text-sm">
                                            <div className="rounded border p-2">
                                                <div className="text-xs font-semibold text-muted-foreground">Sick Leave</div>
                                                <div>{yesNo(note.consultation.sick_leave)}</div>
                                            </div>
                                            <div className="rounded border p-2">
                                                <div className="text-xs font-semibold text-muted-foreground">Days</div>
                                                <div>{fmt(note.consultation.number_of_days)}</div>
                                            </div>
                                            <div className="rounded border p-2">
                                                <div className="text-xs font-semibold text-muted-foreground">Follow-up Required</div>
                                                <div>{yesNo(note.consultation.follow_up_required)}</div>
                                            </div>
                                            <div className="rounded border p-2">
                                                <div className="text-xs font-semibold text-muted-foreground">Follow-up Date</div>
                                                <div>
                                                    {note.consultation.follow_up_date
                                                        ? moment(note.consultation.follow_up_date).format('YYYY-MM-DD')
                                                        : '-'}
                                                </div>
                                            </div>
                                            {note.consultation.reason_for_leave && (
                                                <div className="rounded border p-2 md:col-span-2">
                                                    <div className="text-xs font-semibold text-muted-foreground">Reason for Leave</div>
                                                    <div className="whitespace-pre-wrap">{note.consultation.reason_for_leave}</div>
                                                </div>
                                            )}
                                            {note.consultation.sick_leave_notes && (
                                                <div className="rounded border p-2 md:col-span-2">
                                                    <div className="text-xs font-semibold text-muted-foreground">Leave Notes</div>
                                                    <div className="whitespace-pre-wrap">{note.consultation.sick_leave_notes}</div>
                                                </div>
                                            )}
                                        </div>
                                    </CardContent>
                                </Card>
                            )}
                    </>
                )}

                {/* Consultation details (Referrals) */}
                {note.consultation && (
                    <>
                        {(note.consultation.refer_for_radiology ||
                            note.consultation.refer_for_lab ||
                            note.consultation.refer_to_specialist ||
                            note.consultation.refer_to_casualty ||
                            note.consultation.refer_out_patient ||
                            note.consultation.referral_notes) && (
                                <Card>
                                    <CardHeader>
                                        <CardTitle>Referrals</CardTitle>
                                    </CardHeader>
                                    <CardContent>
                                        <div className="grid grid-cols-2 md:grid-cols-3 gap-3 text-sm">
                                            <div className="rounded border p-2">
                                                <div className="text-xs font-semibold text-muted-foreground">Radiology</div>
                                                <div>{yesNo(note.consultation.refer_for_radiology)}</div>
                                            </div>
                                            <div className="rounded border p-2">
                                                <div className="text-xs font-semibold text-muted-foreground">Lab</div>
                                                <div>{yesNo(note.consultation.refer_for_lab)}</div>
                                            </div>
                                            <div className="rounded border p-2">
                                                <div className="text-xs font-semibold text-muted-foreground">Specialist</div>
                                                <div>
                                                    {yesNo(note.consultation.refer_to_specialist)}
                                                    {note.consultation.specialist_name ? ` • ${note.consultation.specialist_name}` : ''}
                                                </div>
                                            </div>
                                            <div className="rounded border p-2">
                                                <div className="text-xs font-semibold text-muted-foreground">Casualty</div>
                                                <div>{yesNo(note.consultation.refer_to_casualty)}</div>
                                            </div>
                                            <div className="rounded border p-2">
                                                <div className="text-xs font-semibold text-muted-foreground">Out Patient</div>
                                                <div>{yesNo(note.consultation.refer_out_patient)}</div>
                                            </div>
                                            {note.consultation.referral_notes && (
                                                <div className="rounded border p-2 md:col-span-2">
                                                    <div className="text-xs font-semibold text-muted-foreground">Referral Notes</div>
                                                    <div className="whitespace-pre-wrap">{note.consultation.referral_notes}</div>
                                                </div>
                                            )}
                                        </div>
                                    </CardContent>
                                </Card>
                            )}
                    </>
                )}

                {/* Meta (optional vitals snapshot etc.) */}
                {note.meta && Object.keys(note.meta).length > 0 && (
                    <Card>
                        <CardHeader>
                            <CardTitle>Meta</CardTitle>
                        </CardHeader>
                        <CardContent>
                            <div className="grid grid-cols-2 md:grid-cols-4 gap-3 text-sm">
                                {Object.entries(note.meta).map(([k, v]) => (
                                    <div key={k} className="rounded border p-2">
                                        <div className="text-xs font-semibold text-muted-foreground">{k}</div>
                                        <div>{String(v)}</div>
                                    </div>
                                ))}
                            </div>
                        </CardContent>
                    </Card>
                )}
            </div>
        </AppLayout>
    );
}
