import AppLayout from '@/layouts/app-layout';
import { Head, usePage } from '@inertiajs/react';
import { useMemo, useState } from 'react';
import axios from 'axios';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from '@/components/ui/card';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from '@/components/ui/table';
import { Badge } from '@/components/ui/badge';
import { RadioGroup, RadioGroupItem } from '@/components/ui/radio-group';
import { Label } from '@/components/ui/label';
import { Select, SelectTrigger, SelectContent, SelectItem, SelectValue } from '@/components/ui/select';
import { Input } from '@/components/ui/input';
import { Check, AlertTriangle } from 'lucide-react';
import { cn } from '@/lib/utils';
import ReactPaginate from 'react-paginate';
import moment from 'moment';
import { Modal } from 'antd'; // antd only for modal container
import Swal from 'sweetalert2';
import { API } from '@/config';

type Company = { id: number; name: string };
type Patient = {
    id: number;
    first_name: string;
    middle_name?: string | null;
    surname: string;
    id_number?: string | null;
    employee_number?: string | null;
    company?: Company | null;
    phone?: string | null;
    date_of_birth?: string | null;
    parent?: { id: number; first_name: string; surname: string } | null;
    parent_patient_id?: number | null;
};

type Group = {
    type: 'id_number' | 'employee_company' | 'name_dob' | 'name_phone';
    label: string;
    items: Patient[];
};

type PageProps = {
    groups: Group[];
    companies: Company[];
    filters: { company_id?: number; limit?: number };
};

const PAGE_SIZE_GROUPS = 5;
const PAGE_SIZE_ITEMS = 10;

export default function DuplicatesIndex() {
    const { groups, companies, filters } = usePage<PageProps>().props;

    // filters
    const [companyId, setCompanyId] = useState<string>(
        filters.company_id ? String(filters.company_id) : 'all'
    );
    const [limit, setLimit] = useState<string>(filters.limit ? String(filters.limit) : '100');

    // groups pagination
    const [groupPage, setGroupPage] = useState<number>(0);

    // merge modal
    const [open, setOpen] = useState(false);
    const [currentGroup, setCurrentGroup] = useState<Group | null>(null);
    const [targetId, setTargetId] = useState<number | null>(null);
    const [sourceIds, setSourceIds] = useState<number[]>([]);
    const [itemPage, setItemPage] = useState<number>(0);
    const [merging, setMerging] = useState<boolean>(false);

    const nameOf = (p: Patient) =>
        [p.first_name, p.middle_name, p.surname].filter(Boolean).join(' ');

    const dobWithAge = (dob?: string | null) => {
        if (!dob) return '—';
        const m = moment(dob);
        if (!m.isValid()) return dob;
        const age = moment().diff(m, 'years');
        return `${m.format('YYYY-MM-DD')} (${age}y)`;
    };

    const refetchWithFilters = () => {
        const params = new URLSearchParams();
        if (companyId !== 'all') params.set('company_id', companyId);
        if (limit) params.set('limit', limit);
        const qs = params.toString();
        window.location.href = `${API}/duplicates${qs ? `?${qs}` : ''}`;
    };

    // groups paging
    const pageCountGroups = Math.max(1, Math.ceil(groups.length / PAGE_SIZE_GROUPS));
    const pagedGroups = useMemo(() => {
        const start = groupPage * PAGE_SIZE_GROUPS;
        return groups.slice(start, start + PAGE_SIZE_GROUPS);
    }, [groups, groupPage]);
    const onGroupPageChange = (sel: { selected: number }) => setGroupPage(sel.selected);

    // open merge modal
    const openMerge = (g: Group) => {
        setCurrentGroup(g);
        setTargetId(null);
        setSourceIds([]);
        setItemPage(0);
        setOpen(true);
    };

    const toggleSource = (id: number) => {
        setSourceIds(prev => prev.includes(id) ? prev.filter(x => x !== id) : [...prev, id]);
    };

    const submitMerge = async () => {
        if (!targetId || sourceIds.length === 0 || merging) return;

        setMerging(true);

        // Open loading dialog (do NOT await)
        Swal.fire({
            title: 'Merging…',
            html: 'Please wait while we re-link records and clean up duplicates.',
            allowOutsideClick: false,
            allowEscapeKey: false,
            didOpen: () => {
                Swal.showLoading();
            },
        });

        try {
            await axios.post(`${API}/duplicates/merge`, { target_id: targetId, source_ids: sourceIds });

            // Close loading Swal and the AntD modal
            Swal.close();
            setOpen(false); // ⬅️ close the modal immediately on success

            // Success message
            await Swal.fire({
                icon: 'success',
                title: 'Merge complete',
                text: 'All selected records now belong to the chosen patient.',
                confirmButtonText: 'OK',
            });

            // Refresh to reflect changes
            window.location.reload();
        } catch (e: any) {
            console.error(e);
            const msg = e?.response?.data?.message ?? 'Merge failed';

            // Close the loading dialog before showing error
            Swal.close();

            await Swal.fire({
                icon: 'error',
                title: 'Merge failed',
                text: msg,
                confirmButtonText: 'Close',
            });
        } finally {
            setMerging(false);
        }
    };

    // dialog items pagination
    const dialogItems = currentGroup?.items ?? [];
    const dialogPageCount = Math.max(1, Math.ceil(dialogItems.length / PAGE_SIZE_ITEMS));
    const pagedDialogItems = useMemo(() => {
        const start = itemPage * PAGE_SIZE_ITEMS;
        return dialogItems.slice(start, start + PAGE_SIZE_ITEMS);
    }, [dialogItems, itemPage]);
    const onItemPageChange = (sel: { selected: number }) => setItemPage(sel.selected);

    return (
        <AppLayout>
            <Head title="Duplicate Patients" />
            <div className="bg-background mx-8 my-6 rounded-xl p-6 shadow-sm">

                <div className="flex items-center justify-between gap-4 mb-6">
                    <div>
                        <h1 className="text-2xl font-semibold">Duplicate Patients</h1>
                        <p className="text-muted-foreground">
                            Identify and merge duplicated patient records. All relations will be reassigned to the chosen master.
                        </p>
                    </div>
                    <div className="flex items-center gap-2">
                        <Select value={companyId} onValueChange={setCompanyId}>
                            <SelectTrigger className="w-[220px]">
                                <SelectValue placeholder="Filter by company" />
                            </SelectTrigger>
                            <SelectContent>
                                <SelectItem value="all">All companies</SelectItem>
                                {companies.map((c) => (
                                    <SelectItem key={c.id} value={String(c.id)}>
                                        {c.name}
                                    </SelectItem>
                                ))}
                            </SelectContent>
                        </Select>
                        <Input
                            className="w-[120px]"
                            type="number"
                            min={10}
                            step={10}
                            value={limit}
                            onChange={(e) => setLimit(e.target.value)}
                            placeholder="Limit"
                        />
                        <Button onClick={refetchWithFilters} variant="secondary" style={{
                            cursor: 'pointer'
                        }}>
                            Find duplicates
                        </Button>
                    </div>
                </div>

                {groups.length === 0 ? (
                    <Card>
                        <CardHeader>
                            <CardTitle>No duplicates found</CardTitle>
                            <CardDescription>Try adjusting filters or increasing the limit.</CardDescription>
                        </CardHeader>
                    </Card>
                ) : (
                    <>
                        <div className="grid gap-4">
                            {pagedGroups.map((g, idx) => (
                                <Card key={`${groupPage}-${idx}`}>
                                    <CardHeader className="flex items-center justify-between gap-2">
                                        <div className="flex items-center gap-2">
                                            <AlertTriangle className="h-5 w-5 text-amber-500" />
                                            <CardTitle className="text-base">{g.label}</CardTitle>
                                        </div>
                                        <Button size="sm" onClick={() => openMerge(g)} style={{
                                            cursor: 'pointer'
                                        }}>
                                            Review & Merge
                                        </Button>
                                    </CardHeader>
                                    <CardContent>
                                        <Table>
                                            <TableHeader>
                                                <TableRow>
                                                    <TableHead>Name</TableHead>
                                                    <TableHead>DoB (Age)</TableHead>
                                                    <TableHead>Phone</TableHead>
                                                    <TableHead>ID No.</TableHead>
                                                    <TableHead>Emp #</TableHead>
                                                    <TableHead>Company</TableHead>
                                                    <TableHead>Type</TableHead>
                                                </TableRow>
                                            </TableHeader>
                                            <TableBody>
                                                {g.items.map((p) => (
                                                    <TableRow key={p.id} style={{
                                                        cursor: 'pointer'
                                                    }}>
                                                        <TableCell className="font-medium">{nameOf(p)}</TableCell>
                                                        <TableCell>{dobWithAge(p.date_of_birth)}</TableCell>
                                                        <TableCell>{p.phone ?? '—'}</TableCell>
                                                        <TableCell>{p.id_number ?? '—'}</TableCell>
                                                        <TableCell>{p.employee_number ?? '—'}</TableCell>
                                                        <TableCell>{p.company?.name ?? '—'}</TableCell>
                                                        <TableCell>
                                                            {p.parent_patient_id ? (
                                                                <Badge variant="secondary">Dependent</Badge>
                                                            ) : (
                                                                <Badge>Employee</Badge>
                                                            )}
                                                        </TableCell>
                                                    </TableRow>
                                                ))}
                                            </TableBody>
                                        </Table>
                                    </CardContent>
                                </Card>
                            ))}
                        </div>

                        {/* Groups pagination */}
                        <div className="mt-4 flex justify-center">
                            <ReactPaginate
                                pageCount={pageCountGroups}
                                forcePage={groupPage}
                                onPageChange={onGroupPageChange}
                                nextLabel="›"
                                previousLabel="‹"
                                breakLabel="…"
                                containerClassName="flex items-center gap-2"
                                pageClassName="px-3 py-1 rounded border"
                                activeClassName="bg-primary text-primary-foreground"
                                previousClassName="px-3 py-1 rounded border"
                                nextClassName="px-3 py-1 rounded border"
                                breakClassName="px-3 py-1"
                            />
                        </div>
                    </>
                )}

                {/* Merge Modal (antd container only) */}
                <Modal
                    open={open}
                    onCancel={() => setOpen(false)}
                    footer={null}
                    centered
                    width={900}
                    destroyOnClose
                    maskClosable={false}
                >
                    {/* shadcn content inside */}
                    <div className="space-y-4">
                        <div>
                            <h2 className="text-xl font-semibold">Merge duplicate records</h2>
                            <p className="text-sm text-muted-foreground">
                                Choose the master (kept) patient. All visits, triages, notes, prescriptions, etc. will be reassigned to the master.
                                Empty fields on the master will be filled from the others when possible.
                            </p>
                        </div>

                        {currentGroup && (
                            <>
                                <div>
                                    <Label className="text-sm">Select master record</Label>
                                    <RadioGroup
                                        value={targetId ? String(targetId) : ''}
                                        onValueChange={(v) => setTargetId(parseInt(v))}
                                        className="mt-2 grid gap-2"
                                    >
                                        {pagedDialogItems.map((p) => (
                                            <Label
                                                key={`master-${p.id}`}
                                                className={cn(
                                                    'flex items-center justify-between rounded-lg border p-3 cursor-pointer',
                                                    targetId === p.id && 'border-primary'
                                                )}
                                            >
                                                <div className="flex items-center gap-3">
                                                    <RadioGroupItem value={String(p.id)} />
                                                    <div className="leading-tight">
                                                        <div className="font-medium">{nameOf(p)}</div>
                                                        <div className="text-xs text-muted-foreground">
                                                            DoB: {dobWithAge(p.date_of_birth)} · Phone: {p.phone ?? '—'} · ID: {p.id_number ?? '—'} · Emp#:{' '}
                                                            {p.employee_number ?? '—'} · {p.company?.name ?? 'No Company'}
                                                        </div>
                                                    </div>
                                                </div>
                                                {targetId === p.id && <Check className="h-4 w-4 text-primary" />}
                                            </Label>
                                        ))}
                                    </RadioGroup>
                                </div>

                                <div>
                                    <Label className="text-sm">Select records to merge into master</Label>
                                    <div className="mt-2 grid gap-2">
                                        {pagedDialogItems.map((p) => (
                                            <Label
                                                key={`src-${p.id}`}
                                                className={cn(
                                                    'flex items-center justify-between rounded-lg border p-3 cursor-pointer',
                                                    sourceIds.includes(p.id) && 'border-amber-500'
                                                )}
                                            >
                                                <div className="flex items-center gap-3">
                                                    <input
                                                        type="checkbox"
                                                        className="h-4 w-4"
                                                        checked={sourceIds.includes(p.id)}
                                                        onChange={() => toggleSource(p.id)}
                                                        disabled={p.id === targetId}
                                                    />
                                                    <div className="leading-tight">
                                                        <div className="font-medium">{nameOf(p)}</div>
                                                        <div className="text-xs text-muted-foreground">
                                                            DoB: {dobWithAge(p.date_of_birth)} · Phone: {p.phone ?? '—'} · ID: {p.id_number ?? '—'} · Emp#:{' '}
                                                            {p.employee_number ?? '—'} · {p.company?.name ?? 'No Company'}
                                                        </div>
                                                    </div>
                                                </div>
                                                {p.id === targetId && <Badge variant="outline">Master</Badge>}
                                            </Label>
                                        ))}
                                    </div>
                                </div>

                                {dialogItems.length > PAGE_SIZE_ITEMS && (
                                    <div className="pt-2 flex justify-center">
                                        <ReactPaginate
                                            pageCount={dialogPageCount}
                                            forcePage={itemPage}
                                            onPageChange={onItemPageChange}
                                            nextLabel="›"
                                            previousLabel="‹"
                                            breakLabel="…"
                                            containerClassName="flex items-center gap-2"
                                            pageClassName="px-3 py-1 rounded border"
                                            activeClassName="bg-primary text-primary-foreground"
                                            previousClassName="px-3 py-1 rounded border"
                                            nextClassName="px-3 py-1 rounded border"
                                            breakClassName="px-3 py-1"
                                        />
                                    </div>
                                )}

                                <div className="flex justify-end gap-2 pt-2">
                                    <Button variant="secondary" onClick={() => setOpen(false)} disabled={merging} style={{
                                        cursor: 'pointer'
                                    }}>
                                        Cancel
                                    </Button>
                                    <Button onClick={submitMerge} disabled={!targetId || sourceIds.length === 0 || merging} style={{
                                        cursor: 'pointer'
                                    }}>
                                        {merging ? 'Merging…' : 'Merge selected'}
                                    </Button>
                                </div>
                            </>
                        )}
                    </div>
                </Modal>

            </div>
        </AppLayout>
    );
}
