import React, { useMemo, useRef, useState, useMemo as useMemoAlias } from 'react'
import AppLayout from '@/layouts/app-layout'
import { Head, usePage } from '@inertiajs/react'
import Swal from 'sweetalert2'

import { Button } from '@/components/ui/button'
import { Card, CardContent, CardDescription, CardFooter, CardHeader, CardTitle } from '@/components/ui/card'
import { Badge } from '@/components/ui/badge'
import { Table, TableHeader, TableRow, TableHead, TableBody, TableCell } from '@/components/ui/table'
import { Separator } from '@/components/ui/separator'
import { Alert, AlertTitle, AlertDescription } from '@/components/ui/alert'
import { Upload, FileSpreadsheet, CheckCircle2, XCircle, Download } from 'lucide-react'

type InertiaPageProps = {
    templates: {
        members: { headers: string[] }
    }
}

type PreviewRow = {
    index: number
    data: Record<string, any>
    errors: string[]
    raw: Record<string, any>
}

type PreviewResponse = {
    columns: string[]
    rows: PreviewRow[]
    stats: { total: number; valid: number; invalid: number }
}

export default function BulkImportPage() {
    const { templates } = usePage<InertiaPageProps>().props

    const [file, setFile] = useState<File | null>(null)
    const [preview, setPreview] = useState<PreviewResponse | null>(null)
    const [loadingPreview, setLoadingPreview] = useState(false)
    const [committing, setCommitting] = useState(false)

    const fileRef = useRef<HTMLInputElement | null>(null)

    const dataColumns = useMemo(() => {
        if (!preview?.rows?.length) return []
        const set = new Set<string>()
        preview.rows.forEach(r => Object.keys(r.data || {}).forEach(k => set.add(k)))
        return Array.from(set)
    }, [preview])

    const csrf = useMemoAlias(() => {
        const meta = document.querySelector<HTMLMetaElement>('meta[name="csrf-token"]')
        return meta?.content ?? ''
    }, [])

    // @ts-ignore
    const previewRoute: string = route('fund_members.bulk.preview')
    // @ts-ignore
    const commitRoute: string = route('fund_members.bulk.commit')

    const onPickFile = (e: React.ChangeEvent<HTMLInputElement>) => {
        if (e.target.files && e.target.files[0]) {
            setFile(e.target.files[0])
            setPreview(null)
        }
    }

    const clearAll = () => {
        setFile(null)
        setPreview(null)
        if (fileRef.current) fileRef.current.value = ''
    }

    const doPreview = async () => {
        if (!file) {
            Swal.fire({ icon: 'warning', title: 'Choose a file', text: 'Please select an .xlsx or .csv file first.' })
            return
        }

        const fd = new FormData()
        fd.append('file', file)
        fd.append('_token', csrf)

        try {
            setLoadingPreview(true)
            Swal.fire({
                title: 'Parsing file…',
                html: 'Validating rows; please wait.',
                allowOutsideClick: false,
                didOpen: () => Swal.showLoading(),
            })

            const res = await fetch(previewRoute, {
                method: 'POST',
                body: fd,
                headers: { 'X-Requested-With': 'XMLHttpRequest', 'X-CSRF-TOKEN': csrf, 'Accept': 'application/json' },
                credentials: 'same-origin',
            })

            if (!res.ok) {
                Swal.close()
                const text = await res.text()
                throw new Error(text || 'Failed to preview file.')
            }

            const json = (await res.json()) as PreviewResponse
            setPreview(json)

            Swal.close()
            Swal.fire({
                icon: 'success',
                title: 'Preview ready',
                text: `Found ${json.stats.total} rows; ${json.stats.valid} valid, ${json.stats.invalid} invalid.`,
            })
        } catch (err: any) {
            Swal.close()
            Swal.fire({ icon: 'error', title: 'Preview failed', text: err?.message ?? 'Could not parse the file.' })
        } finally {
            setLoadingPreview(false)
        }
    }

    const doCommit = async () => {
        if (!file) {
            Swal.fire({ icon: 'warning', title: 'Choose a file', text: 'Please select a file first.' })
            return
        }
        if (!preview) {
            Swal.fire({ icon: 'info', title: 'No preview yet', text: 'Please preview the file before importing.' })
            return
        }

        const fd = new FormData()
        fd.append('file', file)
        fd.append('_token', csrf)

        try {
            setCommitting(true)
            Swal.fire({
                title: 'Importing…',
                html: 'Creating records; please wait.',
                allowOutsideClick: false,
                didOpen: () => Swal.showLoading(),
            })

            const res = await fetch(commitRoute, {
                method: 'POST',
                body: fd,
                headers: { 'X-Requested-With': 'XMLHttpRequest', 'X-CSRF-TOKEN': csrf, 'Accept': 'application/json' },
                credentials: 'same-origin',
            })

            const json = await res.json()

            Swal.close()
            if (!res.ok || json?.ok === false) {
                throw new Error(json?.message || 'Import failed.')
            }

            const { created, updated, failed } = json
            Swal.fire({
                icon: 'success',
                title: 'Import finished',
                html: `<div style="text-align:left">
          <div><strong>Created:</strong> ${created}</div>
          <div><strong>Updated:</strong> ${updated}</div>
          <div><strong>Failed:</strong> ${failed}</div>
        </div>`,
            })
            clearAll()
        } catch (err: any) {
            Swal.close()
            Swal.fire({ icon: 'error', title: 'Import failed', text: err?.message ?? 'Could not import the file.' })
        } finally {
            setCommitting(false)
        }
    }

    const downloadTemplate = () => {
        const headers = templates.members.headers || []
        const content = headers.join(',') + '\n'
        const blob = new Blob([content], { type: 'text/csv;charset=utf-8;' })
        const url = URL.createObjectURL(blob)
        const a = document.createElement('a')
        a.href = url
        a.download = `fund_members_template.csv`
        document.body.appendChild(a)
        a.click()
        a.remove()
        URL.revokeObjectURL(url)
    }

    const stats = preview?.stats

    return (
        <AppLayout breadcrumbs={[{ title: 'Fund Members', href: route('fund_members.index') }, { title: 'Bulk Import', href: '#' }]}>
            <Head title="Fund Members • Bulk Import" />

            <div className="mx-8 my-6 overflow-x-hidden">
                <Card className="shadow-sm">
                    <CardHeader>
                        <CardTitle>Bulk Import — Fund Members</CardTitle>
                        <CardDescription>Upload an .xlsx or .csv file, preview the rows, then import into the system.</CardDescription>
                    </CardHeader>

                    <CardContent className="space-y-6">
                        <div className="flex items-center justify-between">
                            <div className="text-sm text-muted-foreground">Expected columns: {templates.members.headers.join(', ')}</div>
                            <Button variant="outline" size="sm" onClick={downloadTemplate}>
                                <Download className="mr-2 h-4 w-4" />
                                Download Template
                            </Button>
                        </div>

                        <Separator />

                        <div className="grid gap-4 sm:grid-cols-[1fr_auto] sm:items-end">
                            <div>
                                <label className="mb-1 block text-sm font-medium">Choose file (.xlsx or .csv)</label>
                                <div className="flex items-center gap-2">
                                    <input
                                        ref={fileRef}
                                        type="file"
                                        accept=".xlsx,.csv"
                                        onChange={onPickFile}
                                        className="w-full rounded-md border border-border bg-background p-2 text-sm"
                                    />
                                    {file ? <Badge variant="secondary" className="shrink-0 max-w-[50vw] truncate">{file.name}</Badge> : null}
                                </div>
                            </div>

                            <div className="flex gap-2">
                                <Button onClick={doPreview} disabled={!file || loadingPreview} className="shrink-0" style={{
                                    cursor: 'pointer'
                                }}>
                                    <Upload className="mr-2 h-4 w-4" />
                                    {loadingPreview ? 'Parsing…' : 'Preview'}
                                </Button>
                                <Button onClick={doCommit} disabled={!file || !preview || committing} className="shrink-0" variant="secondary"
                                    style={{
                                        cursor: 'pointer'
                                    }}
                                >
                                    <FileSpreadsheet className="mr-2 h-4 w-4" />
                                    {committing ? 'Importing…' : 'Import'}
                                </Button>
                                <Button variant="outline" onClick={clearAll} disabled={!file && !preview} className="shrink-0"
                                    style={{
                                        cursor: 'pointer'
                                    }}
                                >
                                    Clear
                                </Button>
                            </div>
                        </div>

                        {preview && (
                            <div className="flex flex-wrap items-center gap-3">
                                <Badge variant="outline">Total: {stats?.total ?? 0}</Badge>
                                <Badge className="bg-emerald-600 hover:bg-emerald-600">Valid: {stats?.valid ?? 0}</Badge>
                                <Badge className="bg-rose-600 hover:bg-rose-600">Invalid: {stats?.invalid ?? 0}</Badge>
                            </div>
                        )}

                        {!preview && (
                            <Alert>
                                <AlertTitle>No preview to show yet</AlertTitle>
                                <AlertDescription>Choose a file, then click “Preview”.</AlertDescription>
                            </Alert>
                        )}

                        {preview && (
                            <div className="rounded-md border overflow-hidden">
                                {/* Vertical & horizontal scroll within preview area */}
                                <div className="max-h-[60vh] overflow-y-auto overflow-x-auto">
                                    <Table className="min-w-max">
                                        <TableHeader className="sticky top-0 z-10 bg-background">
                                            <TableRow>
                                                <TableHead className="w-[80px] whitespace-nowrap">Row</TableHead>
                                                {dataColumns.map((col) => (
                                                    <TableHead key={col} className="capitalize whitespace-nowrap align-top">
                                                        {col.replaceAll('_', ' ')}
                                                    </TableHead>
                                                ))}
                                                <TableHead className="w-[220px] whitespace-nowrap">Validation</TableHead>
                                            </TableRow>
                                        </TableHeader>
                                        <TableBody>
                                            {preview.rows.map((r) => {
                                                const isValid = !r.errors || r.errors.length === 0
                                                return (
                                                    <TableRow key={r.index} className={isValid ? '' : 'bg-rose-50/40 dark:bg-rose-950/10'}>
                                                        <TableCell className="font-medium tabular-nums align-top whitespace-nowrap">{r.index}</TableCell>
                                                        {dataColumns.map((col) => (
                                                            <TableCell key={`${r.index}-${col}`} className="align-top whitespace-nowrap" title={String(r.data?.[col] ?? '')}>
                                                                {formatCell(r.data?.[col])}
                                                            </TableCell>
                                                        ))}
                                                        <TableCell className="align-top">
                                                            {isValid ? (
                                                                <span className="inline-flex items-center gap-1 rounded-full bg-emerald-100 px-2 py-0.5 text-xs font-medium text-emerald-800 dark:bg-emerald-900/40 dark:text-emerald-200">
                                                                    <CheckCircle2 className="h-3.5 w-3.5" />
                                                                    OK
                                                                </span>
                                                            ) : (
                                                                <div className="flex flex-wrap gap-1">
                                                                    {r.errors.map((e, i) => (
                                                                        <span key={i} className="inline-flex items-center gap-1 rounded-full bg-rose-100 px-2 py-0.5 text-[11px] font-medium text-rose-800 dark:bg-rose-900/40 dark:text-rose-200">
                                                                            <XCircle className="h-3.5 w-3.5" />
                                                                            {e}
                                                                        </span>
                                                                    ))}
                                                                </div>
                                                            )}
                                                        </TableCell>
                                                    </TableRow>
                                                )
                                            })}
                                        </TableBody>
                                    </Table>
                                </div>
                            </div>
                        )}

                        <div className="text-xs text-muted-foreground">
                            <p>• CSV must be UTF-8, comma-separated, with a header row.</p>
                            <p>• Allowed columns: <code>{templates.members.headers.join(', ')}</code></p>
                            <p>• <code>company</code> (name) will be created automatically if it doesn’t exist.</p>
                        </div>
                    </CardContent>

                    <CardFooter className="justify-between">
                        <div className="text-xs text-muted-foreground">Only valid rows will be created/updated.</div>
                        <div />
                    </CardFooter>
                </Card>
            </div>
        </AppLayout>
    )
}

function formatCell(v: any) {
    if (v == null || v === '') return <span className="text-muted-foreground">—</span>
    if (typeof v === 'boolean') return v ? 'Yes' : 'No'
    return String(v)
}
