// resources/js/Pages/FundMembers/Index.tsx
import React, { useEffect, useMemo, useRef, useState } from 'react'
import { Head, Link, router, usePage } from '@inertiajs/react'
import ReactPaginate from 'react-paginate'

import AppLayout from '@/layouts/app-layout'

import { Badge } from '@/components/ui/badge'
import { Button } from '@/components/ui/button'
import { Input } from '@/components/ui/input'
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from '@/components/ui/select'
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from '@/components/ui/table'

import { Search as SearchIcon, Upload } from 'lucide-react'

// Small debounce hook
function useDebouncedValue<T>(value: T, delay = 450) {
  const [debounced, setDebounced] = useState(value)
  useEffect(() => {
    const id = setTimeout(() => setDebounced(value), delay)
    return () => clearTimeout(id)
  }, [value, delay])
  return debounced
}

type Member = {
  id: number
  employee_number: string
  first_name: string
  middle_name?: string | null
  surname: string
  tag: string
  company?: { id: number; name: string } | null
  created_at?: string
}

type Pagination<T> = {
  data: T[]
  current_page: number
  last_page: number
  total: number
  per_page: number
}

type PageProps = {
  members: Pagination<Member>
  filters: {
    q?: string
    tag?: string
    company_id?: number | null
  }
  companies?: Array<{ id: number; name: string }>
}

export default function FundMembersIndex() {
  const pageProps = usePage().props as unknown as PageProps

  const companies = pageProps.companies ?? []
  const { members, filters } = pageProps

  const [q, setQ] = useState<string>(filters.q ?? '')
  const [tag, setTag] = useState<string>(filters.tag ?? '')
  const [companyId, setCompanyId] = useState<string>(
    filters.company_id ? String(filters.company_id) : ''
  )

  const debouncedQ = useDebouncedValue(q, 450)

  // @ts-ignore (Ziggy global)
  const indexRoute: string = route('fund_members.index')
  // @ts-ignore
  const bulkRoute: string = route('fund_members.bulk.form')

  const applyFilters = (page = 1) => {
    router.get(
      indexRoute,
      {
        page,
        q,
        tag,
        company_id: companyId || undefined,
      },
      { preserveState: true, replace: true }
    )
  }

  const handlePageChange = (selected: { selected: number }) => {
    applyFilters(selected.selected + 1)
  }

  const resetFilters = () => {
    setQ('')
    setTag('')
    setCompanyId('')
    router.get(indexRoute, {}, { preserveState: true, replace: true })
  }

  // Auto-run when typing in search input (debounced)
  const firstRun = useRef(true)
  useEffect(() => {
    if (firstRun.current) {
      firstRun.current = false
      return
    }
    router.get(
      indexRoute,
      {
        page: 1,
        q: debouncedQ,
        tag,
        company_id: companyId || undefined,
      },
      { preserveState: true, replace: true }
    )
  }, [debouncedQ])

  const totalCount = useMemo(() => members?.total ?? 0, [members])

  return (
    <AppLayout breadcrumbs={[{ title: 'Fund Members', href: indexRoute }]}>
      <Head title="Fund Members" />

      <div className="bg-background mx-8 my-6 rounded-xl p-6 shadow-sm">
        {/* Header + actions */}
        <div className="mb-4 flex flex-wrap items-center justify-between gap-3">
          <div>
            <h1 className="text-2xl font-bold">Fund Members</h1>
            <p className="text-sm text-muted-foreground">Simple registry for medical fund membership.</p>
          </div>

          <div className="flex gap-2">
            <Button asChild variant="secondary">
              <Link href={bulkRoute}>
                <Upload className="mr-2 h-4 w-4" />
                Bulk Import
              </Link>
            </Button>
          </div>
        </div>

        {/* Stats */}
        <div className="mb-4 grid grid-cols-1 gap-3 sm:grid-cols-3">
          <div className="rounded-lg border p-4 shadow-sm dark:bg-neutral-900">
            <div className="text-sm text-muted-foreground">Total Members</div>
            <div className="mt-1 text-2xl font-semibold">{totalCount}</div>
          </div>
          <div className="rounded-lg border p-4 shadow-sm dark:bg-neutral-900">
            <div className="text-sm text-muted-foreground">Default Tag</div>
            <div className="mt-1">
              <Badge className="bg-indigo-600 hover:bg-indigo-600">Providence Fund</Badge>
            </div>
          </div>
          <div className="rounded-lg border p-4 shadow-sm dark:bg-neutral-900">
            <div className="text-sm text-muted-foreground">Companies</div>
            <div className="mt-1 text-2xl font-semibold">{companies.length}</div>
          </div>
        </div>

        {/* Filters */}
        <div className="mb-6 grid grid-cols-1 gap-3 md:grid-cols-2 lg:grid-cols-4">
          {/* Search */}
          <div className="relative">
            <SearchIcon className="text-muted-foreground absolute left-3 top-1/2 h-4 w-4 -translate-y-1/2" />
            <Input
              value={q}
              onChange={(e) => setQ(e.target.value)}
              placeholder="Search by name, employee #, tag…"
              className="pl-9"
            />
          </div>

          {/* Tag */}
          <Select value={tag || undefined} onValueChange={setTag}>
            <SelectTrigger>
              <SelectValue placeholder="All Tags" />
            </SelectTrigger>
            <SelectContent>
              <SelectItem value="Providence Fund">Providence Fund</SelectItem>
            </SelectContent>
          </Select>

          {/* Company */}
          <Select
            value={companyId || undefined}
            onValueChange={setCompanyId}
          >
            <SelectTrigger>
              <SelectValue placeholder="All Companies" />
            </SelectTrigger>
            <SelectContent>
              {companies.map((c) => (
                <SelectItem key={c.id} value={String(c.id)}>
                  {c.name}
                </SelectItem>
              ))}
            </SelectContent>
          </Select>

          {/* Buttons */}
          <div className="col-span-full flex flex-wrap gap-2 md:col-span-1 lg:col-span-1">
            <Button onClick={() => applyFilters()}>Search</Button>
            <Button variant="outline" onClick={resetFilters}>Reset</Button>
          </div>
        </div>

        {/* Table */}
        <div className="overflow-x-auto rounded-xl border">
          <Table className="min-w-[900px]">
            <TableHeader>
              <TableRow>
                <TableHead>Employee #</TableHead>
                <TableHead>Name</TableHead>
                <TableHead>Company</TableHead>
                <TableHead>Tag</TableHead>
                <TableHead>Created</TableHead>
              </TableRow>
            </TableHeader>
            <TableBody>
              {members.data.length === 0 && (
                <TableRow>
                  <TableCell colSpan={5} className="py-6 text-center text-muted-foreground">
                    No members found.
                  </TableCell>
                </TableRow>
              )}

              {members.data.map((m) => {
                const fullName = [m.first_name, m.middle_name, m.surname].filter(Boolean).join(' ')
                return (
                  <TableRow key={m.id}>
                    <TableCell className="whitespace-nowrap">{m.employee_number ?? '-'}</TableCell>
                    <TableCell className="whitespace-nowrap">{fullName || '-'}</TableCell>
                    <TableCell className="whitespace-nowrap">
                      {m.company ? <Badge variant="secondary">{m.company.name}</Badge> : <span className="text-muted-foreground">-</span>}
                    </TableCell>
                    <TableCell className="whitespace-nowrap">
                      <Badge className="bg-indigo-600 hover:bg-indigo-600">{m.tag}</Badge>
                    </TableCell>
                    <TableCell className="whitespace-nowrap text-muted-foreground">
                      {m.created_at ? new Date(m.created_at).toISOString().slice(0, 10) : '-'}
                    </TableCell>
                  </TableRow>
                )
              })}
            </TableBody>
          </Table>
        </div>

        {/* Pagination */}
        <div className="mt-4 flex justify-center">
          <ReactPaginate
            pageCount={members.last_page}
            forcePage={(members.current_page ?? 1) - 1}
            onPageChange={handlePageChange}
            marginPagesDisplayed={1}
            pageRangeDisplayed={3}
            previousLabel="← Prev"
            nextLabel="Next →"
            breakLabel="…"
            containerClassName="flex items-center gap-2 text-sm"
            pageClassName="px-3 py-1 border rounded hover:bg-muted"
            activeClassName="bg-primary text-primary-foreground"
            previousClassName="px-3 py-1 border rounded hover:bg-muted"
            nextClassName="px-3 py-1 border rounded hover:bg-muted"
            breakClassName="px-2"
          />
        </div>
      </div>
    </AppLayout>
  )
}
