import { Head, Link, router, usePage } from '@inertiajs/react';
import React from 'react';
import { Button } from '@/components/ui/button';
import { Card, CardHeader, CardContent, CardTitle } from '@/components/ui/card';
import { Tornado, Stethoscope, HeartPulse, Hospital, ClipboardCheck } from 'lucide-react';
import { useMobileNavigation } from '@/hooks/use-mobile-navigation';
import { toast } from 'sonner';

type ModuleSlug = 'dental' | 'primary' | 'chronics' | 'secondary' | 'medicals';

type PageProps = {
  active_module?: ModuleSlug | null;
  allowed?: ModuleSlug[];
  auth?: { user?: { name?: string } };
};

type ModuleTile = {
  slug: ModuleSlug;
  title: string;
  description: string;
  iconBg: string;
  iconColor: string;
  Icon: React.ComponentType<React.SVGProps<SVGSVGElement>>;
  comingSoon?: boolean;
};

// ✅ Secondary & Medicals now enabled (no comingSoon flag)
const ALL_TILES: ModuleTile[] = [
  { slug: 'dental',    title: 'Dental Module',        description: 'Dental patient flow and treatments.',      iconBg: 'bg-blue-100',    iconColor: 'text-blue-700',    Icon: Tornado },
  { slug: 'primary',   title: 'Primary Health Module',description: 'Consultations, vitals, referrals and more.',iconBg: 'bg-emerald-100',  iconColor: 'text-emerald-700', Icon: Stethoscope },
  { slug: 'chronics',  title: 'Chronics Module',      description: 'Long-term condition management.',          iconBg: 'bg-indigo-100',  iconColor: 'text-indigo-700',  Icon: HeartPulse },
  { slug: 'secondary', title: 'Secondary Health',     description: 'Specialized clinical services.',           iconBg: 'bg-amber-100',   iconColor: 'text-amber-700',   Icon: Hospital },
  { slug: 'medicals',  title: 'Medicals',             description: 'Periodic medical assessments.',            iconBg: 'bg-cyan-100',    iconColor: 'text-cyan-700',    Icon: ClipboardCheck },
];

export default function KioskIndex() {
  const { props } = usePage<PageProps>();
  const active = props.active_module ?? null;
  const allowed = props.allowed ?? [];
  const userName = props.auth?.user?.name ?? 'User';

  const cleanup = useMobileNavigation();

  const choose = (module: ModuleSlug) => {
    if (!allowed.includes(module)) {
      toast.error('You do not have access to that module.');
      return;
    }

    toast.promise(
      new Promise<void>((resolve, reject) => {
        router.post(route('kiosk.choose'), { module }, {
          preserveScroll: true,
          onSuccess: () => resolve(),
          onError: () => reject(new Error('Failed to switch module.')),
        });
      }),
      {
        loading: 'Switching module…',
        success: `Switched to ${module.toUpperCase()} module.`,
        error: (e) => e.message || 'Failed to switch module.',
      }
    );
  };

  const handleChangeModule = () => {
    toast.promise(
      new Promise<void>((resolve, reject) => {
        router.post(route('kiosk.reset'), {}, {
          preserveScroll: true,
          onSuccess: () => resolve(),
          onError: () => reject(new Error('Could not reset module selection.')),
        });
      }),
      {
        loading: 'Resetting selection…',
        success: 'You can now choose a different module.',
        error: (e) => e.message || 'Something went wrong.',
      }
    );
  };

  const handleLogout = () => {
    cleanup();
    // Inertia will handle the POST via Link below.
  };

  const tiles = ALL_TILES.filter((t) => allowed.includes(t.slug));
  const threeUp = tiles.length > 2;

  return (
    <>
      <Head title="Choose a Module" />

      <div className="mx-auto my-12 max-w-5xl px-4">
        <div className="mb-8 text-center">
          <h1 className="text-3xl font-bold">Welcome, {userName}</h1>
          <p className="text-muted-foreground mt-2">Please choose the module you want to use.</p>
          {active && (
            <p className="mt-2 text-sm">
              Current selection: <span className="font-semibold uppercase">{active}</span>
            </p>
          )}
        </div>

        {tiles.length === 0 ? (
          <div className="rounded-md border p-6 text-center text-sm text-muted-foreground">
            You don’t have access to any modules yet. Please contact an administrator.
          </div>
        ) : (
          <div className="flex flex-wrap justify-center gap-6">
            {tiles.map(({ slug, title, description, Icon, iconBg, iconColor }) => (
              <Card
                key={slug}
                className={[
                  'transition hover:shadow-md',
                  'w-full',                 // full width on very small screens
                  'sm:w-[320px]',          // fixed width on small screens
                  threeUp ? 'md:basis-1/3 md:max-w-[360px]' : 'md:w-[320px]', // 3-up on md+ when > 2
                ].join(' ')}
              >
                <CardHeader className="text-center">
                  <div className={`mx-auto mb-3 flex h-14 w-14 items-center justify-center rounded-full ${iconBg} ${iconColor}`}>
                    <Icon className="h-7 w-7" />
                  </div>
                  <CardTitle>{title}</CardTitle>
                </CardHeader>
                <CardContent className="text-center">
                  <p className="text-muted-foreground mb-4 text-sm">{description}</p>
                  <Button className="w-full" onClick={() => choose(slug)}>
                    Continue
                  </Button>
                </CardContent>
              </Card>
            ))}
          </div>
        )}

        {active && (
          <div className="mt-8 flex justify-center">
            <Button variant="outline" onClick={handleChangeModule}>
              Change Module
            </Button>
          </div>
        )}

        <div className="mt-6 flex justify-center">
          <Button variant="destructive" asChild>
            <Link
              className="block w-full max-w-[200px] text-center"
              method="post"
              href={route('logout')}
              as="button"
              onClick={handleLogout}
            >
              Logout
            </Link>
          </Button>
        </div>
      </div>
    </>
  );
}
