import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { API } from '@/config';
import AppLayout from '@/layouts/app-layout';
import { Head, useForm } from '@inertiajs/react';
import { CalendarDays, ClipboardList, Stethoscope } from 'lucide-react';
import { FormEvent } from 'react';

interface Props {
    consultationId: number;
    patient: {
        id: number;
        first_name: string;
        surname: string;
    };
    consultation: {
        id: number;
        consultation_date: string;
        presenting_complaint: string;
        clinical_notes: string;
        diagnosis: string;
        treatment_plan: string;
        doctor: any;
    };
    labTechnicians: {
        id: number;
        first_name: string;
        surname: string;
    }[];
    availableTests: string[];
}

export default function LabTestCreate({ consultationId, patient, consultation, labTechnicians, availableTests }: Props) {
    const { data, setData, post, processing, errors } = useForm({
        test_name: '',
        result_value: '',
        result_file: null as File | null,
        lab_technician_id: '',
    });

    const handleSubmit = (e: FormEvent) => {
        e.preventDefault();
        post(route('lab-tests.store', { consultation: consultationId }));
    };

    return (
        <AppLayout
            breadcrumbs={[
                { title: 'Consultations', href: '/consultations' },
                {
                    title: `Patient: ${patient.first_name} ${patient.surname}`,
                    href: `${API}/consultations/${consultationId}`,
                },
                { title: 'Create Lab Test', href: '#' },
            ]}
        >
            <Head title="Create Lab Test" />
            <div className="bg-background text-foreground mx-8 my-6 flex flex-col gap-4 rounded-xl p-6 shadow-sm">
                <h1 className="text-2xl font-bold">
                    Create Lab Test for{' '}
                    <span style={{ textTransform: 'capitalize' }}>
                        {patient.first_name} {patient.surname}
                    </span>
                </h1>

                <div className="grid grid-cols-12 gap-8">
                    <div className="col-span-8 space-y-6">
                        <div className="bg-card rounded-lg border p-6 shadow-sm dark:bg-neutral-900">
                            <form onSubmit={handleSubmit} className="grid grid-cols-1 gap-4">
                                <div className="space-y-2">
                                    <Label htmlFor="test_name">Test Name</Label>
                                    <select
                                        id="test_name"
                                        value={data.test_name}
                                        onChange={(e) => setData('test_name', e.target.value)}
                                        className="w-full rounded-md border border-gray-300 bg-white px-3 py-2 text-sm text-gray-900 dark:border-gray-700 dark:bg-black dark:text-white"
                                    >
                                        <option value="">Select Test</option>
                                        {availableTests.map((test) => (
                                            <option key={test} value={test}>
                                                {test}
                                            </option>
                                        ))}
                                    </select>
                                    {errors.test_name && <div className="text-sm text-red-600">{errors.test_name}</div>}
                                </div>

                                <div className="space-y-2">
                                    <Label htmlFor="lab_technician_id">Assign Lab Technician</Label>
                                    <select
                                        id="lab_technician_id"
                                        value={data.lab_technician_id}
                                        onChange={(e) => setData('lab_technician_id', e.target.value)}
                                        className="w-full rounded-md border border-gray-300 bg-white px-3 py-2 text-sm text-gray-900 dark:border-gray-700 dark:bg-black dark:text-white"
                                    >
                                        <option value="">Select Technician</option>
                                        {labTechnicians.map((tech) => (
                                            <option key={tech.id} value={tech.id}>
                                                {tech.first_name} {tech.surname}
                                            </option>
                                        ))}
                                    </select>
                                    {errors.lab_technician_id && <div className="text-sm text-red-600">{errors.lab_technician_id}</div>}
                                </div>

                                <div className="space-y-2">
                                    <Label htmlFor="result_value">Result Notes (Optional)</Label>
                                    <textarea
                                        id="result_value"
                                        value={data.result_value}
                                        onChange={(e) => setData('result_value', e.target.value)}
                                        className="w-full rounded-md border border-gray-300 bg-white px-3 py-2 text-sm text-gray-900 dark:border-gray-700 dark:bg-black dark:text-white"
                                        rows={4}
                                    />
                                    {errors.result_value && <div className="text-sm text-red-600">{errors.result_value}</div>}
                                </div>

                                <div className="space-y-2">
                                    <Label htmlFor="result_file">Attach File (Optional)</Label>
                                    <Input
                                        type="file"
                                        id="result_file"
                                        onChange={(e) => setData('result_file', e.target.files?.[0] ?? null)}
                                        className="w-full rounded-md border border-gray-300 bg-white px-3 py-2 text-sm text-gray-900 dark:border-gray-700 dark:bg-black dark:text-white"
                                    />
                                    {errors.result_file && <div className="text-sm text-red-600">{errors.result_file}</div>}
                                </div>

                                <div className="mt-4">
                                    <Button
                                        type="submit"
                                        disabled={processing}
                                        className="inline-flex items-center gap-2 rounded-lg bg-green-600 px-5 py-2 text-sm font-medium text-white hover:bg-green-700"
                                    >
                                        {processing ? 'Submitting...' : 'Submit Lab Test'}
                                    </Button>
                                </div>
                            </form>
                        </div>
                    </div>
                    {/* Consultation Summary Sidebar */}
                    <div className="col-span-4 space-y-6">
                        <div className="bg-card rounded-lg border p-6 shadow-sm dark:bg-neutral-900">
                            <h2 className="mb-4 flex items-center gap-2 text-lg font-semibold">
                                <ClipboardList className="h-5 w-5" />
                                Consultation Summary
                            </h2>

                            <div className="space-y-4">
                                <div>
                                    <div className="text-muted-foreground flex items-center gap-2 text-sm">
                                        <CalendarDays className="h-4 w-4" />
                                        <span>Consultation Date</span>
                                    </div>
                                    <p className="mt-1 text-sm">{new Date(consultation.consultation_date).toLocaleDateString()}</p>
                                </div>

                                <div>
                                    <div className="text-muted-foreground flex items-center gap-2 text-sm">
                                        <Stethoscope className="h-4 w-4" />
                                        <span>Doctor</span>
                                    </div>
                                    <p className="mt-1 text-sm">Dr. {consultation.doctor.name}</p>
                                </div>

                                <div>
                                    <h3 className="text-sm font-medium">Presenting Complaint</h3>
                                    <p className="text-muted-foreground mt-1 text-sm">{consultation.presenting_complaint || 'Not specified'}</p>
                                </div>

                                <div>
                                    <h3 className="text-sm font-medium">Diagnosis</h3>
                                    <p className="text-muted-foreground mt-1 text-sm">{consultation.diagnosis || 'Not specified'}</p>
                                </div>

                                <div>
                                    <h3 className="text-sm font-medium">Treatment Plan</h3>
                                    <p className="text-muted-foreground mt-1 text-sm">{consultation.treatment_plan || 'Not specified'}</p>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </AppLayout>
    );
}
