import { useState } from "react";
import { Head, usePage, router } from "@inertiajs/react";
import { Input } from "@/components/ui/input";
import AppLayout from '@/layouts/app-layout';
import { Button } from "@/components/ui/button";
import {
    Table,
    TableHeader,
    TableRow,
    TableHead,
    TableBody,
    TableCell,
} from "@/components/ui/table";
import { Card, CardHeader, CardTitle, CardContent } from "@/components/ui/card";
import { Select, SelectTrigger, SelectValue, SelectContent, SelectItem } from "@/components/ui/select";
import { Pagination, PaginationContent, PaginationItem, PaginationLink } from "@/components/ui/pagination";
import { API } from "@/config";
import { Modal } from "antd";

interface Log {
    id: number;
    user_name: string;
    action: string;
    description: string;
    loggable_type: string;
    loggable_id: number;
    old_values: Record<string, any> | null;
    new_values: Record<string, any> | null;
    ip_address: string | null;
    user_agent: string | null;
    created_at: string;
}

interface Props {
    logs: {
        data: Log[];
        links: { url: string | null; label: string; active: boolean }[];
    };
    filters: {
        search?: string;
        action?: string;
    };
    [key: string]: any;
}

export default function Index() {
    const { logs, filters } = usePage<Props>().props;
    const [search, setSearch] = useState(filters.search || "");
    const [action, setAction] = useState(filters.action || "");
    const [selectedLog, setSelectedLog] = useState<Log | null>(null);
    const [modalOpen, setModalOpen] = useState(false);

    const handleFilter = () => {
        router.get(route("logs.index"), { search, action }, { preserveState: true, replace: true });
    };

    const openModal = (log: Log) => {
        setSelectedLog(log);
        setModalOpen(true);
    };

    return (
        <AppLayout breadcrumbs={[{ title: 'Logs', href: `${API}/logs` }]}>
            <Head title="Logs" />
            <Card className="m-2">
                <CardHeader>
                    <CardTitle>System Logs</CardTitle>
                </CardHeader>
                <CardContent>
                    <div className="flex items-center gap-4 mb-4">
                        <Input
                            placeholder="Search logs..."
                            value={search}
                            onChange={(e) => setSearch(e.target.value)}
                            className="max-w-xs"
                        />
                        <Select value={action || "all"} onValueChange={(value) => setAction(value)}>
                            <SelectTrigger className="w-[180px]">
                                <SelectValue placeholder="Filter by action" />
                            </SelectTrigger>
                            <SelectContent>
                                <SelectItem value=" ">All</SelectItem>
                                <SelectItem value="created">Created</SelectItem>
                                <SelectItem value="updated">Updated</SelectItem>
                                <SelectItem value="deleted">Deleted</SelectItem>
                                <SelectItem value="create">Create</SelectItem>
                                <SelectItem value="approve">Approved</SelectItem>
                                <SelectItem value="transfer">Transfer</SelectItem>
                            </SelectContent>
                        </Select>
                        <Button onClick={handleFilter}>Apply</Button>
                    </div>

                    <Table>
                        <TableHeader>
                            <TableRow>
                                <TableHead>User</TableHead>
                                <TableHead>Action</TableHead>
                                <TableHead>Description</TableHead>
                                <TableHead>Model</TableHead>
                                <TableHead>IP</TableHead>
                                <TableHead>Date</TableHead>
                                <TableHead>Changes</TableHead>
                            </TableRow>
                        </TableHeader>
                        <TableBody>
                            {logs.data.length > 0 ? (
                                logs.data.map((log) => (
                                    <TableRow key={log.id}>
                                        <TableCell>{log.user_name || "-"}</TableCell>
                                        <TableCell className="capitalize">{log.action}</TableCell>
                                        <TableCell>{log.description}</TableCell>
                                        <TableCell>
                                            {log.loggable_type}#{log.loggable_id}
                                        </TableCell>
                                        <TableCell>{log.ip_address || "-"}</TableCell>
                                        <TableCell>{new Date(log.created_at).toLocaleString()}</TableCell>
                                        <TableCell>
                                            {log.old_values || log.new_values ? (
                                                <Button
                                                    size="sm"
                                                    variant="outline"
                                                    onClick={() => openModal(log)}
                                                    style={{
                                                        cursor: "pointer"
                                                    }}
                                                >
                                                    View Changes
                                                </Button>
                                            ) : (
                                                "-"
                                            )}
                                        </TableCell>
                                    </TableRow>
                                ))
                            ) : (
                                <TableRow>
                                    <TableCell colSpan={7} className="text-center text-muted-foreground py-4">
                                        No logs found
                                    </TableCell>
                                </TableRow>
                            )}
                        </TableBody>
                    </Table>

                    {/* Pagination */}
                    <div className="mt-4">
                        <Pagination>
                            <PaginationContent>
                                {logs.links.map((link, i) =>
                                    link.url ? (
                                        <PaginationItem key={i}>
                                            <PaginationLink
                                                href={link.url}
                                                isActive={link.active}
                                                dangerouslySetInnerHTML={{ __html: link.label }}
                                            />
                                        </PaginationItem>
                                    ) : (
                                        <PaginationItem key={i}>
                                            <span
                                                className="px-3 py-2 text-sm text-muted-foreground"
                                                dangerouslySetInnerHTML={{ __html: link.label }}
                                            />
                                        </PaginationItem>
                                    )
                                )}
                            </PaginationContent>
                        </Pagination>
                    </div>
                </CardContent>
            </Card>

            {/* AntD Modal with ShadCN Cards */}
            <Modal
                title={`Changes for Log #${selectedLog?.id}`}
                open={modalOpen}
                onCancel={() => setModalOpen(false)}
                footer={null}
                width={1000}
            >
                <div className="grid grid-cols-2 gap-4">
                    <Card>
                        <CardHeader>
                            <CardTitle>Old Values</CardTitle>
                        </CardHeader>
                        <CardContent>
                            <pre className="bg-muted p-2 rounded text-sm overflow-auto">
                                {JSON.stringify(selectedLog?.old_values, null, 2) || "—"}
                            </pre>
                        </CardContent>
                    </Card>
                    <Card>
                        <CardHeader>
                            <CardTitle>New Values</CardTitle>
                        </CardHeader>
                        <CardContent>
                            <pre className="bg-muted p-2 rounded text-sm overflow-auto">
                                {JSON.stringify(selectedLog?.new_values, null, 2) || "—"}
                            </pre>
                        </CardContent>
                    </Card>
                </div>
            </Modal>
        </AppLayout>
    );
}
