import { API } from '@/config';
import AppLayout from '@/layouts/app-layout';
import { Head, Link, router, usePage } from '@inertiajs/react';
import React, { useMemo, useState } from 'react';
import Swal from 'sweetalert2';
import 'sweetalert2/dist/sweetalert2.min.css';
import {
  Building2,
  ChevronLeft,
  ClipboardPlus,
  Info,
  Save,
} from 'lucide-react';

// shadcn/ui
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from '@/components/ui/select';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Separator } from '@/components/ui/separator';
import { Checkbox } from '@/components/ui/checkbox';

type Company = {
  id: number;
  company_name: string;
};

type Category = {
  id: number;
  slug: string;
  name: string;
};

type PageProps = {
  companies: Company[];
  categories: Category[]; // ⬅️ provided by controller
  defaults?: Partial<FormValues>;
};

type FormValues = {
  id_type: 'zimbabwean_id' | 'international_id';
  company_id: string;
  first_name: string;
  last_name: string;
  date_of_birth: string;
  gender: '' | 'MALE' | 'FEMALE';
  national_id: string;
  phone_number: string;
  employee_number: string;
  x_ray_status: 'PENDING' | 'DONE';
  exam_purpose: '1' | '2' | '3' | '4'; // 2 = Periodical by your legacy code
  country_code: string; // +263 etc
  last_x_ray: string; // "N/A" | "YYYY" | "YYYY-MM"
  // ⬇️ NEW: multi-categories by slug
  categories: string[];
  // legacy single: still sent (first category’s name) to keep compatibility
  category?: string;
  location?: string;
  created_at?: string;
};

const initialForm: FormValues = {
  id_type: 'zimbabwean_id',
  company_id: '',
  first_name: '',
  last_name: '',
  date_of_birth: '',
  gender: '',
  national_id: '',
  phone_number: '',
  employee_number: '',
  x_ray_status: 'PENDING',
  exam_purpose: '2', // Periodical
  country_code: '+263',
  last_x_ray: 'N/A',
  categories: [],     // ⬅️ start empty
  location: '',
  created_at: '',
};

// util: Zimbabwean national id formatting (when not international)
function formatZimNationalId(raw: string) {
  // remove non-alphanumeric and enforce upper
  let v = raw.replace(/[^A-Za-z0-9]/g, '').toUpperCase().slice(0, 16);
  if (v.length > 2) v = v.replace(/(\d{2})/, '$1-'); // 12-...
  if (v.length > 10) v = v.replace(/(\d{4,7})([A-Za-z])/, '$1-$2-');
  if (v.length > 13) v = v.replace(/([A-Za-z])(\d{1,2})$/, '$1-$2');
  return v;
}

export default function MedicalsAttendeesCreate() {
  const { companies, categories } = usePage<PageProps>().props;

  const [values, setValues] = useState<FormValues>(initialForm);

  // post-submit behavior (just like legacy)
  const [redirectToIndex, setRedirectToIndex] = useState<boolean>(true);
  const [continueAdding, setContinueAdding] = useState<boolean>(false);
  const backHref = `${API}/medicals/attendees`;

  // client validation similar to your prior pattern
  const errors = useMemo(() => {
    const e: Partial<Record<keyof FormValues, string>> = {};

    if (!values.company_id) e.company_id = 'Please select a company';
    if (!values.first_name) e.first_name = 'First name is required';
    if (!values.last_name) e.last_name = 'Last name is required';
    if (!values.date_of_birth) e.date_of_birth = 'Date of birth is required';
    if (!values.gender) e.gender = 'Select your gender';

    if (values.id_type === 'zimbabwean_id' && !values.national_id) {
      e.national_id = 'Zimbabwean ID is required';
    }

    if (!values.country_code) e.country_code = 'Select country code';
    if (!values.phone_number) {
      e.phone_number = 'Please enter the Phone Number';
    } else if (!/^\d{9}$/.test(values.phone_number)) {
      e.phone_number = 'Phone Number must be 9 digits (no leading 0)';
    }

    if (!values.x_ray_status) e.x_ray_status = 'Please select the X-ray status';
    if (!values.exam_purpose) e.exam_purpose = 'Please select the exam purpose';

    // ⬇️ NEW: require at least one category
    if (!values.categories || values.categories.length === 0) {
      e.category = 'Please select at least one category';
    }

    if (values.last_x_ray && values.last_x_ray !== 'N/A') {
      if (!/^(N\/A|\d{4}|\d{4}-\d{2})$/.test(values.last_x_ray)) {
        e.last_x_ray = 'Use YYYY or YYYY-MM, or N/A';
      }
    }

    return e;
  }, [values]);

  const hasErrors = Object.keys(errors).length > 0;

  const set = <K extends keyof FormValues>(key: K, val: FormValues[K]) =>
    setValues((v) => ({ ...v, [key]: val }));

  const handleNationalIdChange = (raw: string) => {
    if (values.id_type === 'international_id') {
      set('national_id', raw.replace(/\s+/g, '').toUpperCase());
      return;
    }
    set('national_id', formatZimNationalId(raw));
  };

  // Handle multi-category checkbox toggling (by slug)
  const toggleCategory = (slug: string, checked: boolean) => {
    setValues((prev) => {
      const setNew = new Set(prev.categories);
      if (checked) setNew.add(slug);
      else setNew.delete(slug);
      return { ...prev, categories: Array.from(setNew) };
    });
  };

  const handleSubmit = async (e?: React.FormEvent) => {
    e?.preventDefault();

    if (hasErrors) {
      Swal.fire({
        icon: 'error',
        title: 'Please fix the highlighted fields',
        confirmButtonColor: '#EF4444',
      });
      return;
    }

    // Find the primary (first) selected category NAME for legacy string
    const firstSlug = values.categories[0];
    const firstCat = categories.find((c) => c.slug === firstSlug);
    const legacyCategoryName = firstCat?.name || '';

    const payload: any = {
      ...values,
      first_name: values.first_name.toUpperCase(),
      last_name: values.last_name.toUpperCase(),
      national_id: values.national_id.toUpperCase(),
      employee_number: values.employee_number?.toUpperCase() ?? '',
      // ⬇️ NEW: send multi-categories (slugs)
      categories: values.categories,
      // ⬇️ keep legacy single for backward-compat server-side
      category: legacyCategoryName,
    };

    router.post(route('medicals.attendees.store'), payload, {
      preserveScroll: true,
      onSuccess: () => {
        Swal.fire({
          icon: 'success',
          title: 'Success',
          text: 'New Client Successfully Added',
          timer: 3500,
          confirmButtonColor: '#007a41',
        });

        if (redirectToIndex) {
          router.visit(route('medicals.attendees.index'));
          return;
        }

        if (continueAdding) {
          // quick loop: preserve commonly repeated fields
          setValues({
            ...initialForm,
            company_id: values.company_id,
            country_code: values.country_code,
            location: values.location ?? '',
            x_ray_status: 'PENDING',
            exam_purpose: '2',
            last_x_ray: 'N/A',
          });
          return;
        }

        router.visit(route('medicals.attendees.index'));
      },
      onError: (errs: any) => {
        Swal.fire({
          icon: 'error',
          title: 'Could not save',
          text: errs?.error ?? 'An error occurred while saving the attendee.',
          confirmButtonColor: '#EF4444',
        });
      },
    });
  };

  return (
    <AppLayout
      breadcrumbs={[
        { title: 'Medicals', href: `${API}/medicals` },
        { title: 'Attendees', href: `${API}/medicals/attendees` },
        { title: 'Add', href: '#' },
      ]}
    >
      <Head title="Medicals • Add Attendee" />

      <form onSubmit={handleSubmit} className="px-4 py-6">
        <div className="mb-4 flex items-center justify-between gap-3">
          <div className="flex items-center gap-2">
            <Button asChild variant="outline">
              <Link href={route('medicals.attendees.index')}>
                <ChevronLeft className="mr-2 h-4 w-4" />
                Back
              </Link>
            </Button>
            <h1 className="text-2xl font-bold">Add Attendee</h1>
          </div>

          <Button type="submit" className="gap-2">
            <Save className="h-4 w-4" />
            Save Attendee
          </Button>
        </div>

        <div className="grid grid-cols-1 gap-6 md:grid-cols-3">
          {/* LEFT: Form */}
          <Card className="md:col-span-2">
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <ClipboardPlus className="h-5 w-5" />
                Client Details
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-5">
              {/* Names */}
              <div className="grid grid-cols-1 gap-4 md:grid-cols-2">
                <div>
                  <Label htmlFor="first_name">First Name</Label>
                  <Input
                    id="first_name"
                    value={values.first_name}
                    onChange={(e) => set('first_name', e.target.value)}
                    placeholder="Enter first name"
                  />
                  {errors.first_name && (
                    <p className="mt-1 text-xs text-red-600">{errors.first_name}</p>
                  )}
                </div>
                <div>
                  <Label htmlFor="last_name">Last Name</Label>
                  <Input
                    id="last_name"
                    value={values.last_name}
                    onChange={(e) => set('last_name', e.target.value)}
                    placeholder="Enter last name"
                  />
                  {errors.last_name && (
                    <p className="mt-1 text-xs text-red-600">{errors.last_name}</p>
                  )}
                </div>
              </div>

              {/* Company / Gender */}
              <div className="grid grid-cols-1 gap-4 md:grid-cols-2">
                <div>
                  <Label>Company</Label>
                  <Select
                    value={values.company_id}
                    onValueChange={(v) => set('company_id', v)}
                  >
                    <SelectTrigger>
                      <SelectValue placeholder="Select a company" />
                    </SelectTrigger>
                    <SelectContent>
                      {companies.map((c) => (
                        <SelectItem key={c.id} value={String(c.id)}>
                          <div className="flex items-center gap-2">
                            <Building2 className="h-4 w-4" />
                            {c.company_name}
                          </div>
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                  {errors.company_id && (
                    <p className="mt-1 text-xs text-red-600">{errors.company_id}</p>
                  )}
                </div>

                <div>
                  <Label>Gender</Label>
                  <Select
                    value={values.gender}
                    onValueChange={(v: 'MALE' | 'FEMALE') => set('gender', v)}
                  >
                    <SelectTrigger>
                      <SelectValue placeholder="Select gender" />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="MALE">MALE</SelectItem>
                      <SelectItem value="FEMALE">FEMALE</SelectItem>
                    </SelectContent>
                  </Select>
                  {errors.gender && (
                    <p className="mt-1 text-xs text-red-600">{errors.gender}</p>
                  )}
                </div>
              </div>

              {/* Employee / National ID */}
              <div className="grid grid-cols-1 gap-4 md:grid-cols-3">
                <div>
                  <Label htmlFor="employee_number">Employee Number</Label>
                  <Input
                    id="employee_number"
                    value={values.employee_number}
                    onChange={(e) => set('employee_number', e.target.value)}
                    placeholder="EMP123…"
                  />
                </div>

                <div>
                  <Label>ID Type</Label>
                  <Select
                    value={values.id_type}
                    onValueChange={(v: 'zimbabwean_id' | 'international_id') =>
                      set('id_type', v)
                    }
                  >
                    <SelectTrigger>
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="zimbabwean_id">Zimbabwean ID</SelectItem>
                      <SelectItem value="international_id">International ID</SelectItem>
                    </SelectContent>
                  </Select>
                </div>

                <div>
                  <Label htmlFor="national_id">National ID</Label>
                  <Input
                    id="national_id"
                    value={values.national_id}
                    onChange={(e) => handleNationalIdChange(e.target.value)}
                    placeholder={values.id_type === 'international_id' ? 'Passport/ID' : '12-3456789-A-12'}
                  />
                  {errors.national_id && (
                    <p className="mt-1 text-xs text-red-600">{errors.national_id}</p>
                  )}
                </div>
              </div>

              {/* Phone */}
              <div className="grid grid-cols-[110px_1fr] gap-2 md:max-w-md">
                <div>
                  <Label>Code</Label>
                  <Select
                    value={values.country_code}
                    onValueChange={(v) => set('country_code', v)}
                  >
                    <SelectTrigger>
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="+263">+263</SelectItem>
                    </SelectContent>
                  </Select>
                  {errors.country_code && (
                    <p className="mt-1 text-xs text-red-600">{errors.country_code}</p>
                  )}
                </div>
                <div>
                  <Label htmlFor="phone_number">Phone Number</Label>
                  <Input
                    id="phone_number"
                    value={values.phone_number}
                    onChange={(e) => set('phone_number', e.target.value.replace(/\D/g, '').slice(0, 9))}
                    placeholder="9 digits, no leading 0"
                  />
                  {errors.phone_number && (
                    <p className="mt-1 text-xs text-red-600">{errors.phone_number}</p>
                  )}
                </div>
              </div>

              {/* Dates / x-ray / purpose */}
              <div className="grid grid-cols-1 gap-4 md:grid-cols-3">
                <div>
                  <Label htmlFor="date_of_birth">Date of Birth</Label>
                  <Input
                    id="date_of_birth"
                    type="date"
                    value={values.date_of_birth}
                    onChange={(e) => set('date_of_birth', e.target.value)}
                  />
                  {errors.date_of_birth && (
                    <p className="mt-1 text-xs text-red-600">{errors.date_of_birth}</p>
                  )}
                </div>

                <div>
                  <Label>X-ray Status</Label>
                  <Select
                    value={values.x_ray_status}
                    onValueChange={(v: 'PENDING' | 'DONE') => set('x_ray_status', v)}
                  >
                    <SelectTrigger>
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="PENDING">PENDING</SelectItem>
                      <SelectItem value="DONE">DONE</SelectItem>
                    </SelectContent>
                  </Select>
                  {errors.x_ray_status && (
                    <p className="mt-1 text-xs text-red-600">{errors.x_ray_status}</p>
                  )}
                </div>

                <div>
                  <Label>Exam Purpose</Label>
                  <Select
                    value={values.exam_purpose}
                    onValueChange={(v: FormValues['exam_purpose']) => set('exam_purpose', v)}
                  >
                    <SelectTrigger>
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="2">Periodical</SelectItem>
                    </SelectContent>
                  </Select>
                  {errors.exam_purpose && (
                    <p className="mt-1 text-xs text-red-600">{errors.exam_purpose}</p>
                  )}
                </div>
              </div>

              {/* Multi-category selection (checkbox group) */}
              <div>
                <Label>Categories (choose one or more)</Label>
                <div className="mt-2 grid grid-cols-1 gap-2 sm:grid-cols-2 lg:grid-cols-3">
                  {categories.map((cat) => {
                    const checked = values.categories.includes(cat.slug);
                    return (
                      <label
                        key={cat.slug}
                        className="flex cursor-pointer items-center gap-2 rounded border p-2 hover:bg-accent"
                      >
                        <Checkbox
                          checked={checked}
                          onCheckedChange={(v) => toggleCategory(cat.slug, Boolean(v))}
                        />
                        <span className="text-sm">{cat.name}</span>
                      </label>
                    );
                  })}
                </div>
                {errors.category && (
                  <p className="mt-2 text-xs text-red-600">{errors.category}</p>
                )}
                <p className="mt-1 text-xs text-muted-foreground">
                  The first selected category will also be stored in the legacy <code>patients.category</code> field for backward compatibility.
                </p>
              </div>

              {/* Extra fields */}
              <div className="grid grid-cols-1 gap-4 md:grid-cols-2">
                <div>
                  <Label htmlFor="last_x_ray">Last Chest X-Ray (YYYY-MM / YYYY / N/A)</Label>
                  <Input
                    id="last_x_ray"
                    value={values.last_x_ray}
                    onChange={(e) => set('last_x_ray', e.target.value.toUpperCase())}
                    placeholder="N/A or 2020 or 2020-09"
                  />
                  {errors.last_x_ray && (
                    <p className="mt-1 text-xs text-red-600">{errors.last_x_ray}</p>
                  )}
                </div>

                <div>
                  <Label htmlFor="created_at">Medical Exam Date</Label>
                  <Input
                    id="created_at"
                    type="date"
                    value={values.created_at ?? ''}
                    onChange={(e) => set('created_at', e.target.value)}
                  />
                </div>
              </div>

              <Separator className="my-2" />

              {/* Post-submit behavior */}
              <div className="grid gap-2 md:grid-cols-2">
                <label className="flex items-center gap-2">
                  <Checkbox
                    checked={redirectToIndex}
                    onCheckedChange={(v) => setRedirectToIndex(Boolean(v))}
                  />
                  <span className="text-sm">Redirect after adding a new attendee</span>
                </label>
                <label className="flex items-center gap-2">
                  <Checkbox
                    checked={continueAdding}
                    onCheckedChange={(v) => setContinueAdding(Boolean(v))}
                  />
                  <span className="text-sm">Continue adding attendees</span>
                </label>
              </div>
            </CardContent>
          </Card>

          {/* RIGHT: Help panel */}
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Info className="h-5 w-5 text-blue-600" />
                What’s this form?
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-3 text-sm leading-6 text-muted-foreground">
              <p>
                This screen registers a new <strong>Medicals Attendee</strong> and prepares their linked Patient & records.
              </p>
              <ul className="ml-5 list-disc space-y-1">
                <li>Names, ID, and employee number are uppercased before save.</li>
                <li>Phone number uses your selected country code. Enter <em>9 digits</em>, no leading 0.</li>
                <li>X-ray status defaults to <strong>PENDING</strong>. Exam Purpose defaults to <strong>Periodical</strong>.</li>
                <li>Pick one or more <strong>Categories</strong>. The first one is also stored to the legacy single field.</li>
                <li>Optional <em>Medical Exam Date</em> timestamps downstream records to that day.</li>
              </ul>
              <Separator />
              <p className="text-xs">
                Tip: toggle “Continue adding attendees” to quickly register multiple people.
              </p>
              <Button asChild variant="outline" className="w-full">
                <Link href={backHref}>View Attendees</Link>
              </Button>
            </CardContent>
          </Card>
        </div>

        {/* Footer actions for small screens */}
        <div className="mt-6 flex items-center justify-between gap-2 md:hidden">
          <Button asChild variant="outline">
            <Link href={route('medicals.attendees.index')}>
              <ChevronLeft className="mr-2 h-4 w-4" />
              Back
            </Link>
          </Button>
          <Button type="submit" className="gap-2">
            <Save className="h-4 w-4" />
            Save
          </Button>
        </div>
      </form>
    </AppLayout>
  );
}
