// resources/js/pages/Medicals/Attendees/Edit.tsx
import { API } from '@/config';
import AppLayout from '@/layouts/app-layout';
import { Head, Link, router, usePage } from '@inertiajs/react';
import React, { useEffect, useMemo, useState } from 'react';
import Swal from 'sweetalert2';
import 'sweetalert2/dist/sweetalert2.min.css';
import {
  Building2,
  ChevronLeft,
  ClipboardPlus,
  Info,
  Save,
} from 'lucide-react';

// shadcn/ui
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from '@/components/ui/select';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Separator } from '@/components/ui/separator';
import { Checkbox } from '@/components/ui/checkbox';

type Company = {
  id: number;
  company_name: string;
};

type Attendee = {
  id: number;
  company_id: number | string;
  first_name: string;
  last_name: string;
  date_of_birth?: string | null;
  gender?: '' | 'MALE' | 'FEMALE' | null;
  national_id: string;
  phone_number?: string | null; // combined or raw 9 digits (server may join with country_code)
  employee_number?: string | null;
  x_ray_status?: 'PENDING' | 'DONE' | '' | null;
  exam_purpose?: '1' | '2' | '3' | '4' | '' | null;
  last_x_ray?: string | null;
  category?:
    | ''
    | 'Food Handler (COH)'
    | 'Pneumoconiosis'
    | 'Pre-Employement'
    | 'Exit-Pneumoconiosis'
    | 'Exit-Employement'
    | null;
  location?: string | null;
  created_at?: string | null; // ISO
};

type PageProps = {
  attendee: Attendee;
  companies: Company[];
};

type FormValues = {
  id_type: 'zimbabwean_id' | 'international_id';
  company_id: string;
  first_name: string;
  last_name: string;
  date_of_birth: string;
  gender: '' | 'MALE' | 'FEMALE';
  national_id: string;
  phone_number: string; // 9 digits only
  employee_number: string;
  x_ray_status: 'PENDING' | 'DONE';
  exam_purpose: '1' | '2' | '3' | '4';
  country_code: string; // +263
  last_x_ray: string; // "N/A" | "YYYY" | "YYYY-MM"
  category:
    | ''
    | 'Food Handler (COH)'
    | 'Pneumoconiosis'
    | 'Pre-Employement'
    | 'Exit-Pneumoconiosis'
    | 'Exit-Employement';
  location?: string;
  created_at?: string; // date (YYYY-MM-DD)
};

// util: Zimbabwean national id formatting (when not international)
function formatZimNationalId(raw: string) {
  let v = raw.replace(/[^A-Za-z0-9]/g, '').toUpperCase().slice(0, 16);
  if (v.length > 2) v = v.replace(/(\d{2})/, '$1-');
  if (v.length > 10) v = v.replace(/(\d{4,7})([A-Za-z])/, '$1-$2-');
  if (v.length > 13) v = v.replace(/([A-Za-z])(\d{1,2})$/, '$1-$2');
  return v;
}

// split existing phone into code and 9 digits
function splitPhone(full?: string | null) {
  if (!full) return { code: '+263', number: '' };
  if (full.startsWith('+263')) {
    const num = full.slice(4).replace(/\D/g, '').slice(0, 9);
    return { code: '+263', number: num };
  }
  // fallback: assume stored as digits only
  const digits = full.replace(/\D/g, '');
  if (digits.length === 9) return { code: '+263', number: digits };
  // last resort: trim to 9
  return { code: '+263', number: digits.slice(-9) };
}

export default function MedicalsAttendeesEdit() {
  const { attendee, companies } = usePage<PageProps>().props;

  const phoneParts = useMemo(() => splitPhone(attendee.phone_number), [attendee.phone_number]);
  const initialForm: FormValues = {
    id_type: 'zimbabwean_id',
    company_id: String(attendee.company_id ?? ''),
    first_name: attendee.first_name ?? '',
    last_name: attendee.last_name ?? '',
    date_of_birth: attendee.date_of_birth ?? '',
    gender: (attendee.gender as any) ?? '',
    national_id: attendee.national_id ?? '',
    phone_number: phoneParts.number,
    employee_number: attendee.employee_number ?? '',
    x_ray_status: (attendee.x_ray_status as any) || 'PENDING',
    exam_purpose: (attendee.exam_purpose as any) || '2',
    country_code: phoneParts.code || '+263',
    last_x_ray: attendee.last_x_ray ?? 'N/A',
    category: (attendee.category as any) ?? '',
    location: attendee.location ?? '',
    created_at: attendee.created_at ? attendee.created_at.slice(0, 10) : '',
  };

  const [values, setValues] = useState<FormValues>(initialForm);

  // options: after save
  const [goToShow, setGoToShow] = useState<boolean>(true);

  // validation (client-side light checks to mirror Create)
  const errors = useMemo(() => {
    const e: Partial<Record<keyof FormValues, string>> = {};

    if (!values.company_id) e.company_id = 'Please select a company';
    if (!values.first_name) e.first_name = 'First name is required';
    if (!values.last_name) e.last_name = 'Last name is required';
    if (!values.date_of_birth) e.date_of_birth = 'Date of birth is required';
    if (!values.gender) e.gender = 'Select your gender';

    if (values.id_type === 'zimbabwean_id' && !values.national_id) {
      e.national_id = 'Zimbabwean ID is required';
    }

    if (!values.country_code) e.country_code = 'Select country code';

    if (!values.phone_number) {
      e.phone_number = 'Please enter the Phone Number';
    } else if (!/^\d{9}$/.test(values.phone_number)) {
      e.phone_number = 'Phone Number must be 9 digits (no leading 0)';
    }

    if (!values.x_ray_status) e.x_ray_status = 'Please select the X-ray status';
    if (!values.exam_purpose) e.exam_purpose = 'Please select the exam purpose';
    if (!values.category) e.category = 'Please select a category';

    if (values.last_x_ray && values.last_x_ray !== 'N/A') {
      if (!/^(N\/A|\d{4}|\d{4}-\d{2})$/i.test(values.last_x_ray)) {
        e.last_x_ray = 'Use YYYY or YYYY-MM, or N/A';
      }
    }

    return e;
  }, [values]);

  const hasErrors = Object.keys(errors).length > 0;
  const set = <K extends keyof FormValues>(key: K, val: FormValues[K]) =>
    setValues((v) => ({ ...v, [key]: val }));

  const handleNationalIdChange = (raw: string) => {
    if (values.id_type === 'international_id') {
      set('national_id', raw.replace(/\s+/g, '').toUpperCase());
      return;
    }
    set('national_id', formatZimNationalId(raw));
  };

  const handleSubmit = async (e?: React.FormEvent) => {
    e?.preventDefault();

    if (hasErrors) {
      Swal.fire({
        icon: 'error',
        title: 'Please fix the highlighted fields',
        confirmButtonColor: '#EF4444',
      });
      return;
    }

    const confirm = await Swal.fire({
      title: 'Save changes?',
      text: 'This will update the attendee and related patient profile.',
      icon: 'question',
      showCancelButton: true,
      confirmButtonText: 'Save',
      cancelButtonText: 'Cancel',
      confirmButtonColor: '#2563eb',
      cancelButtonColor: '#6b7280',
      reverseButtons: true,
      focusCancel: true,
    });

    if (!confirm.isConfirmed) return;

    // Transformations (mirror create page behavior)
    const payload: any = {
      ...values,
      first_name: values.first_name.toUpperCase(),
      last_name: values.last_name.toUpperCase(),
      national_id: values.national_id.toUpperCase(),
      employee_number: values.employee_number?.toUpperCase() ?? '',
      // keep both fields; backend can concatenate or read both
      country_code: values.country_code,
      phone_number: values.phone_number,
    };

    router.put(route('medicals.attendees.update', attendee.id), payload, {
      preserveScroll: true,
      onSuccess: () => {
        Swal.fire({
          icon: 'success',
          title: 'Updated',
          timer: 1600,
          showConfirmButton: false,
        });
        if (goToShow) {
          router.visit(route('medicals.attendees.show', attendee.id));
        }
      },
      onError: (errs) => {
        const first = (errs && (Object.values(errs)[0] as string)) || 'Please review your inputs and try again.';
        Swal.fire({
          icon: 'error',
          title: 'Update failed',
          text: first,
        });
      },
    });
  };

  const backHref = `${API}/medicals/attendees`;

  return (
    <AppLayout
      breadcrumbs={[
        { title: 'Medicals', href: `${API}/medicals` },
        { title: 'Attendees', href: `${API}/medicals/attendees` },
        { title: 'Edit', href: '#' },
      ]}
    >
      <Head title="Medicals • Edit Attendee" />

      <form onSubmit={handleSubmit} className="px-4 py-6">
        <div className="mb-4 flex items-center justify-between gap-3">
          <div className="flex items-center gap-2">
            <Button asChild variant="outline">
              <Link href={route('medicals.attendees.index')}>
                <ChevronLeft className="mr-2 h-4 w-4" />
                Back
              </Link>
            </Button>
            <h1 className="text-2xl font-bold">Edit Attendee</h1>
          </div>

          <Button type="submit" className="gap-2">
            <Save className="h-4 w-4" />
            Save Changes
          </Button>
        </div>

        <div className="grid grid-cols-1 gap-6 md:grid-cols-3">
          {/* LEFT: Form */}
          <Card className="md:col-span-2">
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <ClipboardPlus className="h-5 w-5" />
                Client Details
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-5">
              {/* Names */}
              <div className="grid grid-cols-1 gap-4 md:grid-cols-2">
                <div>
                  <Label htmlFor="first_name">First Name</Label>
                  <Input
                    id="first_name"
                    value={values.first_name}
                    onChange={(e) => set('first_name', e.target.value)}
                    placeholder="Enter first name"
                  />
                  {errors.first_name && (
                    <p className="mt-1 text-xs text-red-600">{errors.first_name}</p>
                  )}
                </div>
                <div>
                  <Label htmlFor="last_name">Last Name</Label>
                  <Input
                    id="last_name"
                    value={values.last_name}
                    onChange={(e) => set('last_name', e.target.value)}
                    placeholder="Enter last name"
                  />
                  {errors.last_name && (
                    <p className="mt-1 text-xs text-red-600">{errors.last_name}</p>
                  )}
                </div>
              </div>

              {/* Company / Gender */}
              <div className="grid grid-cols-1 gap-4 md:grid-cols-2">
                <div>
                  <Label>Company</Label>
                  <Select
                    value={values.company_id}
                    onValueChange={(v) => set('company_id', v)}
                  >
                    <SelectTrigger>
                      <SelectValue placeholder="Select a company" />
                    </SelectTrigger>
                    <SelectContent>
                      {companies.map((c) => (
                        <SelectItem key={c.id} value={String(c.id)}>
                          <div className="flex items-center gap-2">
                            <Building2 className="h-4 w-4" />
                            {c.company_name}
                          </div>
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                  {errors.company_id && (
                    <p className="mt-1 text-xs text-red-600">{errors.company_id}</p>
                  )}
                </div>

                <div>
                  <Label>Gender</Label>
                  <Select
                    value={values.gender}
                    onValueChange={(v: 'MALE' | 'FEMALE') => set('gender', v)}
                  >
                    <SelectTrigger>
                      <SelectValue placeholder="Select gender" />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="MALE">MALE</SelectItem>
                      <SelectItem value="FEMALE">FEMALE</SelectItem>
                    </SelectContent>
                  </Select>
                  {errors.gender && (
                    <p className="mt-1 text-xs text-red-600">{errors.gender}</p>
                  )}
                </div>
              </div>

              {/* Employee / ID Type / National ID */}
              <div className="grid grid-cols-1 gap-4 md:grid-cols-3">
                <div>
                  <Label htmlFor="employee_number">Employee Number</Label>
                  <Input
                    id="employee_number"
                    value={values.employee_number}
                    onChange={(e) => set('employee_number', e.target.value)}
                    placeholder="EMP123…"
                  />
                </div>

                <div>
                  <Label>ID Type</Label>
                  <Select
                    value={values.id_type}
                    onValueChange={(v: 'zimbabwean_id' | 'international_id') => set('id_type', v)}
                  >
                    <SelectTrigger>
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="zimbabwean_id">Zimbabwean ID</SelectItem>
                      <SelectItem value="international_id">International ID</SelectItem>
                    </SelectContent>
                  </Select>
                </div>

                <div>
                  <Label htmlFor="national_id">National ID</Label>
                  <Input
                    id="national_id"
                    value={values.national_id}
                    onChange={(e) => handleNationalIdChange(e.target.value)}
                    placeholder={values.id_type === 'international_id' ? 'Passport/ID' : '12-3456789-A-12'}
                  />
                  {errors.national_id && (
                    <p className="mt-1 text-xs text-red-600">{errors.national_id}</p>
                  )}
                </div>
              </div>

              {/* Phone */}
              <div className="grid grid-cols-[110px_1fr] gap-2 md:max-w-md">
                <div>
                  <Label>Code</Label>
                  <Select
                    value={values.country_code}
                    onValueChange={(v) => set('country_code', v)}
                  >
                    <SelectTrigger>
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="+263">+263</SelectItem>
                    </SelectContent>
                  </Select>
                  {errors.country_code && (
                    <p className="mt-1 text-xs text-red-600">{errors.country_code}</p>
                  )}
                </div>
                <div>
                  <Label htmlFor="phone_number">Phone Number</Label>
                  <Input
                    id="phone_number"
                    value={values.phone_number}
                    onChange={(e) =>
                      set('phone_number', e.target.value.replace(/\D/g, '').slice(0, 9))
                    }
                    placeholder="9 digits, no leading 0"
                  />
                  {errors.phone_number && (
                    <p className="mt-1 text-xs text-red-600">{errors.phone_number}</p>
                  )}
                </div>
              </div>

              {/* Dates / x-ray / purpose / category */}
              <div className="grid grid-cols-1 gap-4 md:grid-cols-4">
                <div>
                  <Label htmlFor="date_of_birth">Date of Birth</Label>
                  <Input
                    id="date_of_birth"
                    type="date"
                    value={values.date_of_birth}
                    onChange={(e) => set('date_of_birth', e.target.value)}
                  />
                  {errors.date_of_birth && (
                    <p className="mt-1 text-xs text-red-600">{errors.date_of_birth}</p>
                  )}
                </div>

                <div>
                  <Label>X-ray Status</Label>
                  <Select
                    value={values.x_ray_status}
                    onValueChange={(v: 'PENDING' | 'DONE') => set('x_ray_status', v)}
                  >
                    <SelectTrigger>
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="PENDING">PENDING</SelectItem>
                      <SelectItem value="DONE">DONE</SelectItem>
                    </SelectContent>
                  </Select>
                  {errors.x_ray_status && (
                    <p className="mt-1 text-xs text-red-600">{errors.x_ray_status}</p>
                  )}
                </div>

                <div>
                  <Label>Exam Purpose</Label>
                  <Select
                    value={values.exam_purpose}
                    onValueChange={(v: FormValues['exam_purpose']) => set('exam_purpose', v)}
                  >
                    <SelectTrigger>
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      {/* <SelectItem value="1">Pre-Placement</SelectItem> */}
                      <SelectItem value="2">Periodical</SelectItem>
                    </SelectContent>
                  </Select>
                  {errors.exam_purpose && (
                    <p className="mt-1 text-xs text-red-600">{errors.exam_purpose}</p>
                  )}
                </div>

                <div>
                  <Label>Category</Label>
                  <Select
                    value={values.category}
                    onValueChange={(v: FormValues['category']) => set('category', v)}
                  >
                    <SelectTrigger>
                      <SelectValue placeholder="Select category" />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="Food Handler (COH)">Food Handler</SelectItem>
                      <SelectItem value="Pneumoconiosis">Pneumoconiosis</SelectItem>
                      <SelectItem value="Pre-Employement">Pre-Employment</SelectItem>
                      <SelectItem value="Exit-Pneumoconiosis">Exit (Pneumoconiosis)</SelectItem>
                      <SelectItem value="Exit-Employement">Exit-Employement</SelectItem>
                    </SelectContent>
                  </Select>
                  {errors.category && (
                    <p className="mt-1 text-xs text-red-600">{errors.category}</p>
                  )}
                </div>
              </div>

              <div className="grid grid-cols-1 gap-4 md:grid-cols-2">
                <div>
                  <Label htmlFor="last_x_ray">Last Chest X-Ray (YYYY-MM / YYYY / N/A)</Label>
                  <Input
                    id="last_x_ray"
                    value={values.last_x_ray}
                    onChange={(e) => set('last_x_ray', e.target.value.toUpperCase())}
                    placeholder="N/A or 2020 or 2020-09"
                  />
                  {errors.last_x_ray && (
                    <p className="mt-1 text-xs text-red-600">{errors.last_x_ray}</p>
                  )}
                </div>

                <div>
                  <Label htmlFor="created_at">Medical Exam Date</Label>
                  <Input
                    id="created_at"
                    type="date"
                    value={values.created_at ?? ''}
                    onChange={(e) => set('created_at', e.target.value)}
                  />
                </div>
              </div>

              <Separator className="my-2" />

              {/* Post-save behavior */}
              <div className="grid gap-2 md:grid-cols-2">
                <label className="flex items-center gap-2">
                  <Checkbox
                    checked={goToShow}
                    onCheckedChange={(v) => setGoToShow(Boolean(v))}
                  />
                  <span className="text-sm">Go to the attendee page after saving</span>
                </label>
              </div>
            </CardContent>
          </Card>

          {/* RIGHT: Help panel */}
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Info className="h-5 w-5 text-blue-600" />
                What’s this form?
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-3 text-sm leading-6 text-muted-foreground">
              <p>
                Update an existing <strong>Medicals Attendee</strong> and keep their linked Patient records consistent. This mirrors your create flow:
              </p>
              <ul className="ml-5 list-disc space-y-1">
                <li>Names, ID, and employee number are uppercased before save.</li>
                <li>
                  Phone number uses your selected country code. Enter <em>9 digits</em>, no leading 0.
                </li>
                <li>
                  X-ray status and Exam Purpose drive downstream logic (certificates, records).
                </li>
                <li>
                  Optional <em>Medical Exam Date</em> can align timestamps on related records.
                </li>
              </ul>
              <Separator />
              <div className="grid gap-2">
                <Button asChild variant="outline">
                  <Link href={route('medicals.attendees.show', attendee.id)}>Cancel & View</Link>
                </Button>
                <Button asChild variant="outline">
                  <Link href={route('medicals.attendees.index')}>Back to Attendees</Link>
                </Button>
              </div>
            </CardContent>
          </Card>
        </div>

        {/* Footer actions for small screens */}
        <div className="mt-6 flex items-center justify-between gap-2 md:hidden">
          <Button asChild variant="outline">
            <Link href={route('medicals.attendees.index')}>
              <ChevronLeft className="mr-2 h-4 w-4" />
              Back
            </Link>
          </Button>
          <Button type="submit" className="gap-2">
            <Save className="h-4 w-4" />
            Save
          </Button>
        </div>
      </form>
    </AppLayout>
  );
}
