import { API } from '@/config';
import AppLayout from '@/layouts/app-layout';
import { Head, Link, router, usePage } from '@inertiajs/react';
import React, { useEffect, useMemo, useRef, useState } from 'react';
import ReactPaginate from 'react-paginate';
import {
  Eye,
  Pencil,
  Plus,
  Search,
  Trash2,
  Columns,
  ArrowUpDown,
  ArrowUp,
  ArrowDown,
  Building2,
  BadgeCheck,
} from 'lucide-react';
import Swal from 'sweetalert2';

// shadcn/ui
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow
} from '@/components/ui/table';
import { Badge } from '@/components/ui/badge';
import {
  DropdownMenu,
  DropdownMenuCheckboxItem,
  DropdownMenuContent,
  DropdownMenuLabel,
  DropdownMenuSeparator,
  DropdownMenuTrigger,
} from '@/components/ui/dropdown-menu';

type Company = {
  id: number;
  company_name: string;
};

type Certificate = {
  id: number;
  patient_id: number;
  status: string;
  certificate_location?: string | null;
  created_at: string; // ISO
  updated_at: string; // ISO
};

type Attendee = {
  id: number;
  company: Company | null;
  employee_number?: string | null;
  exam_purpose?: string | null;
  category?: string | null;
  last_x_ray?: string | null;

  swab_number?: string | null;
  x_ray_status?: string | null;
  time_of_entry?: string | null; // ISO

  first_name: string;
  last_name: string;
  date_of_birth?: string | null;
  age?: number | null;
  gender?: string | null;
  national_id: string;
  phone_number?: string | null;

  latest_certificate?: Certificate | null;

  created_at?: string | null;
  updated_at?: string | null;
};

type PageProps = {
  attendees: {
    data: Attendee[];
    current_page: number;
    last_page: number;
    per_page: number;
    total: number;
  };
  filters: {
    search?: string;
    per_page?: number;
    sort?: 'created_at' | 'time_of_entry' | 'last_name';
    direction?: 'asc' | 'desc';
  };
};

// 🔹 Debounce
function useDebouncedValue<T>(value: T, delay = 450) {
  const [debounced, setDebounced] = useState(value);
  useEffect(() => {
    const id = setTimeout(() => setDebounced(value), delay);
    return () => clearTimeout(id);
  }, [value, delay]);
  return debounced;
}

const VISIBLE_COLS_STORAGE_KEY = 'medicals_attendees_visible_cols_v1';
type ColumnKey =
  | 'name'
  | 'national'
  | 'company'
  | 'category'
  | 'swab'
  | 'time'
  | 'cert'
  | 'created'
  | 'actions';

// 🔹 Age formatter: force integer years (e.g., 26.51 → 26y)
const formatAge = (age?: number | null) => {
  if (age == null || !Number.isFinite(age)) return '';
  return `${Math.floor(age)}y`; // use Math.round(age) if you prefer rounding
};

export default function MedicalsAttendeesIndex() {
  const { attendees, filters } = usePage<PageProps>().props;

  const baseHref = `${API}/medicals/attendees`;

  const [search, setSearch] = useState<string>(filters.search || '');
  const [sort, setSort] = useState<'created_at' | 'time_of_entry' | 'last_name'>(filters.sort || 'created_at');
  const [direction, setDirection] = useState<'asc' | 'desc'>(filters.direction || 'desc');

  const debouncedSearch = useDebouncedValue(search, 450);
  const firstRun = useRef(true);

  // ---- Dynamic columns (persisted)
  const defaultVisible: Record<ColumnKey, boolean> = {
    name: true,
    national: true,
    company: true,
    category: true,
    swab: true,
    time: true,
    cert: true,
    created: true,
    actions: true,
  };

  const [visibleCols, setVisibleCols] = useState<Record<ColumnKey, boolean>>(() => {
    try {
      const raw = localStorage.getItem(VISIBLE_COLS_STORAGE_KEY);
      if (raw) {
        const parsed = JSON.parse(raw) as Record<ColumnKey, boolean>;
        return { ...defaultVisible, ...parsed };
      }
    } catch {}
    return defaultVisible;
  });

  useEffect(() => {
    localStorage.setItem(VISIBLE_COLS_STORAGE_KEY, JSON.stringify(visibleCols));
  }, [visibleCols]);

  const toggleCol = (key: ColumnKey) =>
    setVisibleCols((prev) => ({ ...prev, [key]: !prev[key] }));

  // ---- Helpers
  const fullName = (a: Attendee) => `${a.first_name} ${a.last_name}`.trim();
  const fmt = (iso?: string | null) => (iso ? new Date(iso).toLocaleString() : '—');
  const certBadgeVariant = (status?: string) => {
    const s = (status || '').toUpperCase();
    if (s === 'PENDING') return 'secondary' as const;
    if (s === 'APPROVED' || s === 'VALID') return 'default' as const;
    if (s === 'REVOKED' || s === 'EXPIRED') return 'destructive' as const;
    return 'outline' as const;
  };

  // ---- Query
  const applyQuery = (page = 1, opts?: Partial<{ search: string; sort: string; direction: string }>) => {
    router.get(
      baseHref,
      {
        page,
        search: opts?.search ?? search,
        sort: opts?.sort ?? sort,
        direction: opts?.direction ?? direction,
      },
      { preserveState: true, replace: true, preserveScroll: true }
    );
  };

  const handleManualSearch = () => applyQuery(1);

  const handleReset = () => {
    setSearch('');
    setSort('created_at');
    setDirection('desc');
    router.get(baseHref, { page: 1 }, { preserveState: true, replace: true, preserveScroll: true });
  };

  const handlePageChange = (selected: { selected: number }) => {
    applyQuery(selected.selected + 1);
  };

  // 🔹 Auto-search
  useEffect(() => {
    if (firstRun.current) {
      firstRun.current = false;
      return;
    }
    applyQuery(1, { search: debouncedSearch });
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [debouncedSearch]);

  // ---- Sorting
  const toggleSort = (field: 'created_at' | 'time_of_entry' | 'last_name') => {
    if (sort === field) {
      const next = direction === 'asc' ? 'desc' : 'asc';
      setDirection(next);
      applyQuery(1, { sort: field, direction: next });
    } else {
      setSort(field);
      setDirection('asc');
      applyQuery(1, { sort: field, direction: 'asc' });
    }
  };

  const SortIcon = ({ active }: { active: boolean }) =>
    active ? (direction === 'asc' ? <ArrowUp className="ml-1 h-3.5 w-3.5" /> : <ArrowDown className="ml-1 h-3.5 w-3.5" />) : <ArrowUpDown className="ml-1 h-3.5 w-3.5 opacity-50" />;

  // ---- Actions
  const confirmDelete = async (id: number) => {
    const res = await Swal.fire({
      title: 'Delete attendee?',
      text: 'This will remove the attendee and related medical records.',
      icon: 'warning',
      showCancelButton: true,
      confirmButtonText: 'Yes, delete',
      cancelButtonText: 'Cancel',
      confirmButtonColor: '#dc2626',
      cancelButtonColor: '#6b7280',
      reverseButtons: true,
      focusCancel: true,
    });

    if (res.isConfirmed) {
      router.delete(route('medicals.attendees.destroy', id), {
        preserveScroll: true,
      });
    }
  };

  // ---- Columns
  const columns = useMemo(
    () => [
      {
        key: 'name' as const,
        header: (
          <button
            type="button"
            onClick={() => toggleSort('last_name')}
            className="inline-flex items-center uppercase"
            title="Sort by last name"
          >
            Name <SortIcon active={sort === 'last_name'} />
          </button>
        ),
        cell: (a: Attendee) => (
          <div className="flex flex-col">
            <Link href={route('medicals.attendees.show', a.id)} className="font-medium hover:underline">
              {fullName(a)}
            </Link>
            <span className="text-xs text-muted-foreground">
              {a.gender ? a.gender : '—'}
              {a.age != null ? ` • ${formatAge(a.age)}` : ''}
            </span>
          </div>
        ),
        className: 'py-2',
      },
      {
        key: 'national' as const,
        header: 'National ID',
        cell: (a: Attendee) => a.national_id || '—',
        className: 'py-2',
      },
      {
        key: 'company' as const,
        header: 'Company',
        cell: (a: Attendee) =>
          a.company ? (
            <Badge variant="secondary" className="inline-flex items-center gap-1">
              <Building2 className="h-3.5 w-3.5" />
              {a.company.company_name}
            </Badge>
          ) : (
            '—'
          ),
        className: 'py-2',
      },
      {
        key: 'category' as const,
        header: 'Category',
        cell: (a: Attendee) =>
          a.category ? (
            <Badge className="font-medium">{a.category}</Badge>
          ) : (
            '—'
          ),
        className: 'py-2',
      },
      {
        key: 'swab' as const,
        header: 'Swab #',
        cell: (a: Attendee) => a.swab_number ?? '—',
        className: 'py-2',
      },
      {
        key: 'time' as const,
        header: (
          <button
            type="button"
            onClick={() => toggleSort('time_of_entry')}
            className="inline-flex items-center uppercase"
            title="Sort by time of entry"
          >
            Time of Entry <SortIcon active={sort === 'time_of_entry'} />
          </button>
        ),
        cell: (a: Attendee) => fmt(a.time_of_entry),
        className: 'py-2 whitespace-nowrap',
      },
      {
        key: 'cert' as const,
        header: 'Certificate',
        cell: (a: Attendee) => {
          const s = a.latest_certificate?.status ?? '—';
          return s === '—' ? (
            '—'
          ) : (
            <Badge variant={certBadgeVariant(s)} className="inline-flex items-center gap-1">
              <BadgeCheck className="h-3.5 w-3.5" />
              {s}
            </Badge>
          );
        },
        className: 'py-2',
      },
      {
        key: 'created' as const,
        header: (
          <button
            type="button"
            onClick={() => toggleSort('created_at')}
            className="inline-flex items-center uppercase"
            title="Sort by created date"
          >
            Created <SortIcon active={sort === 'created_at'} />
          </button>
        ),
        cell: (a: Attendee) => fmt(a.created_at),
        className: 'py-2 whitespace-nowrap',
      },
      {
        key: 'actions' as const,
        header: 'Actions',
        cell: (a: Attendee) => (
          <div className="flex items-center gap-2">
            <Button asChild size="icon" variant="outline" title="View">
              <Link href={route('medicals.attendees.show', a.id)}>
                <Eye className="h-4 w-4" />
              </Link>
            </Button>
            <Button asChild size="icon" variant="secondary" title="Edit">
              <Link href={route('medicals.attendees.edit', a.id)}>
                <Pencil className="h-4 w-4" />
              </Link>
            </Button>
            <Button
              size="icon"
              variant="destructive"
              onClick={() => confirmDelete(a.id)}
              title="Delete"
            >
              <Trash2 className="h-4 w-4" />
            </Button>
          </div>
        ),
        className: 'py-2',
      },
    ],
    [sort, direction]
  );

  const visibleColumns = columns.filter((c) => visibleCols[c.key]);

  return (
    <AppLayout breadcrumbs={[{ title: 'Medicals', href: `${API}/medicals` }, { title: 'Attendees', href: baseHref }]}>
      <Head title="Medicals • Attendees" />

      <div className="bg-background mx-8 my-6 rounded-xl p-6 shadow-sm">
        <div className="mb-4 flex flex-wrap items-center justify-between gap-3">
          <h1 className="text-2xl font-bold">Attendees</h1>

          <div className="flex items-center gap-2">
            <DropdownMenu>
              <DropdownMenuTrigger asChild>
                <Button variant="outline">
                  <Columns className="mr-2 h-4 w-4" />
                  Columns
                </Button>
              </DropdownMenuTrigger>
              <DropdownMenuContent align="end" className="w-56">
                <DropdownMenuLabel>Toggle columns</DropdownMenuLabel>
                <DropdownMenuSeparator />
                {(columns as { key: ColumnKey; header: React.ReactNode }[]).map((col) => (
                  <DropdownMenuCheckboxItem
                    key={col.key}
                    checked={visibleCols[col.key]}
                    onCheckedChange={() => toggleCol(col.key)}
                  >
                    {typeof col.header === 'string'
                      ? col.header
                      : col.key === 'name'
                      ? 'Name'
                      : col.key === 'time'
                      ? 'Time of Entry'
                      : col.key === 'created'
                      ? 'Created'
                      : col.key === 'national'
                      ? 'National ID'
                      : col.key === 'cert'
                      ? 'Certificate'
                      : col.key === 'actions'
                      ? 'Actions'
                      : (col.header as any)?.props?.children ?? 'Column'}
                  </DropdownMenuCheckboxItem>
                ))}
              </DropdownMenuContent>
            </DropdownMenu>

            <Button asChild>
              <Link href={route('medicals.attendees.create')}>
                <Plus className="mr-2 h-4 w-4" />
                Add Attendee
              </Link>
            </Button>
          </div>
        </div>

        {/* Filters */}
        <div className="mb-4 flex flex-wrap items-center gap-2">
          <div className="relative w-72">
            <Search className="text-muted-foreground absolute left-3 top-2.5 h-5 w-5" />
            <Input
              value={search}
              onChange={(e) => setSearch(e.target.value)}
              onKeyDown={(e) => e.key === 'Enter' && handleManualSearch()}
              placeholder="Search name, swab, ID..."
              className="pl-10"
            />
          </div>

        <Button onClick={handleManualSearch}>Search</Button>
        <Button variant="outline" onClick={handleReset}>Reset</Button>

          <div className="ml-auto text-sm text-muted-foreground">
            Sorting by{' '}
            <span className="font-medium">
              {sort === 'created_at' ? 'Created' : sort === 'time_of_entry' ? 'Time of Entry' : 'Last Name'}
            </span>{' '}
            • <span className="font-medium uppercase">{direction}</span>
          </div>
        </div>

        {/* Table */}
        <div className="overflow-x-auto rounded-xl border">
          <Table className="min-w-full">
            <TableHeader>
              <TableRow>
                {visibleColumns.map((col) => (
                  <TableHead key={col.key} className="uppercase">
                    {col.header}
                  </TableHead>
                ))}
              </TableRow>
            </TableHeader>
            <TableBody>
              {attendees.data.map((a) => (
                <TableRow key={a.id} className="transition hover:bg-muted/50">
                  {visibleColumns.map((col) => (
                    <TableCell key={col.key} className={col.className}>
                      {/* @ts-expect-error generic cell fn signature */}
                      {col.cell(a)}
                    </TableCell>
                  ))}
                </TableRow>
              ))}

              {attendees.data.length === 0 && (
                <TableRow>
                  <TableCell colSpan={visibleColumns.length} className="py-6 text-center text-sm text-muted-foreground">
                    No attendees match your search.
                  </TableCell>
                </TableRow>
              )}
            </TableBody>
          </Table>
        </div>

        {/* Pagination */}
        <div className="mt-4 flex justify-center">
          <ReactPaginate
            pageCount={attendees.last_page}
            forcePage={attendees.current_page - 1}
            onPageChange={handlePageChange}
            marginPagesDisplayed={1}
            pageRangeDisplayed={3}
            previousLabel="← Prev"
            nextLabel="Next →"
            breakLabel="..."
            containerClassName="flex items-center gap-2 text-sm"
            pageClassName="px-3 py-1 rounded border border-border hover:bg-muted"
            activeClassName="bg-blue-600 text-white"
            previousClassName="px-3 py-1 rounded border border-border hover:bg-muted"
            nextClassName="px-3 py-1 rounded border border-border hover:bg-muted"
            breakClassName="px-2"
          />
        </div>
      </div>
    </AppLayout>
  );
}
