// resources/js/pages/Medicals/Attendees/Show.tsx
import AppLayout from '@/layouts/app-layout';
import { Head, Link, router, usePage } from '@inertiajs/react';
import React, { useRef } from 'react';
import moment from 'moment';
import Swal from 'sweetalert2';
import { useReactToPrint } from 'react-to-print';
import { QRCodeSVG } from 'qrcode.react';

import { Button } from '@/components/ui/button';
import { Alert, AlertTitle, AlertDescription } from '@/components/ui/alert';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Separator } from '@/components/ui/separator';

import {
  ArrowLeft,
  BadgeCheck,
  Building2,
  Edit3,
  ImageIcon,
  Printer,
  ShieldAlert,
  Trash2,
} from 'lucide-react';

type Company = {
  id: number;
  company_name: string;
};

type Certificate = {
  id: number;
  patient_id: number;
  status: string;
  certificate_location?: string | null;
  created_at: string | null;
  updated_at: string | null;
};

type Attendee = {
  id: number;
  first_name: string;
  last_name: string;
  gender?: string | null;
  date_of_birth?: string | null;
  age?: number | null;
  national_id: string;
  phone_number?: string | null;

  swab_number?: string | null;
  x_ray_status?: string | null;
  time_of_entry?: string | null;

  employee_number?: string | null;
  exam_purpose?: string | null;
  category?: string | null;
  last_x_ray?: string | null;

  company?: Company | null;

  latest_certificate?: Certificate | null;

  created_at?: string | null;
  updated_at?: string | null;
};

type PageProps = {
  attendee: Attendee;
};

const LOGO_URL = 'https://providencehumancapital.com/headoffice/assets/images/PHC_Logo.png';

export default function MedicalsAttendeesShow() {
  const { attendee } = usePage<PageProps>().props;

  const contentRef = useRef<HTMLDivElement>(null);
  const reactToPrintFn = useReactToPrint({ contentRef });

  const fullName = [attendee.first_name, attendee.last_name].filter(Boolean).join(' ');
  const ageShort = attendee?.age != null ? `${Math.floor(attendee.age)}y` : '—';
  const timeOfEntry = attendee.time_of_entry
    ? moment(attendee.time_of_entry).format('DD MMM YYYY, HH:mm')
    : '—';
  const created = attendee.created_at ? moment(attendee.created_at).format('DD MMM YYYY, HH:mm') : '—';

  const certStatus = attendee.latest_certificate?.status ?? null;

  const onDelete = async () => {
    const res = await Swal.fire({
      title: 'Delete attendee?',
      text: 'This will remove the attendee and related medical records.',
      icon: 'warning',
      showCancelButton: true,
      confirmButtonText: 'Yes, delete',
      cancelButtonText: 'Cancel',
      confirmButtonColor: '#dc2626',
      cancelButtonColor: '#6b7280',
      reverseButtons: true,
      focusCancel: true,
    });

    if (res.isConfirmed) {
      router.delete(route('medicals.attendees.destroy', attendee.id), {
        onSuccess: () => {
          Swal.fire({
            icon: 'success',
            title: 'Deleted',
            timer: 1600,
            showConfirmButton: false,
          });
        },
      });
    }
  };

  // Simple info helpers
  const Info = ({ label, value }: { label: string; value?: string | null }) => (
    <div>
      <div className="text-muted-foreground text-sm">{label}</div>
      <div className="text-foreground font-medium">{value ?? '—'}</div>
    </div>
  );

  const InfoGrid = ({ items }: { items: { label: string; value?: string | null }[] }) => (
    <div className="grid grid-cols-1 gap-4 sm:grid-cols-2">
      {items.map((item) => (
        <Info key={item.label} {...item} />
      ))}
    </div>
  );

  const Section = ({ title, children }: { title: string; children: React.ReactNode }) => (
    <section className="bg-card rounded-lg border p-6 shadow-sm">
      <h2 className="text-foreground mb-4 text-xl font-semibold">{title}</h2>
      {children}
    </section>
  );

  // Printable "attendance" slip (short + clean)
  const attendanceTextDate = attendee.created_at
    ? moment(attendee.created_at).format('MMMM Do YYYY, h:mm A')
    : timeOfEntry !== '—'
    ? moment(attendee.time_of_entry!).format('MMMM Do YYYY, h:mm A')
    : '-';

  return (
    <AppLayout
      breadcrumbs={[
        { title: 'Medicals', href: route('medicals.dashboard') },
        { title: 'Attendees', href: route('medicals.attendees.index') },
        { title: fullName || `#${attendee.id}`, href: route('medicals.attendees.show', attendee.id) },
      ]}
    >
      <Head title={`Attendee: ${fullName}`} />

      {/* Hidden printable block */}
      <div style={{ display: 'none' }}>
        <div ref={contentRef} className="p-6 text-sm text-gray-800">
          <style>
            {`
              @page { size: auto; margin: 12mm 10mm; }
              @media print {
                body { -webkit-print-color-adjust: exact; print-color-adjust: exact; }
              }
            `}
          </style>

          <div className="flex items-center justify-between border-b-2 border-gray-200 pb-4">
            <img src={LOGO_URL} alt="PHC Logo" style={{ height: '3.5rem' }} />
            <div className="text-center">
              <div className="text-lg font-bold">Clinic Attendance</div>
              <div>{attendanceTextDate}</div>
            </div>
            <QRCodeSVG value={`Medicals-Attendee:${attendee.id}`} size={58} />
          </div>

          <div className="mt-6 grid gap-3">
            <div><strong>Name:</strong> {fullName || '-'}</div>
            <div><strong>National ID:</strong> {attendee.national_id || '-'}</div>
            <div><strong>Company:</strong> {attendee.company?.company_name || '-'}</div>
            <div><strong>Category:</strong> {attendee.category || '-'}</div>
            <div><strong>Time of Entry:</strong> {timeOfEntry}</div>
          </div>

          <div className="mt-10 text-xs text-gray-500">
            Printed on {moment().format('MMMM Do YYYY, h:mm A')}
          </div>
        </div>
      </div>

      {/* Page content */}
      <div className="w-full space-y-8 px-4 py-8 sm:px-6">
        {/* Header row */}
        <div className="flex flex-col gap-4 sm:flex-row sm:items-center sm:justify-between">
          <div className="flex items-center gap-4">
            {/* Avatar */}
            <div className="bg-muted text-muted-foreground flex h-20 w-20 items-center justify-center rounded-full border">
              <ImageIcon className="h-8 w-8" />
            </div>

            <div>
              <h1 className="text-foreground text-2xl font-bold">{fullName || `Attendee #${attendee.id}`}</h1>
              <div className="mt-1 text-sm text-muted-foreground">
                {attendee.gender || '—'} {ageShort !== '—' ? `• ${ageShort}` : ''}
              </div>
            </div>
          </div>

          <div className="flex flex-wrap gap-2">
            <Button asChild variant="outline">
              <Link href={route('medicals.attendees.index')}>
                <ArrowLeft className="mr-2 h-4 w-4" />
                Back
              </Link>
            </Button>

            <Button asChild variant="secondary">
              <Link href={route('medicals.attendees.edit', attendee.id)}>
                <Edit3 className="mr-2 h-4 w-4" />
                Edit
              </Link>
            </Button>

            <Button variant="destructive" onClick={onDelete}>
              <Trash2 className="mr-2 h-4 w-4" />
              Delete
            </Button>

            <Button variant="outline" onClick={reactToPrintFn}>
              <Printer className="mr-2 h-4 w-4" />
              Print Slip
            </Button>
          </div>
        </div>

        {/* Optional alert for pending certificate */}
        {certStatus && certStatus.toUpperCase() === 'PENDING' && (
          <Alert className="border-amber-300 bg-amber-50">
            <ShieldAlert className="h-4 w-4" />
            <AlertTitle>Certificate Pending</AlertTitle>
            <AlertDescription>
              This attendee’s certificate is still pending approval.
            </AlertDescription>
          </Alert>
        )}

        <div className="grid grid-cols-12 gap-8">
          {/* Left column */}
          <div className="col-span-12 space-y-6 lg:col-span-8">
            <Section title="Personal Information">
              <InfoGrid
                items={[
                  { label: 'Full Name', value: fullName },
                  { label: 'National ID', value: attendee.national_id },
                  {
                    label: 'Date of Birth',
                    value: attendee.date_of_birth
                      ? moment(attendee.date_of_birth).format('DD MMM YYYY')
                      : null,
                  },
                  { label: 'Gender', value: attendee.gender ?? null },
                  { label: 'Phone', value: attendee.phone_number ?? null },
                  { label: 'Employee Number', value: attendee.employee_number ?? null },
                ]}
              />
            </Section>

            <Section title="Company & Exam">
              <InfoGrid
                items={[
                  { label: 'Company', value: attendee.company?.company_name ?? null },
                  { label: 'Category', value: attendee.category ?? null },
                  { label: 'Exam Purpose', value: attendee.exam_purpose ?? null },
                  { label: 'Last Chest X-Ray', value: attendee.last_x_ray ?? null },
                ]}
              />
            </Section>

            <Section title="Visit Details">
              <InfoGrid
                items={[
                  { label: 'Time of Entry', value: timeOfEntry },
                  { label: 'X-ray Status', value: attendee.x_ray_status ?? null },
                  { label: 'Swab Number', value: attendee.swab_number ?? null },
                  { label: 'Created', value: created },
                ]}
              />
            </Section>

            <Section title="Latest Certificate">
              {attendee.latest_certificate ? (
                <div className="grid gap-2 sm:grid-cols-2">
                  <Info label="Status" value={attendee.latest_certificate.status} />
                  <Info
                    label="Certificate Location"
                    value={attendee.latest_certificate.certificate_location ?? '—'}
                  />
                  <Info
                    label="Created"
                    value={
                      attendee.latest_certificate.created_at
                        ? moment(attendee.latest_certificate.created_at).format('DD MMM YYYY')
                        : '—'
                    }
                  />
                  <Info
                    label="Updated"
                    value={
                      attendee.latest_certificate.updated_at
                        ? moment(attendee.latest_certificate.updated_at).format('DD MMM YYYY')
                        : '—'
                    }
                  />
                </div>
              ) : (
                <div className="text-sm text-muted-foreground">No certificate on record.</div>
              )}
            </Section>
          </div>

          {/* Right column (Actions) */}
          <div className="col-span-12 lg:col-span-4">
            <Card>
              <CardHeader>
                <CardTitle className="text-xl">Actions</CardTitle>
              </CardHeader>
              <CardContent className="space-y-3">
                <Button asChild className="w-full">
                  <Link href={route('medicals.attendees.edit', attendee.id)}>
                    <Edit3 className="mr-2 h-4 w-4" />
                    Edit Attendee
                  </Link>
                </Button>

                <Button variant="outline" className="w-full" onClick={reactToPrintFn}>
                  <Printer className="mr-2 h-4 w-4" />
                  Print Attendance Slip
                </Button>

                <Button variant="destructive" className="w-full" onClick={onDelete}>
                  <Trash2 className="mr-2 h-4 w-4" />
                  Delete
                </Button>

                <Separator />

                <div className="rounded-md border p-3 text-sm">
                  <div className="mb-1 flex items-center gap-2 font-medium">
                    <BadgeCheck className="h-4 w-4 text-emerald-600" />
                    Certificate
                  </div>
                  <div className="text-muted-foreground">
                    {certStatus ? `Current status: ${certStatus}` : 'No certificate available.'}
                  </div>
                </div>

                <Button asChild variant="outline" className="w-full">
                  <Link href={route('medicals.attendees.index')}>
                    <ArrowLeft className="mr-2 h-4 w-4" />
                    Back to Attendees
                  </Link>
                </Button>
              </CardContent>
            </Card>
          </div>
        </div>
      </div>
    </AppLayout>
  );
}
