import AppLayout from '@/layouts/app-layout';
import { Head, Link, useForm } from '@inertiajs/react';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Textarea } from '@/components/ui/textarea';
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { toast } from 'sonner';
import { ChevronLeft, Info } from 'lucide-react';

type FormData = {
  company_name: string;
  address: string;
  site_telephone: string;
  company_email: string;
  contact_person?: string | null;
  province?: string | null;
  designation?: string | null;
  contact_number?: string | null;
};

export default function MedicalsCompaniesCreate() {
  const { data, setData, post, processing, errors } = useForm<FormData>({
    company_name: '',
    address: '',
    site_telephone: '',
    company_email: '',
    contact_person: '',
    province: '',
    designation: '',
    contact_number: '',
  });

  const submit = (e: React.FormEvent) => {
    e.preventDefault();
    post(route('medicals.companies.store'), {
      onSuccess: () => toast.success('Company created successfully.'),
      onError: () => toast.error('Please fix the errors and try again.'),
    });
  };

  return (
    <AppLayout breadcrumbs={[
      { title: 'Medicals', href: route('medicals.dashboard') },
      { title: 'Companies', href: route('medicals.companies.index') },
      { title: 'Create', href: '#' },
    ]}>
      <Head title="Create Company • Medicals" />

      <div className="mx-8 my-6 grid gap-6 md:grid-cols-3">
        {/* Main form */}
        <Card className="md:col-span-2">
          <CardHeader>
            <CardTitle>New Company</CardTitle>
            <CardDescription>Capture company details for the Medicals module.</CardDescription>
          </CardHeader>
          <CardContent>
            <form className="space-y-5" onSubmit={submit}>
              <div>
                <Label htmlFor="company_name">Company Name</Label>
                <Input
                  id="company_name"
                  value={data.company_name}
                  onChange={(e) => setData('company_name', e.target.value)}
                  placeholder="Acme Mining (Pty) Ltd"
                />
                {errors.company_name && <p className="mt-1 text-sm text-red-600">{errors.company_name}</p>}
              </div>

              <div>
                <Label htmlFor="address">Address</Label>
                <Textarea
                  id="address"
                  value={data.address}
                  onChange={(e) => setData('address', e.target.value)}
                  placeholder="Street • City • Province • Code"
                />
                {errors.address && <p className="mt-1 text-sm text-red-600">{errors.address}</p>}
              </div>

              <div className="grid gap-4 md:grid-cols-2">
                <div>
                  <Label htmlFor="site_telephone">Site Telephone</Label>
                  <Input
                    id="site_telephone"
                    value={data.site_telephone}
                    onChange={(e) => setData('site_telephone', e.target.value)}
                    placeholder="+27 ..."
                  />
                  {errors.site_telephone && <p className="mt-1 text-sm text-red-600">{errors.site_telephone}</p>}
                </div>
                <div>
                  <Label htmlFor="company_email">Company Email</Label>
                  <Input
                    id="company_email"
                    type="email"
                    value={data.company_email}
                    onChange={(e) => setData('company_email', e.target.value)}
                    placeholder="admin@company.co.za"
                  />
                  {errors.company_email && <p className="mt-1 text-sm text-red-600">{errors.company_email}</p>}
                </div>
              </div>

              <div className="grid gap-4 md:grid-cols-3">
                <div>
                  <Label htmlFor="contact_person">Contact Person</Label>
                  <Input
                    id="contact_person"
                    value={data.contact_person ?? ''}
                    onChange={(e) => setData('contact_person', e.target.value)}
                    placeholder="Jane Doe"
                  />
                  {errors.contact_person && <p className="mt-1 text-sm text-red-600">{errors.contact_person}</p>}
                </div>
                <div>
                  <Label htmlFor="province">Province</Label>
                  <Input
                    id="province"
                    value={data.province ?? ''}
                    onChange={(e) => setData('province', e.target.value)}
                    placeholder="Gauteng"
                  />
                  {errors.province && <p className="mt-1 text-sm text-red-600">{errors.province}</p>}
                </div>
                <div>
                  <Label htmlFor="designation">Designation</Label>
                  <Input
                    id="designation"
                    value={data.designation ?? ''}
                    onChange={(e) => setData('designation', e.target.value)}
                    placeholder="HSE Manager"
                  />
                  {errors.designation && <p className="mt-1 text-sm text-red-600">{errors.designation}</p>}
                </div>
              </div>

              <div>
                <Label htmlFor="contact_number">Contact Number</Label>
                <Input
                  id="contact_number"
                  value={data.contact_number ?? ''}
                  onChange={(e) => setData('contact_number', e.target.value)}
                  placeholder="+27 ..."
                />
                {errors.contact_number && <p className="mt-1 text-sm text-red-600">{errors.contact_number}</p>}
              </div>

              <div className="flex items-center gap-2">
                <Button type="submit" disabled={processing}>Create</Button>
                <Button type="button" variant="outline" asChild>
                  <Link href={route('medicals.companies.index')}><ChevronLeft className="mr-1 h-4 w-4" /> Back</Link>
                </Button>
              </div>
            </form>
          </CardContent>
        </Card>

        {/* Right panel */}
        <aside className="space-y-4">
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2"><Info className="h-4 w-4" /> What is this?</CardTitle>
              <CardDescription>You’re creating a Company record in the <Badge variant="secondary">Medicals</Badge> database.</CardDescription>
            </CardHeader>
            <CardContent className="space-y-2 text-sm text-muted-foreground">
              <p>These details are used throughout Medical assessments and certificates.</p>
              <ul className="list-disc pl-5">
                <li><strong>Company Email</strong> must be unique.</li>
                <li>Use a main <strong>site telephone</strong> reachable by the clinic.</li>
                <li><strong>Contact Person</strong> is who we call for clarifications.</li>
              </ul>
            </CardContent>
          </Card>

          <Card>
            <CardHeader>
              <CardTitle>Tips</CardTitle>
            </CardHeader>
            <CardContent className="space-y-2 text-sm text-muted-foreground">
              <p>Prefer official company names (e.g., “(Pty) Ltd”).</p>
              <p>Keep addresses standardized for printing on certificates.</p>
            </CardContent>
          </Card>
        </aside>
      </div>
    </AppLayout>
  );
}
