import AppLayout from '@/layouts/app-layout';
import { Head, Link, useForm, usePage } from '@inertiajs/react';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Textarea } from '@/components/ui/textarea';
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { toast } from 'sonner';
import { ChevronLeft, Info, Trash2 } from 'lucide-react';

type Company = {
  id: number;
  company_name: string;
  address?: string | null;
  site_telephone?: string | null;
  company_email?: string | null;
  contact_person?: string | null;
  province?: string | null;
  designation?: string | null;
  contact_number?: string | null;
};

type PageProps = { company: Company };

export default function MedicalsCompaniesEdit() {
  const { company } = usePage<PageProps>().props;

  const { data, setData, put, processing, errors, reset } = useForm({
    company_name: company.company_name ?? '',
    address: company.address ?? '',
    site_telephone: company.site_telephone ?? '',
    company_email: company.company_email ?? '',
    contact_person: company.contact_person ?? '',
    province: company.province ?? '',
    designation: company.designation ?? '',
    contact_number: company.contact_number ?? '',
  });

  const submit = (e: React.FormEvent) => {
    e.preventDefault();
    put(route('medicals.companies.update', company.id), {
      onSuccess: () => toast.success('Company updated successfully.'),
      onError: () => toast.error('Please fix the errors and try again.'),
    });
  };

  const destroy = () => {
    if (!confirm('Delete this company?')) return;
    window.location.href = route('medicals.companies.destroy', company.id);
  };

  return (
    <AppLayout breadcrumbs={[
      { title: 'Medicals', href: route('medicals.dashboard') },
      { title: 'Companies', href: route('medicals.companies.index') },
      { title: 'Edit', href: '#' },
    ]}>
      <Head title={`Edit • ${company.company_name}`} />

      <div className="mx-8 my-6 grid gap-6 md:grid-cols-3">
        {/* Main form */}
        <Card className="md:col-span-2">
          <CardHeader>
            <CardTitle>Edit Company</CardTitle>
            <CardDescription>Update company details in the Medicals module.</CardDescription>
          </CardHeader>
          <CardContent>
            <form className="space-y-5" onSubmit={submit}>
              <div>
                <Label htmlFor="company_name">Company Name</Label>
                <Input
                  id="company_name"
                  value={data.company_name}
                  onChange={(e) => setData('company_name', e.target.value)}
                />
                {errors.company_name && <p className="mt-1 text-sm text-red-600">{errors.company_name}</p>}
              </div>

              <div>
                <Label htmlFor="address">Address</Label>
                <Textarea
                  id="address"
                  value={data.address}
                  onChange={(e) => setData('address', e.target.value)}
                />
                {errors.address && <p className="mt-1 text-sm text-red-600">{errors.address}</p>}
              </div>

              <div className="grid gap-4 md:grid-cols-2">
                <div>
                  <Label htmlFor="site_telephone">Site Telephone</Label>
                  <Input
                    id="site_telephone"
                    value={data.site_telephone}
                    onChange={(e) => setData('site_telephone', e.target.value)}
                  />
                  {errors.site_telephone && <p className="mt-1 text-sm text-red-600">{errors.site_telephone}</p>}
                </div>
                <div>
                  <Label htmlFor="company_email">Company Email</Label>
                  <Input
                    id="company_email"
                    type="email"
                    value={data.company_email}
                    onChange={(e) => setData('company_email', e.target.value)}
                  />
                  {errors.company_email && <p className="mt-1 text-sm text-red-600">{errors.company_email}</p>}
                </div>
              </div>

              <div className="grid gap-4 md:grid-cols-3">
                <div>
                  <Label htmlFor="contact_person">Contact Person</Label>
                  <Input
                    id="contact_person"
                    value={data.contact_person}
                    onChange={(e) => setData('contact_person', e.target.value)}
                  />
                  {errors.contact_person && <p className="mt-1 text-sm text-red-600">{errors.contact_person}</p>}
                </div>
                <div>
                  <Label htmlFor="province">Province</Label>
                  <Input
                    id="province"
                    value={data.province}
                    onChange={(e) => setData('province', e.target.value)}
                  />
                  {errors.province && <p className="mt-1 text-sm text-red-600">{errors.province}</p>}
                </div>
                <div>
                  <Label htmlFor="designation">Designation</Label>
                  <Input
                    id="designation"
                    value={data.designation}
                    onChange={(e) => setData('designation', e.target.value)}
                  />
                  {errors.designation && <p className="mt-1 text-sm text-red-600">{errors.designation}</p>}
                </div>
              </div>

              <div>
                <Label htmlFor="contact_number">Contact Number</Label>
                <Input
                  id="contact_number"
                  value={data.contact_number}
                  onChange={(e) => setData('contact_number', e.target.value)}
                />
                {errors.contact_number && <p className="mt-1 text-sm text-red-600">{errors.contact_number}</p>}
              </div>

              <div className="flex flex-wrap items-center gap-2">
                <Button type="submit" disabled={processing}>Save Changes</Button>
                <Button type="button" variant="outline" asChild>
                  <Link href={route('medicals.companies.show', company.id)}><ChevronLeft className="mr-1 h-4 w-4" /> Back</Link>
                </Button>
                <Button
                  type="button"
                  variant="destructive"
                  onClick={() => {
                    if (confirm('Delete this company?')) {
                      // Use Inertia delete to keep SPA feel
                      window.Inertia.delete(route('medicals.companies.destroy', company.id), {
                        onSuccess: () => toast.success('Company deleted.'),
                        onError: () => toast.error('Failed to delete.'),
                      } as any);
                    }
                  }}
                >
                  <Trash2 className="mr-2 h-4 w-4" /> Delete
                </Button>
              </div>
            </form>
          </CardContent>
        </Card>

        {/* Right panel */}
        <aside className="space-y-4">
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2"><Info className="h-4 w-4" /> What am I editing?</CardTitle>
              <CardDescription>This updates a Company in the <Badge variant="secondary">Medicals</Badge> database.</CardDescription>
            </CardHeader>
            <CardContent className="space-y-2 text-sm text-muted-foreground">
              <p>Be careful when changing <strong>Company Email</strong> (must remain unique).</p>
              <p>Correct addresses & phone numbers help clinics reach the right team quickly.</p>
            </CardContent>
          </Card>

          <Card>
            <CardHeader>
              <CardTitle>Tips</CardTitle>
            </CardHeader>
            <CardContent className="space-y-2 text-sm text-muted-foreground">
              <p>Confirm spelling and <em>company legal name</em> before saving.</p>
              <p>Keep the contact person and number up-to-date to avoid delays.</p>
            </CardContent>
          </Card>
        </aside>
      </div>
    </AppLayout>
  );
}
