import { API } from '@/config';
import AppLayout from '@/layouts/app-layout';
import { Head, Link, router, usePage } from '@inertiajs/react';
import React, { useEffect, useMemo, useRef, useState } from 'react';
import ReactPaginate from 'react-paginate';
import { Pencil, Plus, Search, Trash2, Columns } from 'lucide-react';

// shadcn/ui
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow
} from '@/components/ui/table';
import { Badge } from '@/components/ui/badge';
import {
  DropdownMenu,
  DropdownMenuCheckboxItem,
  DropdownMenuContent,
  DropdownMenuLabel,
  DropdownMenuSeparator,
  DropdownMenuTrigger,
} from '@/components/ui/dropdown-menu';

type Company = {
  id: number;
  company_name: string;
  address?: string | null;
  company_email?: string | null;
  site_telephone?: string | null;
  contact_person?: string | null;
};

type PageProps = {
  companies: {
    data: Company[];
    current_page: number;
    last_page: number;
    per_page: number;
    total: number;
  };
  filters: {
    search?: string;
    per_page?: number;
  };
};

// 🔹 Simple debounce
function useDebouncedValue<T>(value: T, delay = 450) {
  const [debounced, setDebounced] = useState(value);
  useEffect(() => {
    const id = setTimeout(() => setDebounced(value), delay);
    return () => clearTimeout(id);
  }, [value, delay]);
  return debounced;
}

// 🔹 Title Case helper
function toTitleCase(input?: string | null): string {
  if (!input) return '';
  return input
    .trim()
    .toLowerCase()
    .replace(/\s+/g, ' ')
    .replace(/\b\w/g, (m) => m.toUpperCase());
}

const VISIBLE_COLS_STORAGE_KEY = 'medicals_companies_visible_cols_v1';

type ColumnKey = 'name' | 'address' | 'email' | 'phone' | 'contact' | 'actions';

export default function MedicalsCompanyIndex() {
  const { companies, filters } = usePage<PageProps>().props as PageProps;

  const [search, setSearch] = useState<string>(filters.search || '');
  const debouncedSearch = useDebouncedValue(search, 450);
  const firstRun = useRef(true);

  const baseHref = `${API}/medicals/companies`;

  // ---- Dynamic columns (persisted)
  const defaultVisible: Record<ColumnKey, boolean> = {
    name: true,
    address: true,
    email: true,
    phone: true,
    contact: true,
    actions: true,
  };

  const [visibleCols, setVisibleCols] = useState<Record<ColumnKey, boolean>>(() => {
    try {
      const raw = localStorage.getItem(VISIBLE_COLS_STORAGE_KEY);
      if (raw) {
        const parsed = JSON.parse(raw) as Record<ColumnKey, boolean>;
        return { ...defaultVisible, ...parsed };
      }
    } catch {}
    return defaultVisible;
  });

  useEffect(() => {
    localStorage.setItem(VISIBLE_COLS_STORAGE_KEY, JSON.stringify(visibleCols));
  }, [visibleCols]);

  const toggleCol = (key: ColumnKey) =>
    setVisibleCols((prev) => ({ ...prev, [key]: !prev[key] }));

  // ---- Actions
  const applySearch = (page = 1, override?: string) => {
    router.get(
      baseHref,
      { page, search: override ?? search },
      { preserveState: true, replace: true, preserveScroll: true }
    );
  };

  const handleManualSearch = () => applySearch(1);

  const handleReset = () => {
    setSearch('');
    router.get(baseHref, {}, { preserveState: true, replace: true, preserveScroll: true });
  };

  const handleDelete = (id: number) => {
    if (confirm('Delete this company?')) {
      router.delete(route('medicals.companies.destroy', id), {
        preserveScroll: true,
      });
    }
  };

  const handlePageChange = (selected: { selected: number }) => {
    applySearch(selected.selected + 1);
  };

  // 🔹 Auto-search when typing stops
  useEffect(() => {
    if (firstRun.current) {
      firstRun.current = false;
      return;
    }
    applySearch(1, debouncedSearch);
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [debouncedSearch]);

  // ---- Columns definition
  const columns = useMemo(
    () => [
      {
        key: 'name' as const,
        header: 'Name',
        cell: (c: Company) => (
          <Link href={route('medicals.companies.show', c.id)} className="hover:underline">
            {c.company_name}
          </Link>
        ),
        className: 'py-2 font-medium',
      },
      {
        key: 'address' as const,
        header: 'Address',
        cell: (c: Company) => c.address ?? '—',
        className: 'py-2',
      },
      {
        key: 'email' as const,
        header: 'Email',
        cell: (c: Company) => c.company_email ?? '—',
        className: 'py-2',
      },
      {
        key: 'phone' as const,
        header: 'Phone Number',
        cell: (c: Company) => c.site_telephone ?? '—',
        className: 'py-2',
      },
      {
        key: 'contact' as const,
        header: 'Contact Person',
        cell: (c: Company) =>
          c.contact_person ? (
            <Badge variant="secondary" className="font-medium">
              {toTitleCase(c.contact_person)}
            </Badge>
          ) : (
            '—'
          ),
        className: 'py-2',
      },
      {
        key: 'actions' as const,
        header: 'Actions',
        cell: (c: Company) => (
          <div className="flex items-center gap-2">
            <Button asChild size="icon" variant="secondary" title="Edit">
              <Link href={route('medicals.companies.edit', c.id)}>
                <Pencil className="h-4 w-4" />
              </Link>
            </Button>

            <Button
              size="icon"
              variant="destructive"
              onClick={() => handleDelete(c.id)}
              title="Delete"
            >
              <Trash2 className="h-4 w-4" />
            </Button>
          </div>
        ),
        className: 'py-2',
      },
    ],
    []
  );

  const visibleColumns = columns.filter((col) => visibleCols[col.key]);

  return (
    <AppLayout breadcrumbs={[{ title: 'Medicals', href: `${API}/medicals` }, { title: 'Companies', href: baseHref }]}>
      <Head title="Medicals • Companies" />

      <div className="bg-background mx-8 my-6 rounded-xl p-6 shadow-sm">
        <div className="mb-4 flex flex-wrap items-center justify-between gap-3">
          <h1 className="text-2xl font-bold">Companies (Medicals)</h1>

          <div className="flex items-center gap-2">
            <DropdownMenu>
              <DropdownMenuTrigger asChild>
                <Button variant="outline">
                  <Columns className="mr-2 h-4 w-4" />
                  Columns
                </Button>
              </DropdownMenuTrigger>
              <DropdownMenuContent align="end" className="w-56">
                <DropdownMenuLabel>Toggle columns</DropdownMenuLabel>
                <DropdownMenuSeparator />
                {(columns as { key: ColumnKey; header: string }[]).map((col) => (
                  <DropdownMenuCheckboxItem
                    key={col.key}
                    checked={visibleCols[col.key]}
                    onCheckedChange={() => toggleCol(col.key)}
                  >
                    {col.header}
                  </DropdownMenuCheckboxItem>
                ))}
              </DropdownMenuContent>
            </DropdownMenu>

            <Button asChild>
              <Link href={route('medicals.companies.create')}>
                <Plus className="mr-2 h-4 w-4" />
                Add Company
              </Link>
            </Button>
          </div>
        </div>

        {/* Filters */}
        <div className="mb-4 flex flex-wrap items-center gap-2">
          <div className="relative w-72">
            <Search className="text-muted-foreground absolute left-3 top-2.5 h-5 w-5" />
            <Input
              value={search}
              onChange={(e) => setSearch(e.target.value)}
              onKeyDown={(e) => e.key === 'Enter' && handleManualSearch()}
              placeholder="Search companies by name..."
              className="pl-10"
            />
          </div>

          <Button onClick={handleManualSearch}>Search</Button>
          <Button variant="outline" onClick={handleReset}>Reset</Button>
        </div>

        {/* Table */}
        <div className="overflow-x-auto rounded-xl border">
          <Table className="min-w-full">
            <TableHeader>
              <TableRow>
                {visibleColumns.map((col) => (
                  <TableHead key={col.key} className="uppercase">
                    {col.header}
                  </TableHead>
                ))}
              </TableRow>
            </TableHeader>
            <TableBody>
              {companies.data.map((c) => (
                <TableRow key={c.id} className="transition hover:bg-muted/50">
                  {visibleColumns.map((col) => (
                    <TableCell key={col.key} className={col.className}>
                      {col.cell(c)}
                    </TableCell>
                  ))}
                </TableRow>
              ))}

              {companies.data.length === 0 && (
                <TableRow>
                  <TableCell colSpan={visibleColumns.length} className="py-6 text-center text-sm text-muted-foreground">
                    No companies match your search.
                  </TableCell>
                </TableRow>
              )}
            </TableBody>
          </Table>
        </div>

        {/* Pagination */}
        <div className="mt-4 flex justify-center">
          <ReactPaginate
            pageCount={companies.last_page}
            forcePage={companies.current_page - 1}
            onPageChange={handlePageChange}
            marginPagesDisplayed={1}
            pageRangeDisplayed={3}
            previousLabel="← Prev"
            nextLabel="Next →"
            breakLabel="..."
            containerClassName="flex items-center gap-2 text-sm"
            pageClassName="px-3 py-1 rounded border border-border hover:bg-muted"
            activeClassName="bg-blue-600 text-white"
            previousClassName="px-3 py-1 rounded border border-border hover:bg-muted"
            nextClassName="px-3 py-1 rounded border border-border hover:bg-muted"
            breakClassName="px-2"
          />
        </div>
      </div>
    </AppLayout>
  );
}
