import AppLayout from '@/layouts/app-layout';
import { Head, Link, router, usePage } from '@inertiajs/react';
import React, { useState } from 'react';
import Swal from 'sweetalert2';

import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from '@/components/ui/card';
import { Label } from '@/components/ui/label';
import { ArrowLeft, Save, Trash2 } from 'lucide-react';

type Disease = { id: number; name: string };
type PageProps = { disease: Disease };

export default function MedicalsDiseasesEdit() {
  const { disease } = usePage<PageProps>().props;
  const [name, setName] = useState(disease.name);
  const [errors, setErrors] = useState<{ name?: string }>({});

  const onSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    setErrors({});

    router.patch(
      route('medicals.diseases.update', disease.id),
      { name },
      {
        preserveScroll: true,
        onSuccess: () => {
          Swal.fire({ icon: 'success', title: 'Disease updated', timer: 1400, showConfirmButton: false });
          router.visit(route('medicals.diseases.index'));
        },
        onError: (err: any) => setErrors(err || {}),
      },
    );
  };

  const onDelete = async () => {
    const { isConfirmed } = await Swal.fire({
      title: `Delete "${disease.name}"?`,
      text: 'This will permanently remove the disease.',
      icon: 'warning',
      showCancelButton: true,
      confirmButtonText: 'Yes, delete',
      cancelButtonText: 'Cancel',
      confirmButtonColor: '#dc2626',
      cancelButtonColor: '#6b7280',
      reverseButtons: true,
    });
    if (!isConfirmed) return;

    router.delete(route('medicals.diseases.destroy', disease.id), {
      onSuccess: () => {
        Swal.fire({ icon: 'success', title: 'Disease deleted', timer: 1400, showConfirmButton: false });
        router.visit(route('medicals.diseases.index'));
      },
    });
  };

  return (
    <AppLayout
      breadcrumbs={[
        { title: 'Medicals', href: route('medicals.patients.index') },
        { title: 'Diseases', href: route('medicals.diseases.index') },
        { title: 'Edit', href: route('medicals.diseases.edit', disease.id) },
      ]}
    >
      <Head title={`Edit Disease • ${disease.name}`} />
      <div className="mx-4 my-6 md:mx-8">
        <Card className="">
          <CardHeader>
            <CardTitle>Edit Disease</CardTitle>
            <CardDescription>Update the disease name.</CardDescription>
          </CardHeader>
          <CardContent>
            <form onSubmit={onSubmit} className="space-y-6">
              <div className="grid gap-2">
                <Label htmlFor="name">Disease Name</Label>
                <Input
                  id="name"
                  value={name}
                  onChange={(e) => setName(e.target.value)}
                  placeholder="e.g., Tuberculosis"
                />
                {errors.name && <div className="text-sm text-red-600">{errors.name}</div>}
              </div>

              <div className="flex items-center justify-between">
                <Button asChild variant="outline">
                  <Link href={route('medicals.diseases.index')}>
                    <ArrowLeft className="mr-2 h-4 w-4" />
                    Back
                  </Link>
                </Button>

                <div className="flex items-center gap-2">
                  <Button type="button" variant="destructive" onClick={onDelete}>
                    <Trash2 className="mr-2 h-4 w-4" />
                    Delete
                  </Button>
                  <Button type="submit">
                    <Save className="mr-2 h-4 w-4" />
                    Save
                  </Button>
                </div>
              </div>
            </form>
          </CardContent>
        </Card>
      </div>
    </AppLayout>
  );
}
