import AppLayout from '@/layouts/app-layout';
import { Head, Link, router, usePage } from '@inertiajs/react';
import React, { useEffect, useRef, useState } from 'react';
import Swal from 'sweetalert2';

import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from '@/components/ui/table';

import { Plus, Search as SearchIcon, Pencil, Trash2 } from 'lucide-react';

type Disease = { id: number; name: string; created_at?: string | null; updated_at?: string | null };
type Pagination<T> = { data: T[]; current_page: number; last_page: number; per_page: number; total: number };
type PageProps = { diseases: Pagination<Disease>; filters: { search?: string; per_page?: number } };

function useDebounced<T>(value: T, delay = 450) {
  const [v, setV] = useState(value);
  useEffect(() => {
    const id = setTimeout(() => setV(value), delay);
    return () => clearTimeout(id);
  }, [value, delay]);
  return v;
}

export default function MedicalsDiseasesIndex() {
  const { diseases, filters } = usePage<PageProps>().props;

  const [search, setSearch] = useState(filters.search ?? '');
  const [perPage, setPerPage] = useState<number>(filters.per_page ?? 20);
  const debouncedSearch = useDebounced(search, 450);
  const firstRun = useRef(true);

  const applyQuery = (page = 1, overrides?: Partial<Record<string, string | number>>) => {
    router.get(
      route('medicals.diseases.index'),
      {
        page,
        search: overrides?.search ?? search,
        per_page: overrides?.per_page ?? perPage,
      },
      { preserveState: true, replace: true, preserveScroll: true },
    );
  };

  useEffect(() => {
    if (firstRun.current) {
      firstRun.current = false;
      return;
    }
    applyQuery(1, { search: debouncedSearch });
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [debouncedSearch]);

  const confirmDelete = async (disease: Disease) => {
    const { isConfirmed } = await Swal.fire({
      title: `Delete "${disease.name}"?`,
      text: 'This will permanently remove the disease.',
      icon: 'warning',
      showCancelButton: true,
      confirmButtonText: 'Yes, delete',
      cancelButtonText: 'Cancel',
      confirmButtonColor: '#dc2626',
      cancelButtonColor: '#6b7280',
      reverseButtons: true,
    });
    if (!isConfirmed) return;

    router.delete(route('medicals.diseases.destroy', disease.id), {
      preserveScroll: true,
      onSuccess: () => Swal.fire({ icon: 'success', title: 'Disease deleted', timer: 1400, showConfirmButton: false }),
    });
  };

  return (
    <AppLayout
      breadcrumbs={[
        { title: 'Medicals', href: route('medicals.patients.index') },
        { title: 'Diseases', href: route('medicals.diseases.index') },
      ]}
    >
      <Head title="Medicals • Diseases" />

      <div className="mx-4 my-6 space-y-4 md:mx-8">
        <div className="flex flex-col items-start justify-between gap-3 sm:flex-row sm:items-center">
          <h1 className="text-2xl font-bold">Diseases</h1>
          <div className="flex gap-2">
            <Button asChild>
              <Link href={route('medicals.diseases.create')}>
                <Plus className="mr-2 h-4 w-4" />
                Add Disease
              </Link>
            </Button>
          </div>
        </div>

        <Card>
          <CardContent className="pt-6">
            <div className="relative">
              <SearchIcon className="text-muted-foreground absolute left-4 top-1/2 h-5 w-5 -translate-y-1/2" />
              <Input
                value={search}
                onChange={(e) => setSearch(e.target.value)}
                onKeyDown={(e) => e.key === 'Enter' && applyQuery(1)}
                placeholder="Search diseases…"
                className="h-12 w-full rounded-xl pl-12 pr-[8.5rem] text-base"
              />
              <Button
                type="button"
                className="absolute right-2 top-1/2 -translate-y-1/2 h-10 rounded-lg px-4"
                onClick={() => applyQuery(1)}
                title="Search"
              >
                <SearchIcon className="mr-2 h-4 w-4" />
                Search
              </Button>
            </div>

            <div className="mt-3 flex items-center justify-between">
              <div className="text-xs text-muted-foreground">Showing {diseases.data.length} of {diseases.total}</div>
              <div className="flex items-center gap-2">
                <span className="text-sm text-muted-foreground">Per page</span>
                <select
                  value={perPage}
                  onChange={(e) => {
                    const n = parseInt(e.target.value, 10);
                    setPerPage(n);
                    applyQuery(1, { per_page: n });
                  }}
                  className="h-9 rounded-md border px-2 text-sm"
                >
                  <option value={10}>10</option>
                  <option value={20}>20</option>
                  <option value={50}>50</option>
                  <option value={100}>100</option>
                </select>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardHeader>
            <CardTitle>List</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="overflow-x-auto rounded-xl border">
              <Table className="min-w-[720px]">
                <TableHeader>
                  <TableRow>
                    <TableHead className="uppercase">#</TableHead>
                    <TableHead className="uppercase">Disease Name</TableHead>
                    <TableHead className="uppercase text-right">Actions</TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {diseases.data.map((d, idx) => (
                    <TableRow key={d.id}>
                      <TableCell>{(diseases.current_page - 1) * diseases.per_page + idx + 1}</TableCell>
                      <TableCell className="font-medium">{d.name}</TableCell>
                      <TableCell className="text-right">
                        <div className="flex justify-end gap-2">
                          <Button asChild size="icon" variant="secondary" title="Edit">
                            <Link href={route('medicals.diseases.edit', d.id)}>
                              <Pencil className="h-4 w-4" />
                            </Link>
                          </Button>
                          <Button size="icon" variant="destructive" onClick={() => confirmDelete(d)} title="Delete">
                            <Trash2 className="h-4 w-4" />
                          </Button>
                        </div>
                      </TableCell>
                    </TableRow>
                  ))}
                  {diseases.data.length === 0 && (
                    <TableRow>
                      <TableCell colSpan={3} className="py-8 text-center text-sm text-muted-foreground">
                        No diseases found.
                      </TableCell>
                    </TableRow>
                  )}
                </TableBody>
              </Table>
            </div>

            <div className="mt-4 flex items-center justify-center gap-2">
              <Button
                variant="outline"
                disabled={diseases.current_page <= 1}
                onClick={() => applyQuery(diseases.current_page - 1)}
              >
                ← Prev
              </Button>
              <div className="text-sm">Page {diseases.current_page} of {diseases.last_page}</div>
              <Button
                variant="outline"
                disabled={diseases.current_page >= diseases.last_page}
                onClick={() => applyQuery(diseases.current_page + 1)}
              >
                Next →
              </Button>
            </div>
          </CardContent>
        </Card>
      </div>
    </AppLayout>
  );
}
